<?php

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Description
 *
 * @since  1.3
 * @author VanboDevelops | Ivan Andreev
 *
 *        Copyright: (c) 2015 VanboDevelops
 *        License: GNU General Public License v3.0
 *        License URI: http://www.gnu.org/licenses/gpl-3.0.html
 */
class WC_PayTrace_Gateway_Addons_Deprecated extends WC_Gateway_PayTrace_Addons {
	function __construct() {
		parent::__construct();

		// Add support for Subscriptions
		if ( 1 === WC_PayTrace::get_subscriptions_version() ) {
			// Subscriptions Actions
			add_action( 'scheduled_subscription_payment_' . $this->id, array( $this, 'scheduled_subscription_payment_request' ), 10, 3 );
			add_filter( 'woocommerce_subscriptions_renewal_order_meta_query', array( $this, 'remove_renewal_order_meta' ), 10, 4 );
			add_action( 'woocommerce_subscriptions_changed_failing_payment_method_' . $this->id, array( $this, 'changed_failing_payment_method' ), 10, 2 );
			add_filter( 'woocommerce_my_subscriptions_recurring_payment_method', array( $this, 'maybe_render_subscription_payment_method' ), 10, 3 );
		}
	}

	/**
	 * Process the payment
	 *
	 * @param int $order_id
	 *
	 * @return array|type|void
	 */
	function process_payment( $order_id ) {
		$order = WC_Compat_PayTrace::wc_get_order( $order_id );
		if ( $this->order_contains_subscription( $order ) ) {
			// Subscription Payment
			return $this->process_subscription_payment( $order_id );
		} else {
			return parent::process_payment( $order_id );
		}
	}

	/**
	 * Process subscription payment
	 *
	 * @since 1.2
	 *
	 * @param int $order_id
	 *
	 * @return type
	 */
	public function process_subscription_payment( $order_id ) {
		try {
			$order           = WC_Compat_PayTrace::wc_get_order( $order_id );
			$initial_payment = WC_Subscriptions_Order::get_total_initial_payment( $order );

			// Using Check(ACH) to pay
			if ( 'check' == $this->what_payment_customer_used || 'saved_check' == $this->what_payment_customer_used ) {
				$this->process_ach_payment( $order, $initial_payment, true );
			} else {
				$this->process_card_payment( $order, $initial_payment, true );
			}

			// Payment complete
			$order->payment_complete();

			// Remove cart
			WC_Compat_PayTrace::empty_cart();

			// Activate the subscriptions
			WC_Subscriptions_Manager::activate_subscriptions_for_order( $order );

			// Return thank you page redirect
			return array(
				'result'   => 'success',
				'redirect' => $this->get_return_url( $order )
			);

		}
		catch ( Exception $e ) {
			WC_Compat_PayTrace::wc_add_notice( $e->getMessage(), 'error' );

			return;
		}
	}

	/**
	 * Process subscription payment for each period
	 *
	 * @param float    $amount_to_charge The amount to charge.
	 * @param WC_Order $order            The WC_Order object of the order which the subscription was purchased in.
	 * @param int      $product_id       The ID of the subscription product for which this payment relates.
	 *
	 * @access public
	 * @return void
	 */
	function scheduled_subscription_payment_request( $amount_to_charge, \WC_Order $order, $product_id ) {

		try {
			$this->process_profile_payment_request( $order, $amount_to_charge, true );

			WC_Subscriptions_Manager::process_subscription_payments_on_order( $order );
		}
		catch ( Exception $e ) {

			$order->update_status( 'failed', $e->getMessage() );

			// Debug log
			if ( 'yes' == $this->debug ) {
				WC_PayTrace::add_debug_log( $e->getMessage() );
			}

			WC_Subscriptions_Manager::process_subscription_payment_failure_on_order( $order, $product_id );
		}

	}

	/**
	 * Don't transfer PayTrace customer/token meta when creating a parent renewal order.
	 *
	 * @access public
	 *
	 * @param array  $order_meta_query  MySQL query for pulling the metadata
	 * @param int    $original_order_id Post ID of the order being used to purchased the subscription being renewed
	 * @param int    $renewal_order_id  Post ID of the order created for renewing the subscription
	 * @param string $new_order_role    The role the renewal order is taking, one of 'parent' or 'child'
	 *
	 * @return void
	 */
	function remove_renewal_order_meta( $order_meta_query, $original_order_id, $renewal_order_id, $new_order_role ) {

		if ( 'parent' == $new_order_role ) {
			$order_meta_query .= " AND `meta_key` NOT LIKE '_paytrace_customer_id' ";
		}

		return $order_meta_query;
	}

	/**
	 * Update the accountID and the Serial No on the original order, when a renewal order is placed, after it failed previously
	 *
	 * @since 1.2
	 *
	 * @param WC_Order $original_order
	 * @param WC_Order $renewal_order
	 */
	public function changed_failing_payment_method( \WC_Order $original_order, \WC_Order $renewal_order ) {
		update_post_meta( $original_order->id, '_paytrace_customer_id', get_post_meta( $renewal_order->id, '_paytrace_customer_id', true ) );

	}


	/**
	 * Show payment method on My Subscriptions section of My Account page
	 *
	 * @since 1.2
	 *
	 * @param string   $payment_method_to_display
	 * @param array    $subscription_details
	 * @param WC_Order $order
	 *
	 * @return string
	 */
	public function maybe_render_subscription_payment_method( $payment_method_to_display, $subscription_details, \WC_Order $order ) {
		if ( $this->id !== $order->recurring_payment_method ) {
			return $payment_method_to_display;
		}

		$customer_profile = $this->get_profile_from_customer_id( get_post_meta( $order->id, '_paytrace_customer_id', true ) );

		$method_type = ( is_array( $customer_profile ) && isset( $customer_profile['last4'] ) ) ? 'card' : 'check';

		if ( is_array( $customer_profile ) ) {
			if ( 'check' == $method_type ) {
				$payment_method_to_display = sprintf( __( 'Via Check with account ending in %s', WC_PayTrace::TEXT_DOMAIN ), $customer_profile['dda_last4'] );
			} else {
				$payment_method_to_display = sprintf( __( 'Via Credit Card ending in %s', WC_PayTrace::TEXT_DOMAIN ), $customer_profile['last4'] );
			}
		}

		return $payment_method_to_display;
	}

	/**
	 * Returns true, if order contains subscription
	 *
	 * @param WC_Order $order
	 *
	 * @return bool
	 */
	public function order_contains_subscription( WC_Order $order ) {
		if (
			1 === WC_PayTrace::get_subscriptions_version()
			&& WC_Subscriptions_Order::order_contains_subscription( $order )
		) {
			return true;
		}

		return false;
	}

	/**
	 * Get order items description
	 *
	 * @param WC_Order $order
	 * @param bool     $is_subscription
	 *
	 * @return string
	 */
	public function get_order_description( \WC_Order $order, $is_subscription = false ) {
		$desc = '';
		if ( $is_subscription ) {
			$order_items = $order->get_items();
			$product     = $order->get_product_from_item( array_shift( $order_items ) );
			$desc        = $this->html_entity_decode_numeric(
				sprintf(
					__( 'Initial payment for subscription "%s". (Order %s)', WC_PayTrace::TEXT_DOMAIN ),
					$product->get_title(), WC_PayTrace::get_order_number( $order )
				)
			);
		} else {
			if ( sizeof( $order->get_items() ) > 0 ) {
				foreach ( $order->get_items() as $item ) {
					if ( $item['qty'] ) {

						$item_name = $item['name'];

						$item_meta = WC_Compat_PayTrace::get_order_item_meta( $item );

						if ( $meta = $item_meta->display( true, true ) ) {
							$item_name .= ' (' . $meta . ')';
						}
						$desc .= $item['qty'] . ' x ' . $item_name . ', ';
					}
				}
				// Remove the last two chars and decode any html chars
				$desc = $this->html_entity_decode_numeric( substr( $desc, 0, - 2 ) );
			}
		}

		return $desc;
	}
}