<?php

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * PayTrace API Class
 *
 * This class will help in generating and processing a PayTrace payment.
 * The class supports sending and receiving PayTrace requests for Sale, Authorize, Create Subscriptions, Delete Subscriptions,
 * Create Customer, Update Customer, Delete Customer and eCheck processing.
 *
 * @package        PayTrace Gateway
 */
class PayTraceAPI {

	/**
	 * Credit Card Processing Variables
	 *
	 * @var private $CC      - Credit card number
	 * @var private $EXPMNTH - Credit card expiration month - Expiration month must be the two-digit month of the credit cards expiration date
	 * @var private $EXPYR   - Credit card expiration year - Expiration year must be the two digit year of the credit cards expiration date
	 * @var private $AMOUNT  - Dollar amount of the transaction. Must be a positive number up to two decimal places.
	 * @var private $CSC     - CSC is the 3 or 4 digit code found on the signature line of the credit card. CSC is found on the front of Amex cards
	 * @var private $TRANXID - Transaction ID
	 */
	private $CC, $EXPMNTH, $EXPYR, $AMOUNT, $CSC, $TRANXID;

	/**
	 * @var private $request - It holds all set(needed) arguments to make a PayTrace request
	 * @var private $PSWD    - Paytrace API Password
	 * @var private $UN      - PayTrace API Username
	 */
	private $request, $PSWD, $UN;

	/**
	 * Customer Information Variables.
	 *
	 * @var private $BNAME     - Name that appears of the credit card
	 * @var private $BADDRESS  - Address that the credit card statement is delivered.
	 * @var private $BADDRESS2 - Second line of the address the credit card statement is delivered.
	 * @var private $BZIP      - Zip code that the credit card statement is delivered
	 * @var private $BCITY     - City that the credit card statement is delivered.
	 * @var private $BSTATE    - State that the credit card statement is delivered
	 * @var private $BCOUNTRY  - Country code where the billing address is located
	 * @var private $EMAIL     - Customer's email address where the sales receipt may be sent
	 * @var private $CUSTID    - Unique identifier for a customer profile. Each customer must have their own unique ID
	 */
	private $BADDRESS, $BZIP, $BNAME, $BADDRESS2, $BCITY, $BSTATE, $BCOUNTRY, $EMAIL, $CUSTID;

	/**
	 * Customer Shipping Information Variables.
	 *
	 * @var private $SNAME     - Name that appears of the credit card
	 * @var private $SADDRESS  - Address that the credit card statement is delivered.
	 * @var private $SADDRESS2 - Second line of the address the credit card statement is delivered.
	 * @var private $SZIP      - Zip code that the credit card statement is delivered
	 * @var private $SCITY     - City that the credit card statement is delivered.
	 * @var private $SSTATE    - State that the credit card statement is delivered
	 * @var private $SCOUNTRY  - Country code where the billing address is located
	 */
	private $SADDRESS, $SZIP, $SNAME, $SADDRESS2, $SCITY, $SSTATE, $SCOUNTRY;

	/**
	 * Response Variables. Can be returned in more than one type of request.
	 *
	 * @var private $RESPONSE      - Sentence or two that confirms the method that was requested
	 * @var private $ERROR         - PayTrace validates each name / value pair it receives. If any errors or inconsistencies in this data or the request, PayTrace will return an error. Each request may return multiple errors
	 *
	 * @var private $TRANSACTIONID - ID assigned by PayTrace to each transaction at the time the transaction is processed. TransactionID is returned with a successful call to ProcessTranx
	 *
	 * @var private $APPCODE       - Approval code is generated by credit card issuer and returned when a successful call to ProcessTranx is requested
	 * @var private $APPMSG        - Approval message is the textual response from the credit card issuer that is returned when a successful call to ProcessTranx is requested
	 * @var private $AVSRESPONSE   - The address verification system response is generated by the credit card issuer when a successful call to ProcessTranx is requested. AVS compares the billing address and billing zip code provided with the ProcessTranx request to the address where the customer's credit card statement is delivered. See How To Determine if a Transaction has been Approved page for possible AVS responses
	 *
	 * @var private $CSCRESPONSE   - The card security code response is generated by the credit card issuer when a successful call to the ProcessTranx is requested. The CSC provided with the ProcessTranx request. is compared to the CSC assigned to the credit card. See Appendix B for possible AVS responses
	 *
	 * @var private $CUSTOMERID    - ID assigned by PayTrace to each customer at the time the customer profile is created. CustomerID is returned with a successful call to CreateCustomer or UpdateCustomer
	 */
	private $RESPONSE, $ERROR, $TRANSACTIONID, $APPCODE, $APPMSG, $AVSRESPONSE, $CSCRESPONSE, $CUSTOMERID;

	/**
	 * Transaction Related Variables
	 *
	 * @var private $PROXY       - Proxy address if you are using one
	 * @var private $TEST        - The TEST attribute may be used with any of the transaction types (TranxType) of the ProcessTranx method. The value of the TEST attribute may only be a “Y”. Transactions processed with a TEST value of “Y” will be processed as test transactions with standardized test responses. Test transactions will not place a hold on the customer’s credit card
	 *
	 * @var private $METHOD      - Function that you are requesting PayTrace perform. Supported methods: ProcessTranx, createrecur, updaterecur, DeleteRecur,ProcessCheck, DeleteCustomer, CreateCustomer, UpdateCustomer.
	 *
	 * @var private $TRANXTYPE   - The transaction type is the type of transaction you wish to process if the METHOD is set to ProcessTranx. TRANXTYPE must be set to one of the following: Sale, Authorization, Str/Fwd, Refund, Void, Capture,Force and Deleteauthkey (Secure Checkout Only)
	 *
	 * @var private $DESCRIPTION - Optional text describing the transaction, products, customers, or other attributes of the transaction
	 * @var private $INVOICE     - Invoice is the identifier for this transaction in your accounting or inventory management system
	 */
	private $PROXY, $TEST, $METHOD, $TRANXTYPE, $DESCRIPTION, $INVOICE;

	/**
	 * Check Payment Related Variables. Both Request and Response.
	 *
	 * @var private $CHECKTYPE       - The transaction type is the type of transaction you wish to process if the METHOD is set to ProcessCheck. CHECKTYPE must be set to one of the following: Sale, Hold, Refund, Fund, or Void.
	 *
	 * @var private $DDA             - Checking account number for processing check transactions or managing customer profiles
	 * @var private $TR              - Transit routing number for processing check transactions or managing customer profiles
	 * @var private $CHECKIDENTIFIER - ID assigned by PayTrace to each check at the time the check is processed. CHECKIDENTIFIER is returned with a successful call to ProcessCheck
	 *
	 * @var private $ACHCODE         - Flag that is returned for checks processed through a real-time check processor. 0/zero indicates that the check was accepted
	 * @var private $ACHMSG          - Message returned from real-time check processor
	 */
	private $CHECKTYPE, $DDA, $TR, $CHECKIDENTIFIER, $ACHCODE, $ACHMSG, $CHECKID;

	public function __construct( $UN, $PSWD ) {
		if ( empty( $UN ) ) {
			throw new Exception( __( 'Error while setting up payment. Customer ID is required.', WC_PayTrace::TEXT_DOMAIN ) );
		}
		$this->setUN( (string) $UN );

		if ( empty( $PSWD ) ) {
			throw new Exception( __( 'Error while setting up payment. Gateway password is required', WC_PayTrace::TEXT_DOMAIN ) );
		}

		$this->setPSWD( (string) $PSWD );

		$this->TERMS           = "";
		$this->METHOD          = "";
		$this->TRANXTYPE       = "";
		$this->CC              = "";
		$this->EXPMNTH         = "";
		$this->EXPYR           = "";
		$this->AMOUNT          = "";
		$this->CSC             = "";
		$this->BADDRESS        = "";
		$this->BADDRESS2       = "";
		$this->BZIP            = "";
		$this->BNAME           = "";
		$this->BCITY           = "";
		$this->BSTATE          = "";
		$this->BCOUNTRY        = "";
		$this->DESCRIPTION     = "";
		$this->EMAIL           = "";
		$this->INVOICE         = "";
		$this->PROXY           = "";
		$this->TEST            = "";
		$this->CUSTID          = "";
		$this->CUSTOMERID      = "";
		$this->CHECKTYPE       = "";
		$this->DDA             = "";
		$this->TR              = "";
		$this->CHECKIDENTIFIER = "";
		$this->RESPONSE        = "";
		$this->ERROR           = "";
		$this->TRANSACTIONID   = "";
		$this->APPCODE         = "";
		$this->APPMSG          = "";
		$this->AVSRESPONSE     = "";
		$this->CSCRESPONSE     = "";
		$this->ACHCODE         = "";
		$this->ACHMSG          = "";
	}

	/*=================================
	= SETTERS
	=================================*/

	public function SetUN( $p_UN ) {
		$this->UN            = $p_UN;
		$this->request['UN'] = $this->UN;
	}

	public function SetPSWD( $p_PSWD ) {
		$this->PSWD            = $p_PSWD;
		$this->request['PSWD'] = $this->PSWD;
	}

	public function SetTERMS( $p_TERMS ) {
		$this->TERMS            = $p_TERMS;
		$this->request['TERMS'] = $this->TERMS;
	}

	public function SetMETHOD( $p_METHOD ) {
		$this->METHOD            = $p_METHOD;
		$this->request['METHOD'] = $this->METHOD;
	}

	public function SetTRANXTYPE( $p_TRANXTYPE ) {
		$this->TRANXTYPE            = $p_TRANXTYPE;
		$this->request['TRANXTYPE'] = $this->TRANXTYPE;
	}

	public function SetCC( $p_CC ) {
		$this->CC            = $p_CC;
		$this->request['CC'] = $this->CC;
	}

	public function SetEXPMNTH( $p_EXPMNTH ) {
		$this->EXPMNTH            = $p_EXPMNTH;
		$this->request['EXPMNTH'] = $this->EXPMNTH;
	}

	public function SetEXPYR( $p_EXPYR ) {
		$this->EXPYR            = $p_EXPYR;
		$this->request['EXPYR'] = $this->EXPYR;
	}

	public function SetAMOUNT( $p_AMOUNT ) {
		$this->AMOUNT            = $p_AMOUNT;
		$this->request['AMOUNT'] = $this->AMOUNT;
	}

	public function SetCSC( $p_CSC ) {
		$this->CSC            = $p_CSC;
		$this->request['CSC'] = $this->CSC;
	}

	public function SetBADDRESS( $p_BADDRESS ) {
		$this->BADDRESS            = substr( $p_BADDRESS, 0, 50 );
		$this->request['BADDRESS'] = $this->BADDRESS;
	}

	public function SetBZIP( $p_BZIP ) {
		$this->BZIP            = substr( $p_BZIP, 0, 9 );
		$this->request['BZIP'] = $this->BZIP;
	}

	public function SetBADDRESS2( $p_BADDRESS2 ) {
		$this->BADDRESS2            = substr( $p_BADDRESS2, 0, 50 );
		$this->request['BADDRESS2'] = $this->BADDRESS2;
	}

	public function SetBNAME( $p_BNAME ) {
		$this->BNAME            = substr( $p_BNAME, 0, 50 );
		$this->request['BNAME'] = $this->BNAME;
	}

	public function SetBCITY( $p_BCITY ) {
		$this->BCITY            = substr( $p_BCITY, 0, 50 );
		$this->request['BCITY'] = $this->BCITY;
	}

	public function SetBSTATE( $p_BSTATE ) {
		$this->BSTATE            = substr( $p_BSTATE, 0, 2 );
		$this->request['BSTATE'] = $this->BSTATE;
	}

	public function SetBCOUNTRY( $p_BCOUNTRY ) {
		$this->BCOUNTRY            = substr( $p_BCOUNTRY, 0, 2 );
		$this->request['BCOUNTRY'] = $this->BCOUNTRY;
	}

	public function SetSADDRESS( $p_SADDRESS ) {
		$this->SADDRESS            = substr( $p_SADDRESS, 0, 50 );
		$this->request['SADDRESS'] = $this->SADDRESS;
	}

	public function SetSZIP( $p_SZIP ) {
		$this->SZIP            = substr( $p_SZIP, 0, 9 );
		$this->request['SZIP'] = $this->SZIP;
	}

	public function SetSADDRESS2( $p_SADDRESS2 ) {
		$this->SADDRESS2            = substr( $p_SADDRESS2, 0, 50 );
		$this->request['SADDRESS2'] = $this->SADDRESS2;
	}

	public function SetSNAME( $p_SNAME ) {
		$this->SNAME            = substr( $p_SNAME, 0, 50 );
		$this->request['SNAME'] = $this->SNAME;
	}

	public function SetSCITY( $p_SCITY ) {
		$this->SCITY            = substr( $p_SCITY, 0, 50 );
		$this->request['SCITY'] = $this->SCITY;
	}

	public function SetSSTATE( $p_SSTATE ) {
		$this->SSTATE            = substr( $p_SSTATE, 0, 2 );
		$this->request['SSTATE'] = $this->SSTATE;
	}

	public function SetSCOUNTRY( $p_SCOUNTRY ) {
		$this->SCOUNTRY            = substr( $p_SCOUNTRY, 0, 2 );
		$this->request['SCOUNTRY'] = $this->SCOUNTRY;
	}

	public function SetDESCRIPTION( $p_DESCRIPTION ) {
		$p_DESCRIPTION = str_replace( array( '|', '~' ), '', $p_DESCRIPTION );

		$this->DESCRIPTION            = substr( $p_DESCRIPTION, 0, 255 );
		$this->request['DESCRIPTION'] = $this->DESCRIPTION;
	}

	public function SetEMAIL( $p_EMAIL ) {
		$this->EMAIL            = substr( $p_EMAIL, 0, 50 );
		$this->request['EMAIL'] = $this->EMAIL;
	}

	public function SetINVOICE( $p_INVOICE ) {
		$this->INVOICE            = substr( $p_INVOICE, 0, 50 );
		$this->request['INVOICE'] = $this->INVOICE;
	}

	public function SetPROXY( $p_PROXY ) {
		$this->PROXY = $p_PROXY;
	}

	public function SetTEST( $p_TEST ) {
		$this->TEST            = $p_TEST;
		$this->request['TEST'] = $this->TEST;
	}

	public function SetCUSTID( $p_CUSTID ) {
		$this->CUSTID            = $p_CUSTID;
		$this->request['CUSTID'] = $this->CUSTID;
	}

	public function SetCHECKID( $p_CHECKID ) {
		$this->CHECKID            = $p_CHECKID;
		$this->request['CHECKID'] = $this->CHECKID;
	}

	public function setTRANXID( $p_TRANXID ) {
		$this->TRANXID            = $p_TRANXID;
		$this->request['TRANXID'] = $this->TRANXID;
	}

	public function SetCHECKTYPE( $p_CHECKTYPE ) {
		$this->CHECKTYPE            = $p_CHECKTYPE;
		$this->request['CHECKTYPE'] = $this->CHECKTYPE;
	}

	public function SetDDA( $p_DDA ) {
		$this->DDA            = $p_DDA;
		$this->request['DDA'] = $this->DDA;
	}

	public function SetTR( $p_TR ) {
		$this->TR            = $p_TR;
		$this->request['TR'] = $this->TR;
	}

	/*=================================
	= GETTERS
	=================================*/

	// For debug
	public function GetDEBUG_REQUEST() {
		return array(
			'TERMS'           => $this->TERMS,
			'METHOD'          => $this->METHOD,
			'TRANXTYPE'       => $this->TRANXTYPE,
			'AMOUNT'          => $this->AMOUNT,
			'BADDRESS'        => $this->BADDRESS,
			'BZIP'            => $this->BZIP,
			'BADDRESS2'       => $this->BADDRESS2,
			'BNAME'           => $this->BNAME,
			'BCITY'           => $this->BCITY,
			'BSTATE'          => $this->BSTATE,
			'BCOUNTRY'        => $this->BCOUNTRY,
			'SADDRESS'        => $this->SADDRESS,
			'SZIP'            => $this->SZIP,
			'SADDRESS2'       => $this->SADDRESS2,
			'SNAME'           => $this->SNAME,
			'SCITY'           => $this->SCITY,
			'SSTATE'          => $this->SSTATE,
			'SCOUNTRY'        => $this->SCOUNTRY,
			'DESCRIPTION'     => $this->DESCRIPTION,
			'EMAIL'           => $this->EMAIL,
			'INVOICE'         => $this->INVOICE,
			'TEST'            => $this->TEST,
			'CUSTID'          => $this->CUSTID,
			'CUSTOMERID'      => $this->CUSTOMERID,
			'CHECKTYPE'       => $this->CHECKTYPE,
			'DDA'             => $this->DDA,
			'TR'              => $this->TR,
			'CHECKIDENTIFIER' => $this->CHECKIDENTIFIER,
			'CHECKID'         => $this->CHECKID,
			'TRANXID'         => $this->TRANXID,
		);
	}

	// For debug
	public function GetDEBUG_RESPONSE() {
		return array(
			'RESPONSE'        => $this->GetRESPONSE(),
			'APPCODE'         => $this->GetAPPCODE(),
			'APPMSG'          => $this->GetAPPMSG(),
			'TRANSACTIONID'   => $this->GetTRANSACTIONID(),
			'AVSRESPONSE'     => $this->GetAVSRESPONSE(),
			'CSCRESPONSE'     => $this->GetCSCRESPONSE(),
			'ERROR'           => $this->GetERROR(),
			'CUSTOMERID'      => $this->GetCUSTOMERID(),
			'CUSTID'          => $this->GetCUSTID(),
			'ACHCODE'         => $this->GetACHCODE(),
			'ACHMSG'          => $this->GetACHMSG(),
			'CHECKIDENTIFIER' => $this->GetCHECKIDENTIFIER()
		);
	}

	public function GetRESPONSE() {
		return $this->RESPONSE;
	}

	public function GetERROR() {
		return $this->ERROR;
	}

	public function GetTRANSACTIONID() {
		return $this->TRANSACTIONID;
	}

	public function GetAPPCODE() {
		return $this->APPCODE;
	}

	public function GetAPPMSG() {
		return $this->APPMSG;
	}

	public function GetAVSRESPONSE() {
		return $this->AVSRESPONSE;
	}

	public function GetCSCRESPONSE() {
		return $this->CSCRESPONSE;
	}

	public function GetCHECKIDENTIFIER() {
		return $this->CHECKIDENTIFIER;
	}

	public function GetCUSTOMERID() {
		return $this->CUSTOMERID;
	}

	public function GetCUSTID() {
		return $this->CUSTID;
	}

	public function GetACHCODE() {
		return $this->ACHCODE;
	}

	public function GetACHMSG() {
		return $this->ACHMSG;
	}

	public function __destruct() {
		$this->UN              = "";
		$this->PSWD            = "";
		$this->TERMS           = "";
		$this->METHOD          = "";
		$this->TRANXTYPE       = "";
		$this->CC              = "";
		$this->EXPMNTH         = "";
		$this->EXPYR           = "";
		$this->AMOUNT          = "";
		$this->CSC             = "";
		$this->BADDRESS        = "";
		$this->BADDRESS2       = "";
		$this->BZIP            = "";
		$this->BNAME           = "";
		$this->BCITY           = "";
		$this->BSTATE          = "";
		$this->BCOUNTRY        = "";
		$this->DESCRIPTION     = "";
		$this->EMAIL           = "";
		$this->INVOICE         = "";
		$this->PROXY           = "";
		$this->TEST            = "";
		$this->CUSTID          = "";
		$this->CUSTOMERID      = "";
		$this->CHECKTYPE       = "";
		$this->DDA             = "";
		$this->TR              = "";
		$this->CHECKIDENTIFIER = "";

		$this->RESPONSE      = "";
		$this->ERROR         = "";
		$this->TRANSACTIONID = "";
		$this->APPCODE       = "";
		$this->APPMSG        = "";
		$this->AVSRESPONSE   = "";
		$this->CSCRESPONSE   = "";
		$this->ACHCODE       = "";
		$this->ACHMSG        = "";
		$this->CHECKID       = "";
	}

	/**
	 * Process the PayTrace request.
	 * Runs the methods needed to generate and send the request, then parses the response.
	 *
	 * @param string $request_type (Deprecated)Request type. Possible values are 'payment', 'customer', 'subscription', 'check', 'refund'
	 */
	public function ProcessRequest( $request_type = 'payment' ) {
		$get_param_list = $this->format_request();

		$request_response = $this->SendRequest( $get_param_list );
		$this->ParseResponse( $request_response, $request_type );
	}

	/**
	 * Generated the request string for the request
	 */
	private function format_request() {
		if ( empty( $this->request ) ) {
			throw new Exception( 'The parameter for the request to PayTrace are not set correctly' );
		}

		$l_parmlist = '';
		// Go through the set parameters and generate the request string
		foreach ( $this->request as $key => $value ) {
			$l_parmlist .= $key . '~' . $value . '|';
		}

		$l_parmlist = "ParmList=" . urlencode( $l_parmlist );

		return $l_parmlist;
	}

	/**
	 * Send the PayTrace request and return its response body
	 *
	 * @param string $send_param_list
	 *
	 * @return mixed The response body
	 * @throws Exception
	 */
	private function SendRequest( $send_param_list ) {

		if ( ! empty( $this->PROXY ) ) {
			// Set up CURL request if the customer uses proxy
			// Do this to be easier for the customer to use the proxy with PayTrace
			$l_Header = array( "MIME-Version: 1.0", "Content-type: application/x-www-form-urlencoded", "Contenttransfer-encoding: text" );
			$l_URL    = "https://paytrace.com/api/default.pay";
			$l_CURL   = curl_init();
			curl_setopt( $l_CURL, CURLOPT_URL, $l_URL );
			curl_setopt( $l_CURL, CURLOPT_VERBOSE, 1 );
			curl_setopt( $l_CURL, CURLOPT_PROXYTYPE, CURLPROXY_HTTP );
			curl_setopt( $l_CURL, CURLOPT_HTTPHEADER, $l_Header );
			// Set up proxy if one is needed by the hosting server
			if ( ! empty( $this->PROXY ) ) {
				curl_setopt( $l_CURL, CURLOPT_PROXY, $this->PROXY );
			}
			curl_setopt( $l_CURL, CURLOPT_SSL_VERIFYPEER, true );
			curl_setopt( $l_CURL, CURLOPT_SSL_VERIFYHOST, 2 );
			curl_setopt( $l_CURL, CURLOPT_POST, true );
			curl_setopt( $l_CURL, CURLOPT_POSTFIELDS, $send_param_list );
			curl_setopt( $l_CURL, CURLOPT_RETURNTRANSFER, true );
			curl_setopt( $l_CURL, CURLOPT_TIMEOUT, 30 );
			$request_response = curl_exec( $l_CURL );
			curl_close( $l_CURL );
		} else {
			$post_args = array(
				'headers'   => array(
					'MIME-Version' => '1.0',
					'Content-Type' => 'application/x-www-form-urlencoded',
				),
				'method'    => 'POST',
				'body'      => $send_param_list,
				'sslverify' => apply_filters( 'https_local_ssl_verify', true ),
				'timeout'   => 30,
			);
			$url       = "https://paytrace.com/api/default.pay";

			$response = wp_remote_post( $url, $post_args );

			if ( is_wp_error( $response ) ) {
				throw new Exception( sprintf( __( 'Error while processing your payment. Error message: %s.', WC_PayTrace::TEXT_DOMAIN ), $response->get_error_message() ) );
			} else {
				$request_response = $response['body'];
			}
		}

		return $request_response;
	}

	/**
	 * Parses the request response and places all parameters into accessible class variables
	 *
	 * @param string $p_Response The response string
	 */
	private function ParseResponse( $p_Response ) {
		if ( $p_Response == "" ) {
			$this->ERROR .= "The request did not receive a valid response from the network, please try again.";
		} else {
			$l_HasAPipe  = strpos( $p_Response, "|" );
			$l_HasATilde = strpos( $p_Response, "~" );
			if ( $l_HasAPipe === false || $l_HasATilde == false ) {
				$this->ERROR .= "The request did not receive a valid response from the network, please try again.";
			} else {
				// Parse through the response and put pairs into an array
				$l_ResponseArray = explode( '|', $p_Response );
				$l_ArrayOfPairs  = array();
				foreach ( $l_ResponseArray as $l_NameValuePair ) {
					$l_HasATilde = strpos( $l_NameValuePair, "~" );
					if ( $l_HasATilde == true ) {
						$l_NameValuePairArray                       = explode( '~', $l_NameValuePair );
						$l_ArrayOfPairs[ $l_NameValuePairArray[0] ] = $l_NameValuePairArray[1];
					}
				}

				// Loop through the array of pairs and put the data into the class properties
				foreach ( $l_ArrayOfPairs as $l_Name => $l_Value ) {
					if ( $l_Name == "RESPONSE" ) {
						$this->RESPONSE = $l_Value;
					} elseif ( $l_Name == "ERROR" ) {
						$this->ERROR .= $l_Value;
					} elseif ( $l_Name == "TRANSACTIONID" ) {
						$this->TRANSACTIONID = $l_Value;
					} elseif ( $l_Name == "APPCODE" ) {
						$this->APPCODE = $l_Value;
					} elseif ( $l_Name == "APPMSG" ) {
						$this->APPMSG = $l_Value;
					} elseif ( $l_Name == "AVSRESPONSE" ) {
						$this->AVSRESPONSE = $l_Value;
					} elseif ( $l_Name == "CSCRESPONSE" ) {
						$this->CSCRESPONSE = $l_Value;
					} elseif ( $l_Name == "CHECKIDENTIFIER" ) {
						$this->CHECKIDENTIFIER = $l_Value;
					} elseif ( $l_Name == "CUSTOMERID" ) {
						$this->CUSTOMERID = $l_Value;
					} elseif ( $l_Name == "CUSTID" ) {
						$this->CUSTID = $l_Value;
					} elseif ( $l_Name == "ACHCODE" ) {
						$this->ACHCODE = $l_Value;
					} elseif ( $l_Name == "ACHMSG" ) {
						$this->ACHMSG = $l_Value;
					}
				}
			}
		}
	}

	public function DidErrorOccur() {
		$DidErrorOccur = false;
		if ( $this->ERROR != '' ) {
			$DidErrorOccur = true;
		}

		return $DidErrorOccur;
	}

	public function WasTransactionApproved() {
		$l_WasApproved = false;
		if ( $this->DidErrorOccur() == false ) {
			if ( $this->APPCODE != '' ) {
				$l_WasApproved = true;
			}
		}

		return $l_WasApproved;
	}

	public function WasAchTransactionApproved() {
		$request_approved = false;
		if ( $this->DidErrorOccur() == false ) {
			if ( $this->RESPONSE != '' && $this->CHECKIDENTIFIER != '' ) {
				$request_approved = true;
			}
		}

		return $request_approved;
	}

	public function WasCustomerRequestApproved() {
		$request_approved = false;
		if ( $this->DidErrorOccur() == false ) {
			if ( $this->CUSTOMERID != '' ) {
				$request_approved = true;
			}
		}

		return $request_approved;
	}

	public function was_refund_approved( $type ) {
		$refund_approved = false;
		if ( $this->DidErrorOccur() == false ) {
			if ( 'check' == $type ) {
				// We processed Check method
				if ( '' != $this->CHECKIDENTIFIER && '' != $this->RESPONSE ) {
					$refund_approved = true;
				}
			} else {
				// We processed Card method
				if ( '' != $this->TRANSACTIONID && '' != $this->RESPONSE ) {
					$refund_approved = true;
				}
			}
		}

		return $refund_approved;
	}
}