<?php

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * WC_Gateway_PayTrace Class
 *
 * WC_Gateway_PayTrace manages the main functionality of the PayTrace extension.
 * Managed: Credit Card and Check payments, creation and management of customer payment profiles.
 *
 * @package PayTrace Gateway
 * @author  VanboDevelops | Ivan Andreev
 */
class WC_Gateway_PayTrace extends WC_Payment_Gateway {

	/* Holds what type of payment the customer chose */
	public $what_payment_customer_used;

	public static $count = 0;

	private $cc_type = '';
	private $cc_number = '';
	private $cc_exp_month = '';
	private $cc_exp_year = '';
	private $cc_cvc = '';

	public function __construct() {

		$this->id           = 'paytrace';
		$this->icon         = apply_filters( 'woocommerce_paytrace_icon', plugin_dir_url( __FILE__ ) . '../paytrace_cc.png' );
		$this->has_fields   = true;
		$this->method_title = __( 'PayTrace', WC_PayTrace::TEXT_DOMAIN );
		$this->card_options = apply_filters(
			'paytrace_card_options', array(
				'VISA'     => __( 'Visa', WC_PayTrace::TEXT_DOMAIN ),
				'MC'       => __( 'Master Card', WC_PayTrace::TEXT_DOMAIN ),
				'AMEX'     => __( 'American Express', WC_PayTrace::TEXT_DOMAIN ),
				'DISCOVER' => __( 'Discover', WC_PayTrace::TEXT_DOMAIN ),
				'JCB'      => __( 'JCB', WC_PayTrace::TEXT_DOMAIN )
			)
		);
		$this->check_pass   = false;

		$this->supports = array(
			'subscriptions',
			'products',
			'subscription_cancellation',
			'subscription_reactivation',
			'subscription_suspension',
			'subscription_amount_changes',
			'subscription_payment_method_change', // Subs 1.n compatibility
			'subscription_payment_method_change_customer',
			'subscription_payment_method_change_admin',
			'subscription_date_changes',
			'multiple_subscriptions',
			'refunds',
			'pre-orders',
		);

		// Load the form fields.
		$this->init_form_fields();

		// Load the settings.
		$this->init_settings();

		// Define user set variables
		$this->enabled                = $this->get_option( 'enabled', 'no' );
		$this->title                  = $this->get_option( 'title' );
		$this->description            = $this->get_option( 'description' );
		$this->testmode               = $this->get_option( 'testmode', 'yes' );
		$this->user_name              = $this->get_option( 'user_name' );
		$this->password               = $this->get_option( 'password' );
		$this->trans_type             = $this->get_option( 'trans_type', 'Sale' );
		$this->available_cc           = $this->get_option( 'available_cc' );
		$this->debug                  = $this->get_option( 'debug', 'no' );
		$this->save_customers         = $this->get_option( 'save_customers', 'yes' );
		$this->save_card_text         = $this->get_option( 'save_card_text' );
		$this->support_check          = $this->get_option( 'support_check', 'yes' );
		$this->ach_payment_onhold     = $this->get_option( 'ach_payment_onhold', 'yes' );
		$this->send_order_description = $this->get_option( 'send_order_description', 'yes' );

		// Add Refunds support
		if ( 'yes' == $this->save_customers ) {
			$this->supports = array_merge(
				$this->supports,
				array(
					'refunds',
				)
			);
		}

		// Actions
		add_action( 'woocommerce_receipt_paytrace', array( $this, 'receipt_page' ) );

		// Save options
		add_action( 'woocommerce_update_options_payment_gateways', array( $this, 'process_admin_options' ) );
		add_action( 'woocommerce_update_options_payment_gateways_' . $this->id, array( $this, 'process_admin_options' ) );

		// Disable if currency is not supported
		if ( ! $this->is_valid_currency() ) {
			$this->enabled = false;
		}

	}

	/**
	 * Check if the store currency is supported by the gateway
	 */
	function is_valid_currency() {
		if ( ! in_array( get_woocommerce_currency(), apply_filters( 'wc_paytrace_allowed_currency', array( 'USD' ) ) ) ) {
			return false;
		}

		return true;
	}

	/**
	 * Admin Panel Options
	 */
	public function admin_options() {
		?>
		<h3><?php _e( 'PayTrace Gateway', WC_PayTrace::TEXT_DOMAIN ); ?></h3>
		<p><?php _e(
				'PayTrace Gateway enables your store to accept credit card, Check/ACH and subscription payments straight from your checkout page.
				The customer will enter his/her payment information and be presented the \'Thank You\' page on successful payment.', WC_PayTrace::TEXT_DOMAIN
			);
			?>
			<br /><span style="color: red;"><strong><?php _e( 'PayTrace accepts payments in US Dollars (USD) only.', WC_PayTrace::TEXT_DOMAIN ); ?></strong></span>
		</p>
		<table class="form-table">
			<?php
			if ( $this->is_valid_currency() ) {

				// Generate the HTML For the settings form.
				$this->generate_settings_html();

			} else {
				?>
				<div class="inline error"><p><strong>
							<?php _e( 'Gateway Disabled', WC_PayTrace::TEXT_DOMAIN ); ?>
						</strong>: <?php _e( 'PayTrace does not support your store currency.', WC_PayTrace::TEXT_DOMAIN ); ?>
					</p>
				</div>
				<?php
			}
			?>
		</table><!--/.form-table-->
		<?php
		ob_start();
		?>
		$('#woocommerce_paytrace_save_customers').change(
		function() {
		var saveCustomersText = $(this).closest('tr').next();

		if ( $(this).is(':checked') ) {
		saveCustomersText.show();
		} else {
		saveCustomersText.hide();
		}
		}).change();

		$('#woocommerce_paytrace_support_check').change(
		function() {
		var markEcheckOnHold = $(this).closest('tr').next();

		if ( $(this).is(':checked') ) {
		markEcheckOnHold.show();
		} else {
		markEcheckOnHold.hide();
		}
		}).change();
		<?php
		$javascript = ob_get_clean();
		WC_Compat_PayTrace::wc_include_js( $javascript );
	} // End admin_options()

	/**
	 * Initialise Gateway Settings Form Fields
	 **/
	function init_form_fields() {

		$this->form_fields = array(
			'enabled'                => array(
				'title'   => __( 'Enable/Disable', WC_PayTrace::TEXT_DOMAIN ),
				'type'    => 'checkbox',
				'label'   => __( 'Enable PayTrace', WC_PayTrace::TEXT_DOMAIN ),
				'default' => 'no'
			),
			'testmode'               => array(
				'title'       => __( 'Test Mode', WC_PayTrace::TEXT_DOMAIN ),
				'type'        => 'checkbox',
				'label'       => __( 'Enable PayTrace Test Mode', WC_PayTrace::TEXT_DOMAIN ),
				'description' => __(
					'This option adds a test parameter to the payment request.
 									<br/>It is not required to be enabled in order to make a test payment.
 									If you use a <strong>test PayTrace User Name</strong> and <strong>test PayTrace Password</strong>,
 								 	then you will be able to make test payments, even if test mode is disabled.
 								  	<br/>When in test mode, a test payment will not return AVS and CSC match responses.
 								  	<br/><strong>Disable Test Mode to accept live payments.</strong>', WC_PayTrace::TEXT_DOMAIN
				),
				'default'     => 'yes'
			),
			'title'                  => array(
				'title'       => __( 'Method Title', WC_PayTrace::TEXT_DOMAIN ),
				'type'        => 'text',
				'description' => __( 'This controls the title which the user sees during checkout.', WC_PayTrace::TEXT_DOMAIN ),
				'default'     => __( 'Credit/Debit Card', WC_PayTrace::TEXT_DOMAIN )
			),
			'description'            => array(
				'title'       => __( 'Description', WC_PayTrace::TEXT_DOMAIN ),
				'type'        => 'textarea',
				'description' => __(
					'This controls the description which the user sees during checkout.
									Will show above the credit card fields.', WC_PayTrace::TEXT_DOMAIN
				),
				'default'     => __( "Please make sure you enter your correct billing information.", WC_PayTrace::TEXT_DOMAIN )
			),
			'user_name'              => array(
				'title'       => __( 'PayTrace User Name', WC_PayTrace::TEXT_DOMAIN ),
				'type'        => 'text',
				'description' => __( 'Your PayTrace user name is required to authenticate your request.', WC_PayTrace::TEXT_DOMAIN ),
				'default'     => ''
			),
			'password'               => array(
				'title'       => __( 'PayTrace Password', WC_PayTrace::TEXT_DOMAIN ),
				'type'        => 'password',
				'description' => __(
					'Your PayTrace password is required to authenticate your request.
				 					The Password field will always appear empty. Just enter the password,
				 				 	when you want to set it. Leave empty otherwise.', WC_PayTrace::TEXT_DOMAIN
				),
				'placeholder' => 'API Password',
				'default'     => ''
			),
			'send_order_description' => array(
				'type'        => 'checkbox',
				'label'       => __( 'Send Order Description', WC_PayTrace::TEXT_DOMAIN ),
				'description' => __(
					"Enable to send the order description to PayTrace.
									Description will contain the order items and/or the order number.", WC_PayTrace::TEXT_DOMAIN
				),
				'default'     => 'yes'
			),
			'available_cc'           => array(
				'title'       => __( 'Accepted Cards', WC_PayTrace::TEXT_DOMAIN ),
				'type'        => 'multiselect',
				'description' => __( 'Choose the cards you can and want to accept payments from.', WC_PayTrace::TEXT_DOMAIN ),
				'options'     => $this->card_options,
				'class'       => 'chosen_select',
				'css'         => 'min-width: 350px;',
				'default'     => array( 'VISA', 'MC', 'AMEX', 'DISCOVER', 'JCB' )
			),
			'trans_type'             => array(
				'title'       => __( 'Transaction Type', WC_PayTrace::TEXT_DOMAIN ),
				'type'        => 'select',
				'description' => __( 'The transaction type is the type of transaction you wish to process.', WC_PayTrace::TEXT_DOMAIN ) . '<br/>' .
					sprintf( __( '%sSale%s means that the transaction is authorized and settled automatically.', WC_PayTrace::TEXT_DOMAIN ), '<strong>', '</strong>' ) . '<br/>' .
					sprintf( __( '%sAuthorization%s means that the transaction is only authorized and will need to be manually settled.', WC_PayTrace::TEXT_DOMAIN ), '<strong>', '</strong>' ),
				'options'     => array( 'Sale' => 'Sale', 'Authorization' => 'Authorization' ),
				'class'       => 'chosen_select',
				'css'         => 'min-width: 350px;',
				'default'     => ''
			),
			'support_check'          => array(
				'title'       => __( 'Check Payment', WC_PayTrace::TEXT_DOMAIN ),
				'type'        => 'checkbox',
				'label'       => __( 'Enable Check Payments', WC_PayTrace::TEXT_DOMAIN ),
				'description' => __( 'Check the box, if you want to accept Check(ACH) payments.', WC_PayTrace::TEXT_DOMAIN ),
				'default'     => 'yes'
			),
			'ach_payment_onhold'     => array(
				'title'       => __( 'Mark Orders Paid by Check as On-Hold', WC_PayTrace::TEXT_DOMAIN ),
				'type'        => 'checkbox',
				'label'       => __( 'Mark Orders Paid by Check as On-Hold', WC_PayTrace::TEXT_DOMAIN ),
				'description' => __(
					"If enabled, it will mark the orders paid by Check as 'On-Hold'
									(Recommended: because the Check payments may be accepted, but still not processed),
								 	otherwise those orders will be marked as 'Processing'.", WC_PayTrace::TEXT_DOMAIN
				),
				'default'     => 'yes'
			),
			'save_customers'         => array(
				'title'       => __( "Save Customers", WC_PayTrace::TEXT_DOMAIN ),
				'type'        => 'checkbox',
				'label'       => __( "Save Customers Payment Info on Secure PayTrace Server", WC_PayTrace::TEXT_DOMAIN ),
				'description' => __(
					"Check if you want to save the customers payment info on PayTrace secure servers.
 									This option will allow the customers to save their info and use it in future payments without entering their information again.
 									<br/> Only the last 4 digits of the Credit Card, expiration date and their generated customer ID will be saved in the WooCommerce database.
 								 	Customer will be able to delete or edit their saved information from their 'My Account' Page.", WC_PayTrace::TEXT_DOMAIN
				),
				'default'     => 'yes'
			),
			'save_card_text'         => array(
				'title'       => __( 'Save Your Card Text', WC_PayTrace::TEXT_DOMAIN ),
				'type'        => 'textarea',
				'description' => __( 'Enter the text to prompt your customers to save their credit card info for easier future payments.', WC_PayTrace::TEXT_DOMAIN ),
				'default'     => 'Securely Save Your Details.<br/><span class="help">Don\'t enter your credit card info everytime,
 								save it to PayTrace secure servers and just use it next time.</span>'
			),
			'proxy'                  => array(
				'title'       => __( 'PayTrace HTTPS Proxy Server', WC_PayTrace::TEXT_DOMAIN ),
				'type'        => 'text',
				'description' => __(
					"If your web host requires a proxy server for a SSL connection,
				 					enter here the proxy server and port. Leave empty, if you don't require a proxy server.", WC_PayTrace::TEXT_DOMAIN
				),
				'default'     => ''
			),
			'debug'                  => array(
				'title'       => __( 'Debug Log', WC_PayTrace::TEXT_DOMAIN ),
				'type'        => 'checkbox',
				'label'       => __( 'Recommended: Test Mode only', WC_PayTrace::TEXT_DOMAIN ),
				'default'     => 'no',
				'description' => sprintf(
					__( 'Debug log will provide you with most of the data and events generated by the payment process. Logged inside %s' ),
					'<code>' . WC_Compat_PayTrace::wc_get_debug_file_path( 'paytrace' ) . '</code>'
				),
			),
		);

	} // End init_form_fields()

	/**
	 * Filter the gateway icon according to the accepted cards option
	 *
	 * @return string The Card images in a html format
	 */
	function get_icon() {
		if ( $this->available_cc ) {
			$icon = '';
			foreach ( $this->available_cc as $card ) {
				if ( file_exists( WC_PayTrace::plugin_path() . '/assets/images/' . strtolower( $card ) . '.png' ) ) {
					$icon .= '<img src="' . esc_url( WC_Compat_PayTrace::force_https( WC_PayTrace::plugin_url() . '/assets/images/' . strtolower( $card ) . '.png' ) ) . '" alt="' . esc_attr( strtolower( $card ) ) . '" />';
				}
			}
			if ( 'yes' == $this->support_check ) {
				if ( file_exists( WC_PayTrace::plugin_path() . '/assets/images/echeck.png' ) ) {
					$icon .= '<img src="' . esc_url( WC_Compat_PayTrace::force_https( WC_PayTrace::plugin_url() . '/assets/images/echeck.png' ) ) . '" alt="echeck" />';
				}
			}
		} else {
			$icon = '<img src="' . esc_url( WC_Compat_PayTrace::force_https( $this->icon ) ) . '" alt="' . esc_attr( $this->title ) . '" />';
		}

		return $icon;
	}

	/**
	 * Show PayTrace payment fields
	 *
	 * TODO:        There are a lot of checks performed in order to display
	 *          the Saved Profiles or/and the Save Your Details checkbox.
	 *                        Look into optimizing this for readability and complexity.
	 **/
	function payment_fields() {
		if ( $this->description ) {
			echo wpautop( wptexturize( $this->description ) );
		}

		$saved_profiles = array();
		if ( 'yes' == $this->save_customers ) {
			if ( is_user_logged_in() ) {
				$saved_profiles = $this->get_customer_saved_profiles();
			}
		}

		$support_checks     = ( 'yes' == $this->support_check ) ? true : false;
		$show_save_customer = false;
		$order              = WC_Compat_PayTrace::wc_get_order( WC_PayTrace::get_get( 'order_id' ) );

		if ( 'yes' == $this->save_customers ) {
			if (
				WC_PayTrace::is_subscriptions_active()
				&&
				// Order contains Subscriptions or Pre-orders
				(
				$order ? $this->order_contains_subscription( $order ) : false
					||
					// Cart contains Subscriptions or Pre-orders
					WC_Subscriptions_Cart::cart_contains_subscription()
				)
			) {
				// We always save customers in Subscriptions or Pre-orders
				// No need to show the option
				$show_save_customer = false;
			} elseif (
				WC_PayTrace::is_pre_orders_active()
				&&
				(
					// Order contains Pre-orders
				$order ? $this->order_contains_pre_order( $order ) : false
					||
					// Cart contains Pre-orders
					WC_Pre_Orders_Cart::cart_contains_pre_order()
				)
			) {
				$show_save_customer = false;
			} else {
				$show_save_customer = true;
			}
		}

		$saved_checks = array();
		$saved_cards  = array();
		if ( $saved_profiles ) {
			foreach ( $saved_profiles as $n => $profile ) {
				if ( isset( $profile['tr_last4'] ) ) {
					$saved_checks[ $n ] = $profile;
				} elseif ( isset( $profile['last4'] ) ) {
					$saved_cards[ $n ] = $profile;
				}
			}
		}

		// Load the Cards/Check forms switch
		if ( $support_checks ) {
			// Load the Checks form template
			WC_Compat_PayTrace::wc_get_template(
				'checkout/paytrace-switch-forms.php',
				array(
					'gateway' => $this->id,
				),
				'',
				WC_PayTrace::plugin_path() . '/templates/'
			);
		}

		// Do we save Customer Profiles?
		if ( 'yes' == $this->save_customers ) {

			// Do we have saved CC profiles?
			if ( $saved_cards ) {
				// Load the saved cards form template
				WC_Compat_PayTrace::wc_get_template(
					'checkout/paytrace-saved-cards.php',
					array(
						'saved_cards' => $saved_cards,
						'gateway'     => $this->id,
					),
					'',
					WC_PayTrace::plugin_path() . '/templates/'
				);
			} /// We don't have saved CC Profiles
			else {
				// Output a the new CC card used field
				?>
				<input hidden="hidden" type="radio" id="new" name="paytrace-used-cc" style="width:auto;" <?php checked( 1, 1 ) ?> value="new" />
				<?php
			}

			// Do we support checks?
			if ( $support_checks ) {

				// Are there any saved checks?
				if ( $saved_checks ) {

					// Load the saved checks form template
					WC_Compat_PayTrace::wc_get_template(
						'checkout/paytrace-saved-checks.php',
						array(
							'gateway'      => $this->id,
							'saved_checks' => $saved_checks,
						),
						'',
						WC_PayTrace::plugin_path() . '/templates/'
					);
				} // Output the new Check used field
				else {
					?>
					<input hidden="hidden" type="radio" id="new" name="paytrace-used-check" style="width:auto;" <?php checked( 1, 1 ) ?> value="new" />
					<?php
				}
			}
		} // We don't save Customer Profiles?
		else {
			// Output a the new CC card used field
			?>
			<input hidden="hidden" type="radio" id="new" name="paytrace-used-cc" style="width:auto;" <?php checked( 1, 1 ) ?> value="new" />
			<?php if ( $support_checks ) { // Output the new Check used field, if we support Check payments ?>
				<input hidden="hidden" type="radio" id="new" name="paytrace-used-check" style="width:auto;" <?php checked( 1, 1 ) ?> value="new" />
			<?php }
		}

		// Checks template loaded after Saved profiles
		// Do we support Check payments?
		if ( $support_checks ) {
			// Load the checks form template
			WC_Compat_PayTrace::wc_get_template(
				'checkout/paytrace-checks-form.php',
				array(
					'save_customers'     => $this->save_customers,
					'gateway'            => $this->id,
					'save_card_text'     => $this->save_card_text,
					'show_save_customer' => $show_save_customer,
				),
				'',
				WC_PayTrace::plugin_path() . '/templates/'
			);
		}

		// Load the credit card form template
		WC_Compat_PayTrace::wc_get_template(
			'checkout/paytrace-payment-form.php',
			array(
				'save_customers'     => $this->save_customers,
				'saved_profiles'     => $saved_profiles,
				'gateway'            => $this->id,
				'available_cc'       => $this->available_cc,
				'card_options'       => $this->card_options,
				'save_card_text'     => $this->save_card_text,
				'show_save_customer' => $show_save_customer
			),
			'',
			WC_PayTrace::plugin_path() . '/templates/'
		);
	}

	/**
	 * Process the payment and return the result
	 **/
	function process_payment( $order_id ) {
		try {
			if ( ! $this->check_pass ) {
				// Check payment info before it is send to the gateway
				$this->validate_payment_information();
			}

			$order = WC_Compat_PayTrace::wc_get_order( $order_id );

			// Using Check(ACH) to pay
			if ( 'check' == $this->what_payment_customer_used
				|| 'saved_check' == $this->what_payment_customer_used
			) {
				$this->process_ach_payment( $order );
			} else { // Using Card to pay
				$this->process_card_payment( $order );
			}

			// Empty cart
			WC_Compat_PayTrace::empty_cart();

			// Redirect to the thank you page
			return array(
				'result'   => 'success',
				'redirect' => $this->get_return_url( $order )
			);

		}
		catch ( Exception $e ) {
			WC_Compat_PayTrace::wc_add_notice( $e->getMessage(), 'error' );
		}

	}

	/**
	 * Process automatic refunds
	 *
	 * @since 1.3
	 *
	 * @param int    $order_id
	 * @param null   $amount
	 * @param string $reason
	 *
	 * @return bool|WP_Error
	 */
	public function process_refund( $order_id, $amount = null, $reason = '' ) {

		try {
			$order = WC_Compat_PayTrace::wc_get_order( $order_id );

			// Get the saved card
			$customer_id    = $this->get_customer_id_from_order( $order );
			$transaction_id = get_post_meta( $order->id, '_paytrace_transaction_id', true );
			$payment_type   = get_post_meta( $order->id, '_paytrace_payment_type', true );

			// Bail, if there is no reference ID
			if ( '' == $customer_id && '' == $transaction_id ) {
				throw new Exception( __( 'Error: Missing Reference ID. The order does not have all required information to process a refund.', WC_PayTrace::TEXT_DOMAIN ) );
			}

			// Init the class
			$pt = new PayTraceAPI( $this->user_name, $this->password );

			// If we have a profile saved we will use it in priority
			if ( '' != $customer_id ) {
				$this->set_refund_with_profile_request_data( $customer_id, $pt );
			} else {
				// If there is no profile, we will use the transaction ID.

				$method_type = 'card' == $payment_type ? 'ProcessTranx' : 'ProcessCheck';
				$this->set_refund_with_transaction_request_data( $transaction_id, $payment_type, $method_type, $pt );
			}

			$pt->SetTERMS( "Y" );
			$pt->SetAMOUNT( $amount );

			if ( $this->testmode == 'yes' ) {
				$pt->SetTEST( 'Y' );
			}

			// Debug
			$this->log_request( $pt, 'Refund Request:' );

			// Process the request which will format the request, send it to the API, and parse the response
			$pt->ProcessRequest();

			$this->log_response( $pt, 'Refund Response:' );

			// There was an error, so mark the order
			if ( $pt->DidErrorOccur() == true ) {
				throw new Exception( sprintf( __( 'Refund not successful. Error Message: %s.', WC_PayTrace::TEXT_DOMAIN ), $pt->GetERROR() ) );
			}

			if ( $pt->was_refund_approved( $payment_type ) ) {
				// Refund was approved
				switch ( (int) $pt->GetRESPONSE() ) {
					case '106' : // Your transaction was successfully refunded.
					case '108' : // Your TEST transaction was successfully refunded HOWEVER, NO FUNDS WILL BE REFUNDED.
					case '122' : // Your check was successfully refunded.
					case '123' : // Your TEST check was successfully refunded. HOWEVER, NO FUNDS WILL BE TRANSFERRED.
						// Debug log
						WC_PayTrace::add_debug_log( 'Refund completed.' );

						// Add order note
						$order->add_order_note(
							sprintf(
								__(
									'Refunded %s. Refund ID: %s. %s',
									WC_PayTrace::TEXT_DOMAIN
								),
								$amount,
								$pt->GetTRANSACTIONID(),
								( '' != $reason ) ? sprintf(
									__(
										'Credit Note: %s.'
									), $reason
								) : ''
							)
						);

						return true;

						break;
					case '107' : // Your transaction was not successfully refunded.
					default :
						// Add order note
						$order->add_order_note( sprintf( __( 'Refund declined. Message: %s.', WC_PayTrace::TEXT_DOMAIN ), $pt->GetRESPONSE() ) );

						return false;

						break;
				}
			} else {
				// Refund was not successful
				throw new Exception( sprintf( __( 'Refund not successful. Message: %s.', WC_PayTrace::TEXT_DOMAIN ), $pt->GetRESPONSE() ) );
			}

		}
		catch ( Exception $ex ) {
			return new WP_Error( 'paytrace-error', $ex->getMessage() );
		}

		return false;
	}

	/**
	 * Get order items description
	 *
	 * @param WC_Order $order
	 * @param bool     $is_subscription
	 *
	 * @return string
	 */
	public function get_order_description( \WC_Order $order, $is_subscription = false ) {
		$desc = '';
		if ( count( $order->get_items() ) > 0 ) {
			foreach ( $order->get_items() as $item ) {
				if ( $item['qty'] ) {

					$item_name = $item['name'];

					$item_meta = WC_Compat_PayTrace::get_order_item_meta( $item );

					if ( $meta = $item_meta->display( true, true ) ) {
						$item_name .= ' (' . $meta . ')';
					}
					$desc .= $item['qty'] . ' x ' . $item_name . ', ';
				}
			}
			// Remove the last two chars and decode any html chars
			$desc = $this->html_entity_decode_numeric( substr( $desc, 0, - 2 ) );
		}

		return $desc;
	}

	/**
	 * Process a single payment request
	 *
	 * @param WC_Order $order
	 * @param null     $amount
	 * @param bool     $is_subscription
	 *
	 * @throws Exception
	 */
	function process_single_payment_request( \WC_Order $order, $amount = null, $is_subscription = false ) {

		// Debug log
		WC_PayTrace::add_debug_log( 'Generating payment request for order #' . $order->id );

		// Init the class
		$pt = new PayTraceAPI( $this->user_name, $this->password );

		$desc = $this->get_order_description( $order, $is_subscription );

		// Set the properties for this request in the class
		$pt->SetTERMS( "Y" );

		// Set the cc or check fields
		if ( $this->what_payment_customer_used == 'check' ) {
			$this->set_request_method_type( 'ProcessCheck', $pt, $this->get_check_transaction_type() );
			$this->set_customer_sensitive_data_to_request( 'check', $pt );
		} else {
			$this->set_request_method_type( 'ProcessTranx', $pt, $this->get_card_transaction_type() );
			$this->set_customer_sensitive_data_to_request( 'card', $pt );
		}

		// Set the Billing address data
		$this->set_order_billing_address_data( $order, $pt );

		// Set the Shipping address data
		$this->set_order_shipping_address_data( $order, $pt );

		// If we have a subs payment the amount should be send as a param
		$order_total = empty( $amount ) ? number_format( $order->get_total(), 2, '.', '' ) : number_format( $amount, 2, '.', '' );
		$pt->SetAMOUNT( $order_total );

		if ( 'yes' == $this->send_order_description ) {
			$pt->SetDESCRIPTION( $desc );
		}

		$pt->SetINVOICE( WC_PayTrace::get_order_number( $order ) );
		if ( $this->testmode == 'yes' ) {
			$pt->SetTEST( 'Y' );
		}

		$this->log_request( $pt, 'Payment request:' );

		// Process the request which will format the request, send it to the API, and parse the response
		$pt->ProcessRequest();

		$this->log_response( $pt );

		// There was an error. Mark order and throw Exception
		if ( $pt->DidErrorOccur() == true ) {
			$this->mark_payment_error( $order, $pt );
		}

		// Transaction processed
		if ( $pt->WasTransactionApproved() == true ) {

			// Mark the approved credit card order
			$this->mark_approved_credit_card_payment( $order, $pt );

		} elseif ( $pt->WasAchTransactionApproved() == true ) {

			$this->mark_approved_echeck_payment( $order, $pt );

		} else {
			// Transaction not approved. Declined responses are missing the APPCODE

			//Add order note that the payment was not approved with all info supplied
			$order->add_order_note(
				sprintf(
					__(
						'PayTrace Payment Declined.
						Transaction Response: %s', WC_PayTrace::TEXT_DOMAIN
					),
					$pt->GetRESPONSE()
				)
			);

			// Change status Failed
			$order->update_status( 'failed' );

			// Debug log
			WC_PayTrace::add_debug_log( 'Payment declined. Message: ' . $pt->GetRESPONSE() );

			// Show declined to the customer
			throw new Exception(
				sprintf(
					__(
						'Transaction declined.'
						. ' Message: %s', WC_PayTrace::TEXT_DOMAIN
					),
					$pt->GetRESPONSE()
				)
			);

		}
	}

	/**
	 * Mark the order after a payment error
	 *
	 * @param WC_Order $order
	 * @param          $response
	 *
	 * @throws Exception
	 */
	public function mark_payment_error( \WC_Order $order, $response ) {
		// Add order note that the payment was not approved with the needed info
		$order->add_order_note(
			sprintf(
				__(
					'Error occurred while processing the payment.'
					. ' Error message: %s', WC_PayTrace::TEXT_DOMAIN
				),
				$response->GetERROR()
			)
		);

		// Debug log
		WC_PayTrace::add_debug_log( 'Error occurred while processing the payment. Error message: ' . $response->GetERROR() );

		// Change status Failed
		$order->update_status( 'failed' );

		// Show error to the customer
		throw new Exception(
			sprintf(
				__(
					'Error occurred while processing the payment.'
					. ' Error message: %s', WC_PayTrace::TEXT_DOMAIN
				),
				$response->GetERROR()
			)
		);
	}

	/**
	 * Mark Card payment as approved
	 *
	 * @param WC_Order $order
	 * @param          $response
	 *
	 * @throws Exception
	 */
	public function mark_approved_credit_card_payment( \WC_Order $order, $response ) {
		// Transaction was approved
		switch ( (int) $response->GetRESPONSE() ) {
			case '101' : // Payment approved
			case '104' : // Payment approved in test mode
				// Add order note with the payment info.
				$order->add_order_note(
					sprintf(
						__(
							'PayTrace Payment Completed.
					Transaction Response: %s,
					Transaction ID: %s,
					Approval Code: %s,
					Approval Message: %s,
					AVS Response: %s,
					CSC Response: %s,
					Card Type: %s.', WC_PayTrace::TEXT_DOMAIN
						),
						$response->GetRESPONSE(),
						$response->GetTRANSACTIONID(),
						$response->GetAPPCODE(),
						$response->GetAPPMSG(),
						$response->GetAVSRESPONSE(),
						$response->GetCSCRESPONSE(),
						$this->cc_type
					)
				);

				// Debug log
				WC_PayTrace::add_debug_log( 'Payment completed.' );

				update_post_meta( $order->id, '_paytrace_transaction_id', WC_Compat_PayTrace::wc_clean( $response->GetTRANSACTIONID() ) );
				update_post_meta( $order->id, '_paytrace_payment_type', 'card' );

				// Process the order
				WC_Compat_PayTrace::payment_complete( $order, $response->GetTRANSACTIONID() );

				break;
			case '103' : // Payment approved, but voided due to AVS and/or CVC mismatch

				// Add order note that the payment was not approved with the needed info
				$order->add_order_note(
					sprintf(
						__(
							'PayTrace payment was approved, but voided because of address and/or CSC mismatch.
					Transaction Response: %s,
					Transaction ID: %s,
					Decline Code: %s,
					Message: %s,
					AVS Response: %s,
					CSC Response: %s.', WC_PayTrace::TEXT_DOMAIN
						),
						$response->GetRESPONSE(),
						$response->GetTRANSACTIONID(),
						$response->GetAPPCODE(),
						$response->GetAPPMSG(),
						$response->GetAVSRESPONSE(),
						$response->GetCSCRESPONSE()
					)
				);

				// Change status Failed
				$order->update_status( 'failed' );

				// Debug log
				WC_PayTrace::add_debug_log( 'Payment declined. Code 103. Message: ' . $response->GetAPPMSG() );

				// Show declined to the customer
				throw new Exception(
					sprintf(
						__(
							'Transaction declined.'
							. ' Message: %s', WC_PayTrace::TEXT_DOMAIN
						), $response->GetAPPMSG()
					)
				);

				break;
			default : // Any other code did not approve the payment

				// Add order note that the payment was not approved with the needed info
				$order->add_order_note(
					sprintf(
						__(
							'PayTrace Payment Declined.
					Transaction Response: %s,
					Transaction ID: %s,
					Decline Code: %s,
					Message: %s,
					AVS Response: %s,
					CSC Response: %s.', WC_PayTrace::TEXT_DOMAIN
						),
						$response->GetRESPONSE(),
						$response->GetTRANSACTIONID(),
						$response->GetAPPCODE(),
						$response->GetAPPMSG(),
						$response->GetAVSRESPONSE(),
						$response->GetCSCRESPONSE()
					)
				);

				// Change status Failed
				$order->update_status( 'failed' );

				// Debug log
				WC_PayTrace::add_debug_log( 'Payment declined. Message: ' . $response->GetAPPMSG() );

				// Show declined to the customer
				throw new Exception(
					sprintf(
						__(
							'Transaction declined.'
							. ' Message: %s', WC_PayTrace::TEXT_DOMAIN
						),
						$response->GetAPPMSG()
					)
				);

				break;
		}
	}

	/**
	 * Mark approved Check payment
	 *
	 * @param WC_Order    $order
	 * @param PayTraceAPI $response
	 *
	 * @throws Exception
	 */
	public function mark_approved_echeck_payment( \WC_Order $order, $response ) {
		switch ( (int) $response->GetRESPONSE() ) :
			case '120' : // Payment approved
			case '121' : // Payment approved in test mode
				// Add order note with the payment info.
				$order->add_order_note(
					sprintf(
						__(
							'PayTrace Check/Ach Payment Completed.
					Transaction Response: %s,
					Check Identifier: %s,
					Approval Code: %s,
					Approval Message: %s
					', WC_PayTrace::TEXT_DOMAIN
						),
						$response->GetRESPONSE(),
						$response->GetCHECKIDENTIFIER(),
						$response->GetACHCODE(),
						$response->GetACHMSG()
					)
				);

				// Debug log
				WC_PayTrace::add_debug_log( 'PayTrace Check/Ach Payment Accepted.' );

				update_post_meta( $order->id, '_paytrace_transaction_id', $response->GetCHECKIDENTIFIER() );
				update_post_meta( $order->id, '_paytrace_payment_type', 'check' );

				// Change status on-hold
				if ( $this->ach_payment_onhold == 'yes' ) {
					$order->update_status( 'on-hold' );
				} else {
					// Process the order
					WC_Compat_PayTrace::payment_complete( $order, $response->GetCHECKIDENTIFIER() );
				}

				break;
			default : // Any other code did not approve the payment

				// Add order note that the payment was not approved with the needed info
				$order->add_order_note(
					sprintf(
						__(
							'PayTrace Check/ACH Payment Declined.
					Transaction Response: %s,
					Check Identifier: %s,
					Approval Code: %s,
					Approval Message: %s
					', WC_PayTrace::TEXT_DOMAIN
						),
						$response->GetRESPONSE(),
						$response->GetCHECKIDENTIFIER(),
						$response->GetACHCODE(),
						$response->GetACHMSG()
					)
				);

				// Change status Failed
				$order->update_status( 'failed' );

				// Debug log
				WC_PayTrace::add_debug_log( 'Payment declined. Message: ' . $response->GetAPPMSG() );

				// Show declined to the customer
				throw new Exception(
					sprintf(
						__(
							'Transaction declined.'
							. ' Message: %s', WC_PayTrace::TEXT_DOMAIN
						),
						$response->GetAPPMSG()
					)
				);

				break;
		endswitch;
	}

	/**
	 * Handle payments with saved cards
	 *
	 * @param WC_Order $order
	 * @param float    $amount          (Optional) Payment amount
	 * @param bool     $is_subscription (Optional) Is the payment for a subscription product
	 *
	 * @throws Exception
	 */
	function process_profile_payment_request( \WC_Order $order, $amount = null, $is_subscription = false ) {

		// Get the saved card
		$custid = $this->get_customer_id_from_order( $order );

		if ( ! $custid ) {
			throw new Exception( __( 'Customer profile not found.', WC_PayTrace::TEXT_DOMAIN ) );
		}

		$saved_profile = $this->get_profile_from_customer_id( $custid );
		$method_type   = ( isset( $saved_profile['last4'] ) ) ? 'ProcessTranx' : 'ProcessCheck';

		// Card payments just run the set transaction type
		if ( 'ProcessTranx' == $method_type ) {
			$transaction_type = $this->get_card_transaction_type();
		} // Check payments transaction type is either 'Hold' or 'Sale'.
		else {
			$transaction_type = $this->get_check_transaction_type();
		}

		// Get Order description
		$desc = $this->get_order_description( $order, $is_subscription );

		// Debug log
		WC_PayTrace::add_debug_log( 'Generating Profile payment request for order #' . $order->id );

		// Init the class
		$pt = new PayTraceAPI( $this->user_name, $this->password );

		// If we have a subs payment the amount should be send as a param
		$order_total = empty( $amount ) ? number_format( $order->get_total(), 2, '.', '' ) : number_format( $amount, 2, '.', '' );

		// Set the properties for this request in the class
		$this->set_request_method_type( $method_type, $pt, $transaction_type );

		// Set the Billing address data
		$this->set_order_billing_address_data( $order, $pt );

		// Set the Shipping address data
		$this->set_order_shipping_address_data( $order, $pt );

		$pt->SetTERMS( "Y" );
		$pt->SetCUSTID( $custid );
		$pt->SetAMOUNT( $order_total );

		// If the customer gave us a CVC number, us it in the request
		if ( ! empty( $this->cc_cvc ) ) {
			$pt->SetCSC( $this->cc_cvc );
		}

		if ( 'yes' == $this->send_order_description ) {
			$pt->SetDESCRIPTION( $desc );
		}

		$pt->SetINVOICE( WC_PayTrace::get_order_number( $order ) );
		if ( $this->testmode == 'yes' ) {
			$pt->SetTEST( 'Y' );
		}

		// Debug
		$this->log_request( $pt, 'Profile Payment Request:' );

		// Process the request which will format the request, send it to the API, and parse the response
		$pt->ProcessRequest();

		$this->log_response( $pt, 'Profile Payment Response:' );

		// There was an error, so mark the order
		if ( $pt->DidErrorOccur() == true ) {

			$this->mark_payment_error( $order, $pt );

		}

		// Transaction processed
		if ( $pt->WasTransactionApproved() == true ) {

			$this->mark_approved_credit_card_payment( $order, $pt );

		} elseif ( $pt->WasAchTransactionApproved() == true ) {

			$this->mark_approved_echeck_payment( $order, $pt );

		} else {
			// Transaction not approved. Declined responses are missing the APPCODE

			// Add order note that the payment was not approved with all info supplied
			$order->add_order_note(
				sprintf(
					__(
						'PayTrace Payment Declined.
				Transaction Response: %s,
				Transaction ID: %s,
				Message: %s,
				AVS Response: %s,
				CSC Response: %s.', WC_PayTrace::TEXT_DOMAIN
					),
					$pt->GetRESPONSE(),
					$pt->GetTRANSACTIONID(),
					$pt->GetAPPMSG(),
					$pt->GetAVSRESPONSE(),
					$pt->GetCSCRESPONSE()
				)
			);

			// Change status Failed
			$order->update_status( 'failed' );

			// Debug log
			WC_PayTrace::add_debug_log( 'Payment declined. Message: ' . $pt->GetRESPONSE() );

			// Show declined to the customer
			throw new Exception( sprintf( __( 'Transaction declined. Message: %s', WC_PayTrace::TEXT_DOMAIN ), $pt->GetRESPONSE() ) );
		}
	}

	/**
	 * Create a customer profile on the PayTrace system.
	 *
	 * @param WC_Order $order
	 * @param string   $payment_type Type of payment we will run, 'card' or 'check'
	 *
	 * @return string The created Customer ID
	 * @throws Exception
	 */
	function create_customer_profile( \WC_Order $order, $payment_type = 'card' ) {

		// Init the class
		$pt = new PayTraceAPI( $this->user_name, $this->password );

		// Set the properties for this request in the class
		$pt->SetTERMS( "Y" );
		$pt->SetCUSTID( uniqid() . '-' . get_current_user_id() );

		// Set the request method
		$this->set_request_method_type( "CreateCustomer", $pt );

		// Set the Billing address data
		$this->set_order_billing_address_data( $order, $pt );

		// Set the Shipping address data
		$this->set_order_shipping_address_data( $order, $pt );

		// Set the customer sensitive data (card,exp,cvv or routing/acc number)
		$this->set_customer_sensitive_data_to_request( $payment_type, $pt );

		// Debug
		$this->log_request( $pt, 'Create Customer Profile Request:' );

		// Run the Request
		$pt->ProcessRequest();

		$this->log_response( $pt, 'Create Customer Profile Response:' );

		if ( $pt->DidErrorOccur() == true ) {
			// Add order note that the payment was not approved with the needed info
			$order->add_order_note(
				sprintf(
					__(
						'Error occurred while creating a customer payment profile.'
						. ' Error message: %s', WC_PayTrace::TEXT_DOMAIN
					),
					$pt->GetERROR()
				)
			);

			// Debug log
			WC_PayTrace::add_debug_log( 'Error occurred while creating a customer payment profile. Error message: ' . $pt->GetERROR() );

			// Show error to the customer
			throw new Exception(
				sprintf(
					__(
						'Error occurred while creating your payment profile.'
						. ' Error message: %s', WC_PayTrace::TEXT_DOMAIN
					),
					$pt->GetERROR()
				)
			);
		}

		if ( $pt->WasCustomerRequestApproved() == true ) {
			switch ( (int) $pt->GetRESPONSE() ) :
				case '160' : // Profile created
					// Add an order note.
					$order->add_order_note( __( 'Customer payment profile created.', WC_PayTrace::TEXT_DOMAIN ) );

					// Profile was created, save the info to the customer user data
					if ( is_user_logged_in() ) {

						if ( 'check' == $payment_type ) {
							$profile = array(
								'profile_id'  => $pt->GetCUSTOMERID(),
								'customer_id' => $pt->GetCUSTID(),
								'tr_last4'    => substr( $this->ach_routing, - 4, 4 ),
								'dda_last4'   => substr( $this->ach_account, - 4, 4 ),
							);
						} else {
							$profile = array(
								'profile_id'  => $pt->GetCUSTOMERID(),
								'customer_id' => $pt->GetCUSTID(),
								'last4'       => substr( $this->cc_number, - 4, 4 ),
								'exp_year'    => $this->cc_exp_year,
								'exp_month'   => $this->cc_exp_month,
							);
						}

						add_user_meta( get_current_user_id(), '_paytrace_saved_profiles', $profile );
					}

					$this->save_used_customer_id_to_order_meta( $order, $pt->GetCUSTID() );

					return $pt->GetCUSTID();
					break;
				default : // Any other code did not create customer

					// Show error to the customer
					throw new Exception(
						sprintf(
							__(
								'Creating your payment profile failed.'
								. ' Error message: %s', WC_PayTrace::TEXT_DOMAIN
							),
							$pt->GetRESPONSE()
						)
					);

					break;
			endswitch;
		} else {
			// Show error to the customer
			throw new Exception(
				sprintf(
					__(
						'Creating your payment profile failed.'
						. ' Error message: %s', WC_PayTrace::TEXT_DOMAIN
					),
					$pt->GetRESPONSE()
				)
			);
		}
	}

	/**
	 * Set the payment request sensitive data to the request object.
	 *
	 * @since 1.3
	 *
	 * @param string      $payment_type
	 * @param PayTraceAPI $pt
	 */
	function set_customer_sensitive_data_to_request( $payment_type, &$pt ) {

		if ( 'check' == $payment_type ) {
			$pt->SetDDA( $this->ach_account );
			$pt->SetTR( $this->ach_routing );
		} else {
			$pt->SetCC( $this->cc_number );
			$pt->SetEXPMNTH( $this->cc_exp_month );
			$pt->SetEXPYR( $this->cc_exp_year );
			$pt->SetCSC( $this->cc_cvc );
		}

	}

	/**
	 * Set the billing address to the PayTrace Request object
	 *
	 * @param \WC_Order   $order
	 * @param PayTraceAPI $pt
	 */
	public function set_order_billing_address_data( \WC_Order $order, &$pt ) {
		$pt->SetBNAME( $order->billing_first_name . ' ' . $order->billing_last_name );
		$pt->SetBADDRESS( $order->billing_address_1 );
		$pt->SetBZIP( $order->billing_postcode );
		$pt->SetBADDRESS2( $order->billing_address_2 );
		$pt->SetBCITY( $order->billing_city );
		// State cannot be more than two characters,
		// so just dont send it if it is longer than two characters
		$pt->SetBSTATE( ( strlen( $order->billing_state ) > 2 ) ? '' : $order->billing_state );
		$pt->SetBCOUNTRY( $order->billing_country );
		$pt->SetEMAIL( $order->billing_email );
	}

	/**
	 * Set the shipping address to the PayTrace Request object
	 *
	 * @param WC_Order    $order WC Order object
	 * @param PayTraceAPI $pt    Paytrace API object
	 */
	private function set_order_shipping_address_data( \WC_Order $order, &$pt ) {
		if ( '' != $order->shipping_address_1 ) {
			$pt->SetSNAME( $order->shipping_first_name . ' ' . $order->shipping_last_name );
			$pt->SetSADDRESS( $order->shipping_address_1 );
			$pt->SetSZIP( $order->shipping_postcode );
			$pt->SetSADDRESS2( $order->shipping_address_2 );
			$pt->SetSCITY( $order->shipping_city );
			// State cannot be more than two characters,
			// so just don't send it if it is longer than two characters
			$pt->SetSSTATE( ( strlen( $order->shipping_state ) > 2 ) ? '' : $order->shipping_state );
			$pt->SetSCOUNTRY( $order->shipping_country );
		}
	}

	/**
	 * Update a customer profile from both PayTrace and store systems.
	 *
	 * @param $update_card_id
	 *
	 * @throws Exception
	 */
	function update_customer_profile( $update_card_id ) {

		$saved_profiles    = $this->get_customer_saved_profiles();
		$profile_to_update = $saved_profiles[ $update_card_id ];

		if ( null != WC_PayTrace::get_post( 'paytrace_update_check' ) ) {
			$update_type = 'check';

			WC_PayTrace::add_debug_log( 'Update type: ' . $update_type );
			$name_on_check = WC_PayTrace::get_post( 'paytrace_check_name' );
			$routing       = $this->remove_spaces( WC_PayTrace::get_post( 'paytrace_routing_number' ) );
			$account       = $this->remove_spaces( WC_PayTrace::get_post( 'paytrace_account_number' ) );

			$this->validate_ach_fields( $routing, $account );
		} else {
			$update_type = 'card';
			WC_PayTrace::add_debug_log( 'Update type: ' . $update_type );
			$cc_number    = $this->remove_spaces( WC_PayTrace::get_post( 'paytrace_cc_number' ) );
			$cc_exp_month = WC_PayTrace::get_post( 'paytrace_cc_exp_month' );
			$cc_exp_year  = WC_PayTrace::get_post( 'paytrace_cc_exp_year' );
			$card_type    = WC_PayTrace::get_post( 'paytrace_cc_type' );
			$cc_name      = WC_PayTrace::get_post( 'paytrace_cc_name' );

			$this->validate_cc_number( $cc_number, $card_type );
		}

		// Init the class
		$pt = new PayTraceAPI( $this->user_name, $this->password );

		// Set the properties for this request in the class
		$pt->SetTERMS( "Y" );
		$this->set_request_method_type( 'UpdateCustomer', $pt );
		$pt->SetCUSTID( $profile_to_update['customer_id'] );

		if ( 'check' == $update_type ) {
			$pt->SetBNAME( $name_on_check );
			$pt->SetTR( $routing );
			$pt->SetDDA( $account );
		} else {
			$pt->SetBNAME( $cc_name );
			$pt->SetCC( $cc_number );
			$pt->SetEXPMNTH( $cc_exp_month );
			$pt->SetEXPYR( $cc_exp_year );
		}

		// Debug
		$this->log_request( $pt, 'Update Customer Profile Request:' );

		$pt->ProcessRequest();

		$this->log_response( $pt, 'Update Customer Profile Response:' );

		if ( $pt->DidErrorOccur() == true ) {

			// Debug log
			WC_PayTrace::add_debug_log( 'Error occurred while updating a customer payment profile. Error message: ' . $pt->GetERROR() );

			// Show error to the customer
			throw new Exception(
				sprintf(
					__(
						'Error occurred while updating your payment profile.'
						. ' Error message: %s', WC_PayTrace::TEXT_DOMAIN
					),
					$pt->GetERROR()
				)
			);
		}

		if ( $pt->WasCustomerRequestApproved() == true ) {
			switch ( (int) $pt->GetRESPONSE() ) {
				case '161' : // Profile updated

					if ( 'check' == $update_type ) {
						$new_profile = array(
							'profile_id'  => $pt->GetCUSTOMERID(),
							'customer_id' => $pt->GetCUSTID(),
							'tr_last4'    => substr( $routing, - 4, 4 ),
							'dda_last4'   => substr( $account, - 4, 4 ),
						);
					} else {
						$new_profile = array(
							'profile_id'  => $pt->GetCUSTOMERID(),
							'customer_id' => $pt->GetCUSTID(),
							'last4'       => substr( $cc_number, - 4, 4 ),
							'exp_year'    => $cc_exp_year,
							'exp_month'   => $cc_exp_month,
						);
					}

					update_user_meta( get_current_user_id(), '_paytrace_saved_profiles', WC_Compat_PayTrace::wc_clean( $new_profile ), $profile_to_update );

					break;
				default : // Any other code did not update customer

					// Show error to the customer
					throw new Exception(
						sprintf(
							__(
								'Updating your payment profile failed.'
								. ' Error message: %s', WC_PayTrace::TEXT_DOMAIN
							),
							$pt->GetRESPONSE()
						)
					);
			}
		} else {
			// Show error to the customer
			throw new Exception(
				sprintf(
					__(
						'Updating your payment profile failed.'
						. ' Error message: %s', WC_PayTrace::TEXT_DOMAIN
					),
					$pt->GetRESPONSE()
				)
			);
		}
	}

	/**
	 * Delete a customer profile from both PayTrace and store systems.
	 *
	 * @param string $delete_card_id The ID of the save, in the database, profile
	 *
	 * @throws Exception
	 */
	function delete_customer_profile( $delete_card_id ) {

		$saved_profiles = $this->get_customer_saved_profiles();
		if ( ! empty( $saved_profiles ) ) {
			$profile_to_delete = $saved_profiles[ $delete_card_id ];
		} else {
			throw new Exception( sprintf( __( 'No cards to delete.', WC_PayTrace::TEXT_DOMAIN ) ) );
		}

		// Init the class
		$pt = new PayTraceAPI( $this->user_name, $this->password );

		// Set the properties for this request in the class
		$pt->SetTERMS( "Y" );
		$this->set_request_method_type( 'DeleteCustomer', $pt );
		$pt->SetCUSTID( $profile_to_delete['customer_id'] );

		// Debug
		$this->log_request( $pt, 'Delete Customer Profile Request:' );

		$pt->ProcessRequest();

		$this->log_response( $pt, 'Delete Customer Profile Response:' );

		if ( $pt->DidErrorOccur() == true ) {
			// Debug log
			WC_PayTrace::add_debug_log( 'Error occurred while deleting a customer payment profile. Error message: ' . $pt->GetERROR() );

			switch ( (int) $pt->GetERROR() ) {
				case '30' : // Customer ID, xxxxx, was not found or is incomplete.
					// Even though it is an error, we know from the message that this customer
					// no longer exists in the PayTrace system, so we will delete it from ours, too
					delete_user_meta( get_current_user_id(), '_paytrace_saved_profiles', $profile_to_delete );

					return;

					break;
			}

			// Show error to the customer
			throw new Exception(
				sprintf(
					__(
						'Error occurred while deleting your payment profile.'
						. ' Error message: %s', WC_PayTrace::TEXT_DOMAIN
					),
					$pt->GetERROR()
				)
			);
		}

		if ( $pt->WasCustomerRequestApproved() == true ) {
			switch ( (int) $pt->GetRESPONSE() ) {
				case '162' : // The customer profile for customer ID/customer Name was successfully deleted.

					delete_user_meta( get_current_user_id(), '_paytrace_saved_profiles', $profile_to_delete );

					break;
				default : // Any other code did not delete customer

					// Show error to the customer
					throw new Exception(
						sprintf(
							__(
								'Deleting your payment profile failed.'
								. ' Error message: %s', WC_PayTrace::TEXT_DOMAIN
							),
							$pt->GetRESPONSE()
						)
					);

					break;
			}
		} else {
			// Show error to the customer
			throw new Exception(
				sprintf(
					__(
						'Deleting your payment profile failed.'
						. ' Error message: %s', WC_PayTrace::TEXT_DOMAIN
					),
					$pt->GetRESPONSE()
				)
			);
		}
	}

	/**
	 * Message for the receipt page
	 *
	 * @param $order
	 */
	function receipt_page( $order ) {

		echo '<p>' . __( 'Thank you for your order.', WC_PayTrace::TEXT_DOMAIN ) . '</p>';

	}

	/**
	 * Validate payment fields
	 *
	 * @return bool
	 */
	function validate_fields() {
		try {
			// Check payment info before it is send to the gateway
			$this->validate_payment_information();

			// Note the credit card fields check was passed
			$this->check_pass = true;

			return true;
		}
		catch ( Exception $e ) {
			WC_Compat_PayTrace::wc_add_notice( $e->getMessage(), 'error' );

			return false;
		}
	}

	/**
	 * Validate the received payment information,
	 * before it is passed to the payment gateway.
	 *
	 * @return void
	 */
	function validate_payment_information() {
		// If Check is chosen
		if ( WC_PayTrace::get_post( 'paytrace_type_choice' ) == '2' ) {
			$this->set_ach_details();

			if ( 'yes' == $this->save_customers
				&& ( WC_PayTrace::get_post( 'paytrace-used-check' ) !== 'new' && is_user_logged_in() )
			) {
				$this->what_payment_customer_used = 'saved_check';
			} else {
				// Check and validate the posted card fields
				$this->validate_ach_fields( $this->ach_routing, $this->ach_account );
				$this->what_payment_customer_used = 'check';
			}
		} else {
			if ( 'yes' == $this->save_customers
				&& ( WC_PayTrace::get_post( 'paytrace-used-cc' ) !== 'new' && is_user_logged_in() )
			) {
				$this->what_payment_customer_used = 'saved_cc';
			} else {
				// Set the card details
				$this->set_credit_card_details();

				// Check and validate the posted card fields
				$this->validate_card_fields(
					$this->cc_number,
					$this->cc_cvc,
					$this->cc_exp_month,
					$this->cc_exp_year,
					$this->cc_type
				);
				$this->what_payment_customer_used = 'new_cc';

				WC_PayTrace::add_debug_log( 'We have a new card option' );
			}
		}
	}

	/**
	 * Check if this gateway is enabled and available in the user's country
	 */
	function is_available() {
		if ( $this->enabled == 'yes' ) {
			// Check if the required fields are set
			if ( ! $this->user_name ) {
				return false;
			}
			if ( ! $this->password ) {
				return false;
			}

			return true;
		}

		return false;
	}

	/**
	 * Set credit card details into object variable
	 */
	public function set_credit_card_details() {
		if ( WC_Compat_PayTrace::is_wc_2_1() ) {
			$cc_exp_date        = $this->format_exp_date( $this->remove_spaces( WC_PayTrace::get_post( 'paytrace-card-expiry' ) ) );
			$this->cc_exp_month = $cc_exp_date['month'];
			$this->cc_exp_year  = $cc_exp_date['year'];
		} else {
			$this->cc_exp_month = $this->remove_spaces( WC_PayTrace::get_post( 'paytrace-card-exp-month' ) );
			$this->cc_exp_year  = $this->remove_spaces( WC_PayTrace::get_post( 'paytrace-card-exp-year' ) );
		}

		$this->cc_type   = $this->remove_spaces( WC_PayTrace::get_post( 'paytrace-card-type' ) );
		$this->cc_number = $this->remove_spaces( WC_PayTrace::get_post( 'paytrace-card-number' ) );
		$this->cc_cvc    = $this->remove_spaces( WC_PayTrace::get_post( 'paytrace-card-cvc' ) );
	}

	/**
	 * Check and validate the card fields
	 *
	 * @param null $cc_number    Credit card number
	 * @param null $cc_cvc       Card Security Code
	 * @param null $cc_exp_month Expiration month
	 * @param null $cc_exp_year  Expiration year
	 * @param null $cc_type      Card Type
	 *
	 * @throws Exception
	 */
	public function validate_card_fields( $cc_number = null, $cc_cvc = null, $cc_exp_month = null, $cc_exp_year = null, $cc_type = null ) {

		// Check the cc type: present and available
		if ( empty( $cc_type ) || ! in_array( $cc_type, $this->available_cc ) ) {
			throw new Exception( __( 'Credit Card Type Invalid.', WC_PayTrace::TEXT_DOMAIN ) );
		}

		// Check credit card number: present, numeric, lenght, match the cc type, pass luhn's check
		// Checks done to ensure the customer will submit correct cc information
		if ( empty( $cc_number ) ) {
			throw new Exception( __( 'Credit Card Number Required.', WC_PayTrace::TEXT_DOMAIN ) );
		} else {
			if ( ! is_numeric( $cc_number ) || ( strlen( $cc_number ) > 19 || strlen( $cc_number ) < 12 ) ) {
				throw new Exception( __( 'Invalid Credit Card Number.', WC_PayTrace::TEXT_DOMAIN ) );
			}

			$this->validate_cc_number( $cc_number, $cc_type );

			if ( ! $this->luhn_check( $cc_number ) ) {
				throw new Exception( __( 'Invalid credit card number.', WC_PayTrace::TEXT_DOMAIN ) );
			}

		}

		// Check credit card CV2 number
		if ( empty( $cc_cvc ) ) {
			throw new Exception( __( 'CVC number required.', WC_PayTrace::TEXT_DOMAIN ) );
		} else {
			// Should be numbers only
			if ( ! is_numeric( $cc_cvc ) ) {
				throw new Exception( __( 'Invalid Card Verification Code. Only numbers are allowed.', WC_PayTrace::TEXT_DOMAIN ) );
			}
			// Should be 4 digit for AMEX and 3 for all other types
			if ( ( $cc_type == 'AMEX' && strlen( $cc_cvc ) != 4 ) ||
				( in_array( $cc_type, array( 'VISA', 'MC', 'DISCOVER', 'JCB' ) ) && strlen( $cc_cvc ) != 3 )
			) {
				throw new Exception( __( 'Invalid Card Verification Code length.', WC_PayTrace::TEXT_DOMAIN ) );
			}
		}

		if ( empty( $cc_exp_month ) ) {
			throw new Exception( __( 'Expiration month required.', WC_PayTrace::TEXT_DOMAIN ) );
		}

		if ( empty( $cc_exp_year ) ) {
			throw new Exception( __( 'Expiration year required.', WC_PayTrace::TEXT_DOMAIN ) );
		}

		// Check exp date
		$current_year  = date( 'y' );
		$current_month = date( 'n' );
		if ( ! is_numeric( $cc_exp_year ) ||
			$cc_exp_year < $current_year ||
			$cc_exp_year > ( $current_year + 10 ) ||
			! is_numeric( $cc_exp_month ) ||
			$cc_exp_month < 0 ||
			$cc_exp_month > 12 ||
			( $cc_exp_year == $current_year && $cc_exp_month < $current_month )
		) {
			throw new Exception( __( 'Invalid expiration date.', WC_PayTrace::TEXT_DOMAIN ) );
		}
	}

	/**
	 * Validate the card number against its type and luhn's check.
	 *
	 * @param int    $cc_number The card number, checked to be only a numbers
	 * @param string $cc_type   The card type
	 *
	 * @return void
	 * @throws Exception
	 */
	private function validate_cc_number( $cc_number, $cc_type = '' ) {

		// Match the cc number with the type
		switch ( $cc_type ) :
			case 'VISA' :
				if ( ! preg_match( '/^4[0-9]{12}(?:[0-9]{3})?$/', $cc_number ) ) {
					throw new Exception( __( 'Invalid Visa card number.', WC_PayTrace::TEXT_DOMAIN ) );
				}
				break;
			case 'MC' :
				if ( ! preg_match( '/^5[1-5][0-9]{14}$/', $cc_number ) ) {
					throw new Exception( __( 'Invalid Master Card number.', WC_PayTrace::TEXT_DOMAIN ) );
				}
				break;
			case 'AMEX' :
				if ( ! preg_match( '/^3[47][0-9]{13}$/', $cc_number ) ) {
					throw new Exception( __( 'Invalid American Express card number.', WC_PayTrace::TEXT_DOMAIN ) );
				}
				break;
			case 'DISCOVER' :
				if ( ! preg_match( '/^6(?:011|5[0-9]{2})[0-9]{12}$/', $cc_number ) ) {
					throw new Exception( __( 'Invalid Discover card number.', WC_PayTrace::TEXT_DOMAIN ) );
				}
				break;
			case 'JCB' :
				if ( ! preg_match( '/^(?:2131|1800|35\d{3})\d{11}$/', $cc_number ) ) {
					throw new Exception( __( 'Invalid JCB card number.', WC_PayTrace::TEXT_DOMAIN ) );
				}
				break;
			default :
				if ( ! preg_match( '/^(?:4[0-9]{12}(?:[0-9]{3})?|5[1-5][0-9]{14}|6(?:011|5[0-9][0-9])[0-9]{12}|3[47][0-9]{13}|3(?:0[0-5]|[68][0-9])[0-9]{11}|(?:2131|1800|35\d{3})\d{11})$/', $cc_number ) ) {
					throw new Exception( __( 'Invalid Card Number', WC_PayTrace::TEXT_DOMAIN ) );
				}
				break;
		endswitch;

	}

	/**
	 * Perform a luhn check on the card number<br/>
	 * This will not ensure the card is legitimate, <br/>
	 * it will only ensure the number passes checksum.
	 *
	 * @param int $cc_number The credit card number
	 *
	 * @return void
	 */
	private function luhn_check( $cc_number ) {
		// Reverse the number to start from the back
		$cc_number = strrev( $cc_number );
		$sum       = 0;

		for ( $i = 0; $i < strlen( $cc_number ); $i ++ ) {
			$n = substr( $cc_number, $i, 1 );
			// Double every second digit
			$n = ( $i % 2 ) ? $n * 2 : $n;
			// Sum the digits
			$sum += $n > 9 ? $n - 9 : $n;
		}

		return ( $sum % 10 == 0 );
	}

	/**
	 * Validate the Check fields
	 *
	 * @param mixed $ach_routing
	 * @param mixed $ach_account
	 *
	 * @throws Exception
	 */
	private function validate_ach_fields( $ach_routing, $ach_account ) {
		if ( empty( $ach_routing ) ) {
			throw new Exception( __( 'Check Routing Number Required.', WC_PayTrace::TEXT_DOMAIN ) );
		} else {
			if ( ! is_numeric( $ach_routing ) ) {
				throw new Exception( __( 'Invalid Check Routing Number.', WC_PayTrace::TEXT_DOMAIN ) );
			}
		}

		if ( empty( $ach_account ) ) {
			throw new Exception( __( 'Check Account Number Required.', WC_PayTrace::TEXT_DOMAIN ) );
		} else {
			if ( ! is_numeric( $ach_account ) ) {
				throw new Exception( __( 'Invalid Check Account Number.', WC_PayTrace::TEXT_DOMAIN ) );
			}
		}
	}

	/**
	 * Decodes all HTML entities, including numeric and hexadecimal ones.
	 *
	 * @param        $string
	 * @param int    $quote_style
	 * @param string $charset
	 *
	 * @return mixed|string
	 */
	function html_entity_decode_numeric( $string, $quote_style = ENT_COMPAT, $charset = 'utf-8' ) {
		$string = html_entity_decode( $string, $quote_style, $charset );
		$string = preg_replace_callback( '~&#x([0-9a-fA-F]+);~i', array( $this, 'chr_utf8_callback' ), $string );
		$string = preg_replace_callback( '~&#([0-9]+);~', array( $this, 'chr_utf8' ), $string );
		$string = str_ireplace( '&apos;', "'", $string );

		return $string;
	}

	/**
	 * Callback helper
	 */
	function chr_utf8_callback( $matches ) {
		return chr_utf8( hexdec( $matches[1] ) );
	}

	/**
	 * Multi-byte chr(): Will turn a numeric argument into a UTF-8 string.
	 *
	 * @param array $num
	 *
	 * @return string
	 */
	function chr_utf8( $num ) {
		if ( $num[1] < 128 ) {
			return chr( $num[1] );
		}
		if ( $num[1] < 2048 ) {
			return chr( ( $num[1] >> 6 ) + 192 ) . chr( ( $num[1] & 63 ) + 128 );
		}
		if ( $num[1] < 65536 ) {
			return chr( ( $num[1] >> 12 ) + 224 ) . chr( ( ( $num[1] >> 6 ) & 63 ) + 128 ) . chr( ( $num[1] & 63 ) + 128 );
		}
		if ( $num[1] < 2097152 ) {
			return chr( ( $num[1] >> 18 ) + 240 ) . chr( ( ( $num[1] >> 12 ) & 63 ) + 128 ) . chr( ( ( $num[1] >> 6 ) & 63 ) + 128 ) . chr( ( $num[1] & 63 ) + 128 );
		}

		return '';
	}

	/**
	 * Remove spaces
	 *
	 * @param string $string Input string
	 *
	 * @return string The string with removed spaces
	 */
	private function remove_spaces( $string ) {
		return str_replace( array( ' ', '-' ), '', $string );
	}

	/**
	 * Format and return the expiration month and year
	 *
	 * @param $expiration_date
	 *
	 * @return array
	 */
	public function format_exp_date( $expiration_date ) {
		$cc_exp_date = str_replace( ' ', '', explode( '/', $expiration_date ) );

		return array( 'month' => $cc_exp_date[0], 'year' => $cc_exp_date[1] );
	}

	/**
	 * Loop through the user saved cards and find the one matching the account_id
	 *
	 * @param $customer_id
	 *
	 * @return bool
	 */
	public function get_profile_from_customer_id( $customer_id ) {
		$saved_cards = $this->get_customer_saved_profiles();

		foreach ( $saved_cards as $key => $value ) {
			if ( $value['customer_id'] == $customer_id ) {
				return $saved_cards[ $key ];
			}
		}

		return false;
	}

	/**
	 * Return the profile key from the customer profiles based on the customer ID.
	 *
	 * @param $customer_id
	 *
	 * @return string|bool
	 */
	private function get_profile_id_from_customer_id( $customer_id ) {
		$saved_cards = $this->get_customer_saved_profiles();
		foreach ( $saved_cards as $key => $value ) {
			if ( $value['customer_id'] == $customer_id ) {
				return $key;
			}
		}

		return false;
	}

	/**
	 * Validate Password Field.
	 *
	 * Make sure the data is escaped correctly, etc.
	 * We are not showing the password value to the front end,
	 * so we will overwrite the password validation, so we can update the password only when it is not empty.
	 * If left empty the password will be saved with the old value.
	 *
	 * @since 1.2
	 *
	 * @param mixed $key
	 *
	 * @return string
	 */
	public function validate_password_field( $key ) {
		$text = $this->get_option( $key );

		if ( isset( $_POST[ $this->plugin_id . $this->id . '_' . $key ] ) && '' != $_POST[ $this->plugin_id . $this->id . '_' . $key ] ) {
			$text = WC_Compat_PayTrace::wc_clean( stripslashes( $_POST[ $this->plugin_id . $this->id . '_' . $key ] ) );
		}

		return $text;
	}

	/**
	 * Generate Password Input HTML.
	 * Overwrite here so it is accessible for WC 2.0
	 *
	 * @since 1.2.1
	 *
	 * @param mixed $key
	 * @param mixed $data
	 *
	 * @return string
	 */
	public function generate_password_html( $key, $data ) {
		$data['type'] = 'password';

		return $this->generate_text_html( $key, $data );
	}

	/**
	 * Generate Text Input HTML.
	 * Modify the text html to remove the password value from the front end.
	 *
	 * @since 1.2
	 *
	 * @param mixed $key
	 * @param mixed $data
	 *
	 * @return string
	 */
	public function generate_text_html( $key, $data ) {
		$field    = $this->plugin_id . $this->id . '_' . $key;
		$defaults = array(
			'title'             => '',
			'disabled'          => false,
			'class'             => '',
			'css'               => '',
			'placeholder'       => '',
			'type'              => 'text',
			'desc_tip'          => false,
			'description'       => '',
			'custom_attributes' => array()
		);

		$data = wp_parse_args( $data, $defaults );

		If ( 'password' == $data['type'] ) {
			$value = '';
		} else {
			$value = esc_attr( $this->get_option( $key ) );
		}

		ob_start();
		?>
		<tr valign="top">
			<th scope="row" class="titledesc">
				<label for="<?php echo esc_attr( $field ); ?>"><?php echo wp_kses_post( $data['title'] ); ?></label>
				<?php echo $this->get_tooltip_html( $data ); ?>
			</th>
			<td class="forminp">
				<fieldset>
					<legend class="screen-reader-text">
						<span><?php echo wp_kses_post( $data['title'] ); ?></span></legend>
					<input class="input-text regular-input <?php echo esc_attr( $data['class'] ); ?>"
					       type="<?php echo esc_attr( $data['type'] ); ?>"
					       name="<?php echo esc_attr( $field ); ?>"
					       id="<?php echo esc_attr( $field ); ?>"
					       style="<?php echo esc_attr( $data['css'] ); ?>"
					       value="<?php echo $value; ?>"
					       placeholder="<?php echo esc_attr( $data['placeholder'] ); ?>"
						<?php disabled( $data['disabled'], true ); ?> <?php echo $this->get_custom_attribute_html( $data ); ?> />
					<?php echo $this->get_description_html( $data ); ?>
				</fieldset>
			</td>
		</tr>
		<?php
		return ob_get_clean();
	}

	/**
	 * Get HTML for tooltips
	 * Overwrite here so it is accessible for WC 2.0
	 *
	 * @param array $data
	 *
	 * @return string
	 */
	public function get_tooltip_html( $data ) {
		if ( $data['desc_tip'] === true ) {
			$tip = $data['description'];
		} elseif ( ! empty( $data['desc_tip'] ) ) {
			$tip = $data['desc_tip'];
		} else {
			$tip = '';
		}

		return $tip ? '<img class="help_tip" data-tip="' . esc_attr( $tip ) . '" src="' . WC_Compat_PayTrace::get_wc_global()->plugin_url() . '/assets/images/help.png" height="16" width="16" />' : '';
	}

	/**
	 * Get HTML for descriptions
	 * Overwrite here so it is accessible for WC 2.0
	 *
	 * @param array $data
	 *
	 * @return string
	 */
	public function get_description_html( $data ) {
		if ( $data['desc_tip'] === true ) {
			$description = '';
		} elseif ( ! empty( $data['desc_tip'] ) ) {
			$description = $data['description'];
		} elseif ( ! empty( $data['description'] ) ) {
			$description = $data['description'];
		} else {
			$description = '';
		}

		return $description ? '<p class="description">' . wp_kses_post( $description ) . '</p>' . "\n" : '';
	}

	/**
	 * Get custom attributes
	 * Overwrite here so it is accessible for WC 2.0
	 *
	 * @param array $data
	 *
	 * @return string
	 */
	public function get_custom_attribute_html( $data ) {
		$custom_attributes = array();

		if ( ! empty( $data['custom_attributes'] ) && is_array( $data['custom_attributes'] ) ) {
			foreach ( $data['custom_attributes'] as $attribute => $attribute_value ) {
				$custom_attributes[] = esc_attr( $attribute ) . '="' . esc_attr( $attribute_value ) . '"';
			}
		}

		return implode( ' ', $custom_attributes );
	}

	/**
	 * Generate and log the response for Debug purposes
	 *
	 * @since 1.3
	 *
	 * @param PayTraceAPI $pt
	 * @param string      $message
	 */
	public function log_response( $pt, $message = 'Received Response:' ) {
		// Debug log
		WC_PayTrace::add_debug_log( $message . ' ' . print_r( $pt->GetDEBUG_RESPONSE(), true ) );
	}

	/**
	 * Generate and log the request for Debug purposes
	 *
	 * @param PayTraceAPI $pt
	 * @param string      $message
	 */
	public function log_request( $pt, $message = 'Generated request:' ) {
		// Debug log
		WC_PayTrace::add_debug_log( $message . ' ' . print_r( $pt->GetDEBUG_REQUEST(), true ) );
	}

	/**
	 * Set the ACH routing and account number
	 *
	 * @since 1.3
	 */
	public function set_ach_details() {
		// Set the ACH sensitive info
		$this->ach_routing = $this->remove_spaces( WC_PayTrace::get_post( 'paytrace_check_routing_number' ) );
		$this->ach_account = $this->remove_spaces( WC_PayTrace::get_post( 'paytrace_check_account_number' ) );
	}

	/**
	 * Process ACH Payment. <br/>
	 * Process flow: <br/>
	 * I. Saving Checks AND User is logged in
	 *        1. Using a New Check
	 *                1.a. Saving the Check
	 *                        1.a.1. Create a Customer Profile using the Check details <br/>
	 *                        1.a.2. Charge the CF for the order<br/>
	 *                                1.a.2.a. If Charge is not successful, remove the created CF <br/>
	 *        2. Using Saved Check
	 *                2.a. Prepare Customer Profile for payment
	 *                1.b. Charge the used Customer Profile
	 * II. Not Saving Cards OR User not logged in
	 *        1. Run a single payment transaction
	 *
	 * @since 1.3
	 *
	 * @param WC_Order   $order
	 * @param null|float $initial_payment Total amount to be charged
	 * @param bool       $is_subscription Is the payment we are processing for a Subscription
	 *
	 * @throws Exception
	 */
	public function process_ach_payment( \WC_Order $order, $initial_payment = null, $is_subscription = false ) {
		// Set the ACH details
		$this->set_ach_details();
		$should_we_charge_the_profile = ( $is_subscription && $initial_payment > 0 ) || ! $is_subscription;

		if ( 'yes' == $this->save_customers && is_user_logged_in() ) {

			// Customer used new ACH/Check
			if ( 'check' == $this->what_payment_customer_used ) {

				// If customer chose to save the Check or Subscription payment(always saving the Check)
				if ( WC_PayTrace::get_post( 'paytrace-save-card' ) == '1' || $is_subscription ) {
					// Create profile first
					$customer_id = $this->create_customer_profile( $order, 'check' );

					try {
						// If there is an initial payment
						if ( $should_we_charge_the_profile ) {
							// Process a profile payment on a saved check
							$this->process_profile_payment_request( $order, $initial_payment, $is_subscription );
						}
					}
					catch ( Exception $e ) {
						try {
							$profile_id_to_delete = $this->get_profile_id_from_customer_id( $customer_id );

							// If we could not charge the new profile, then we are to delete it
							$this->delete_customer_profile( $profile_id_to_delete );
						}
						catch ( Exception $delete_error ) {
							// Throw an error with both error messages, from failed transaction and
							throw new Exception( $e->getMessage() . ' ' . $delete_error->getMessage() );
						}

						// Throw an error message to be displayed to the customer
						throw new Exception( $e->getMessage() );
					}

				} // If customer not saving the Check and not Subscription payment
				else {
					// Process Check payment
					$this->process_single_payment_request( $order );
				}
			} // If customer is using a saved profile to pay
			else {
				// Customer used saved ACH/Check
				$this->prepare_customer_id_for_request( $order, WC_PayTrace::get_post( 'paytrace-used-check' ) );

				// Should we charge the payment. Is Subs and no initial payment, we should skip
				if ( $should_we_charge_the_profile ) {
					// Process a profile payment on a saved check
					$this->process_profile_payment_request( $order, $initial_payment, $is_subscription );
				}
			}
		} // If we don't save Checks to customer profiles or the user is not logged in
		else {
			// Process Credit Card or Check payment
			$this->process_single_payment_request( $order );
		}
	}

	/**
	 * Process Card Payment. <br/>
	 * Process flow: <br/>
	 * I. Saving Cards AND User is logged in
	 *        1. Using a New card
	 *                1.a. Saving the card
	 *                        1.a.1 Create a Customer Profile using the Card details <br/>
	 *                        1.a.2 Charge the CF for the order<br/>
	 *                                1.a.2.a If Charge is not successful, remove the created CF <br/>
	 *        2. Using Saved Card
	 *                2.a. Prepare Customer Profile for payment
	 *                2.b. Charge the used Customer Profile
	 * II. Not Saving Cards OR User not logged in
	 *        1. Run a single payment transaction
	 *
	 * @since 1.3
	 *
	 * @param WC_Order   $order
	 * @param null|float $initial_payment Total amount to be charged
	 * @param bool       $is_subscription Is the payment we are processing for a Subscription
	 *
	 * @throws Exception
	 */
	protected function process_card_payment( \WC_Order $order, $initial_payment = null, $is_subscription = false ) {

		$should_we_charge_the_profile = ( $is_subscription && $initial_payment > 0 ) || ! $is_subscription;

		// If Saving Customers and User Logged in
		if ( 'yes' == $this->save_customers && is_user_logged_in() ) {

			// If entered a new CC
			if ( 'new_cc' == $this->what_payment_customer_used ) {
				$this->set_credit_card_details();

				// If customer chose to save the card or Subscription payment(always saving the card)
				if ( WC_PayTrace::get_post( 'paytrace-save-card' ) == '1' || $is_subscription ) {

					// Create customer profile
					$customer_id = $this->create_customer_profile( $order, 'card' );

					try {
						// If there is something to charge
						if ( $should_we_charge_the_profile ) {
							// Process a profile payment on a saved card
							$this->process_profile_payment_request( $order, $initial_payment, $is_subscription );
						}
					}
					catch ( Exception $e ) {
						try {
							$profile_id_to_delete = $this->get_profile_id_from_customer_id( $customer_id );

							// If we could not charge the new profile, then we are to delete it
							$this->delete_customer_profile( $profile_id_to_delete );
						}
						catch ( Exception $delete_error ) {
							// Throw an error with both error messages, from failed transaction and
							throw new Exception( $e->getMessage() . ' ' . $delete_error->getMessage() );
						}

						// Throw an error message to be displayed to the customer
						throw new Exception( $e->getMessage() );
					}
				} else // If customer not saving the card and not Subscription payment
				{
					// We will process a single payment
					$this->process_single_payment_request( $order );
				}
			} else // If customer is using a saved profile to pay
			{
				// Add the profile to be charged to the order
				$this->prepare_customer_id_for_request( $order, WC_PayTrace::get_post( 'paytrace-used-cc' ) );

				// Should we charge the payment. Is Subs and no initial payment, we should skip
				if ( $should_we_charge_the_profile ) {
					// Process a profile payment on a saved card
					$this->process_profile_payment_request( $order, $initial_payment, $is_subscription );
				}
			}
		} else // If we don't save cards to customer profiles or the user is not logged in
		{
			// Set Credit Card details
			$this->set_credit_card_details();

			// Process Credit Card
			$this->process_single_payment_request( $order );
		}
	}

	/**
	 * Get customer saved profiles
	 *
	 * @return mixed
	 */
	private function get_customer_saved_profiles() {
		return get_user_meta( get_current_user_id(), '_paytrace_saved_profiles', false );
	}

	/**
	 * Saved Customer ID to the order meta
	 *
	 * @since 1.3
	 *
	 * @param WC_Order $order
	 * @param string   $customer_id
	 */
	public function save_used_customer_id_to_order_meta( \WC_Order $order, $customer_id ) {
		// Add the used profile to the order meta
		update_post_meta( $order->id, '_paytrace_customer_id', WC_Compat_PayTrace::wc_clean( $customer_id ) );
	}

	/**
	 * Returns customer ID saved to the given order.
	 *
	 * @since 1.4
	 *
	 * @param WC_Order $order
	 *
	 * @return mixed
	 */
	public function get_customer_id_from_order( WC_Order $order ) {
		return get_post_meta( $order->id, '_paytrace_customer_id', true );
	}

	/**
	 * Retrieve the Customer Profile and save the used Customer ID to the order
	 *
	 * @since 1.3
	 *
	 * @param WC_Order $order
	 * @param string   $profile_id
	 *
	 * @return mixed
	 */
	protected function prepare_customer_id_for_request( \WC_Order $order, $profile_id ) {
		// Get all saved profiles
		$profiles = $this->get_customer_saved_profiles();
		// Get the used profile key
		$used_check = $profiles[ $profile_id ];
		// Save the customer ID to the order
		$this->save_used_customer_id_to_order_meta( $order, $used_check['customer_id'] );

		return $used_check['customer_id'];
	}

	/**
	 * Set the Request Type for the PayTrace API request. <br/>
	 * Will also set the Transaction Type if the request is ProcessTranx(Card) or ProcessCheck(Check)
	 *
	 * @since 1.3
	 *
	 * @param string      $method
	 * @param PayTraceAPI $pt
	 * @param string      $transaction_type (Optional) The transaction type. Default 'Sale'
	 */
	protected function set_request_method_type( $method, &$pt, $transaction_type = 'Sale' ) {

		// Set the method
		$pt->SetMETHOD( $method );

		// Set the transaction type, if method is a transaction
		if ( 'ProcessTranx' == $method ) {
			$pt->SetTRANXTYPE( $transaction_type );
		} elseif ( 'ProcessCheck' == $method ) {
			$pt->SetCHECKTYPE( $transaction_type );
		}
	}

	/**
	 * Set the parameters for refund with a saved profile
	 *
	 * @since 1.3
	 *
	 * @param string      $customer_id
	 * @param PayTraceAPI $pt
	 */
	private function set_refund_with_profile_request_data( $customer_id, &$pt ) {
		$saved_profile = $this->get_profile_from_customer_id( $customer_id );
		$method_type   = isset( $saved_profile['last4'] ) ? 'ProcessTranx' : 'ProcessCheck';

		// Set the customer ID to be refunded
		$pt->SetCUSTID( $customer_id );

		// Set the method type for check or card
		$this->set_request_method_type( $method_type, $pt, 'Refund' );
	}

	/**
	 * Set the parameters for a refund with transaction ID
	 *
	 * @since 1.3
	 *
	 * @param string      $transaction_id
	 * @param string      $payment_type
	 * @param string      $method_type
	 * @param PayTraceAPI $pt
	 */
	private function set_refund_with_transaction_request_data( $transaction_id, $payment_type, $method_type, &$pt ) {
		if ( 'card' == $payment_type ) {
			$pt->setTRANXID( $transaction_id );
		} else {
			$pt->setCHECKID( $transaction_id );
		}

		// Set the method type for check or card
		$this->set_request_method_type( $method_type, $pt, 'Refund' );
	}

	/**
	 * Returns the Check payment transaction type.
	 * If Authorization, it should be 'Hold'
	 * If Sale, it should be 'Sale'.
	 *
	 * @return string
	 */
	public function get_check_transaction_type() {
		return ( 'Authorization' == $this->trans_type ) ? 'Hold' : 'Sale';
	}

	/**
	 * Returns the Card payment transaction type.
	 * If Authorization, it should be 'Authorization'
	 * If Sale, it should be 'Sale'.
	 *
	 * @return string
	 */
	public function get_card_transaction_type() {
		return $this->trans_type;
	}

	/**
	 * Returns true, if order contains Subscription
	 *
	 * @since 1.4
	 *
	 * @param WC_Order $order
	 *
	 * @return bool
	 */
	public function order_contains_subscription( WC_Order $order ) {
		return false;
	}

	/**
	 * Returns true, if order contains Pre-Order
	 *
	 * @since 1.4
	 *
	 * @param WC_Order $order
	 *
	 * @return bool
	 */
	public function order_contains_pre_order( WC_Order $order ) {
		return false;
	}

} // End PayTrace class