<?php

/**
 * WC_Gateway_PayTrace_Addons class.
 *
 * WC_Gateway_PayTrace_Addons manages the subscription part of the PayTrace functionality.
 *
 * @extends WC_Gateway_PayTrace
 * @author  VanboDevelops
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

class WC_Gateway_PayTrace_Addons extends WC_Gateway_PayTrace {

	function __construct() {

		parent::__construct();

		if ( 2 === WC_PayTrace::get_subscriptions_version() ) {
			// Scheduled payment
			add_action( 'woocommerce_scheduled_subscription_payment_' . $this->id, array( $this, 'scheduled_subscription_payment_request' ), 10, 2 );

			// Meta data renewal remove
			add_action( 'wcs_resubscribe_order_created', array( $this, 'remove_renewal_order_meta' ), 10 );

			// Update change payment method
			add_action( 'woocommerce_subscription_failing_payment_method_updated_' . $this->id, array( $this, 'changed_failing_payment_method' ), 10, 2 );

			// Display card used details
			add_filter( 'woocommerce_my_subscriptions_payment_method', array( $this, 'maybe_render_subscription_payment_method' ), 10, 2 );

			// Handle display of the Admin facing payment method change
			add_filter( 'woocommerce_subscription_payment_meta', array( $this, 'add_subscription_payment_meta' ), 10, 2 );
			// Handle validation of the Admin facing payment method change
			add_filter( 'woocommerce_subscription_validate_payment_meta', array( $this, 'validate_subscription_payment_meta' ), 10, 2 );
		}

		// Add support for Pre-Orders
		if ( WC_PayTrace::is_pre_orders_active() ) {
			add_action( 'wc_pre_orders_process_pre_order_completion_payment_' . $this->id, array( $this, 'process_pre_order_release_payment' ) );
		}
	}

	/**
	 * Process the payment
	 *
	 * @param int $order_id
	 *
	 * @return array|type|void
	 */
	function process_payment( $order_id ) {
		$order = WC_Compat_PayTrace::wc_get_order( $order_id );

		if ( $this->order_contains_pre_order( $order ) ) {
			// Pre-Orders Payment
			return $this->process_pre_orders_payment( $order );
		} else if ( $this->order_contains_subscription( $order ) ) {
			// Subscription Payment
			return $this->process_subscription_payment( $order );
		} else {
			// Normal Products Payment
			return parent::process_payment( $order_id );
		}
	}

	/**
	 * Process subscription payment
	 *
	 * @since 1.2
	 *
	 * @param WC_Order $order
	 *
	 * @return type
	 */
	public function process_subscription_payment( $order ) {
		try {
			$initial_payment = $order->get_total();

			// Using Check(ACH) to pay
			if ( 'check' == $this->what_payment_customer_used || 'saved_check' == $this->what_payment_customer_used ) {
				$this->process_ach_payment( $order, $initial_payment, true );
			} else {
				$this->process_card_payment( $order, $initial_payment, true );
			}

			// Remove cart
			WC_Compat_PayTrace::empty_cart();

			$customer_id = $this->get_customer_id_from_order( $order );
			$this->save_meta_data_to_subscription( $order, $customer_id );

			// Return thank you page redirect
			return array(
				'result'   => 'success',
				'redirect' => $this->get_return_url( $order )
			);
		}
		catch ( Exception $e ) {
			WC_Compat_PayTrace::wc_add_notice( $e->getMessage(), 'error' );

			return;
		}
	}

	/**
	 * Process subscription payment for each period
	 *
	 * @since 1.4
	 *
	 * @param float     $amount_to_charge
	 * @param \WC_Order $renewal_order
	 */
	public function scheduled_subscription_payment_request( $amount_to_charge, \WC_Order $renewal_order ) {
		try {
			WC_PayTrace::add_debug_log( 'Scheduled payment: ' . print_r( $renewal_order->id, true ) );

			$this->process_profile_payment_request( $renewal_order, $amount_to_charge, true );
		}
		catch ( Exception $e ) {
			$renewal_order->update_status( 'failed', $e->getMessage() );

			// Debug log
			WC_PayTrace::add_debug_log( $e->getMessage() );
		}
	}

	/**
	 * Save the transaction details to the Subscription
	 *
	 * @since 1.4
	 *
	 * @param $order
	 * @param $customer_id
	 */
	public function save_meta_data_to_subscription( $order, $customer_id ) {
		// Also store it on the subscriptions being purchased or paid for in the order
		if ( wcs_order_contains_subscription( $order ) ) {
			$subscriptions = wcs_get_subscriptions_for_order( $order );
		} elseif ( wcs_order_contains_renewal( $order ) ) {
			$subscriptions = wcs_get_subscriptions_for_renewal_order( $order );
		} else {
			$subscriptions = array();
		}

		foreach ( $subscriptions as $subscription ) {
			// Debug log
			WC_PayTrace::add_debug_log( 'Saving details to subscription: ' . print_r( $subscription->id, true ) );

			$this->save_used_customer_id_to_order_meta( $subscription, $customer_id );
		}
	}

	/**
	 * Don't transfer PayTrace meta to resubscribe orders.
	 *
	 * @since 1.4
	 *
	 * @param int $resubscribe_order The order created for the customer to resubscribe to the old expired/cancelled subscription
	 *
	 * @return void
	 */
	public function remove_renewal_order_meta( $resubscribe_order ) {
		delete_post_meta( $resubscribe_order->id, '_paytrace_customer_id' );
	}

	/**
	 * Update the accountID and the Serial No on the original order, when a renewal order is placed, after it failed previously
	 *
	 * @since 1.2
	 *
	 * @param WC_Order $subscription
	 * @param WC_Order $renewal_order
	 */
	public function changed_failing_payment_method( \WC_Order $subscription, \WC_Order $renewal_order ) {
		update_post_meta( $subscription->id, '_paytrace_customer_id', get_post_meta( $renewal_order->id, '_paytrace_customer_id', true ) );
	}


	/**
	 * Show payment method on My Subscriptions section of My Account page
	 *
	 * @since 1.2
	 *
	 * @param string   $payment_method_to_display
	 * @param WC_Order $subscription
	 *
	 * @return string
	 */
	public function maybe_render_subscription_payment_method( $payment_method_to_display, $subscription ) {
		if ( $this->id !== $subscription->payment_method ) {
			return $payment_method_to_display;
		}

		$customer_profile = $this->get_profile_from_customer_id( get_post_meta( $subscription->id, '_paytrace_customer_id', true ) );

		if ( is_array( $customer_profile ) ) {
			$method_type = ( isset( $customer_profile['last4'] ) ) ? 'card' : 'check';
			if ( 'check' == $method_type ) {
				$payment_method_to_display = sprintf( __( 'Via Check with account ending in %s', WC_PayTrace::TEXT_DOMAIN ), $customer_profile['dda_last4'] );
			} else {
				$payment_method_to_display = sprintf( __( 'Via Credit Card ending in %s', WC_PayTrace::TEXT_DOMAIN ), $customer_profile['last4'] );
			}
		}

		return $payment_method_to_display;
	}

	/**
	 * Add payment method change fields
	 *
	 * @since 1.4
	 *
	 * @param $payment_meta
	 * @param $subscription
	 *
	 * @return mixed
	 */
	public function add_subscription_payment_meta( $payment_meta, $subscription ) {
		$payment_meta[ $this->id ] = array(
			'post_meta' => array(
				'_paytrace_customer_id' => array(
					'value' => get_post_meta( $subscription->id, '_paytrace_customer_id', true ),
					'label' => 'PayTrace Customer Profile ID',
				),
			),
		);

		return $payment_meta;
	}

	/**
	 * Validate Payment method change
	 *
	 * @since 1.4
	 *
	 * @param $payment_method_id
	 * @param $payment_meta
	 *
	 * @throws Exception
	 */
	public function validate_subscription_payment_meta( $payment_method_id, $payment_meta ) {
		if ( $this->id === $payment_method_id ) {
			if ( ! isset( $payment_meta['post_meta']['_paytrace_customer_id']['value'] )
				|| empty( $payment_meta['post_meta']['_paytrace_customer_id']['value'] )
			) {
				throw new Exception( 'A "_paytrace_customer_id"(PayTrace Customer Profile ID) value is required.' );
			}
		}
	}

	/**
	 * Process Pre-Order payment request.
	 * Only Register the customer payment method, pre-order payment is delayed one.
	 *
	 * @since 1.2
	 *
	 * @param WC_Order $order
	 *
	 * @return array
	 */
	private function process_pre_orders_payment( WC_Order $order ) {

		if ( WC_Pre_Orders_Order::order_requires_payment_tokenization( $order ) ) {
			try {
				// Using Check(ACH) to pay
				if ( 'check' == $this->what_payment_customer_used || 'saved_check' == $this->what_payment_customer_used ) {
					// Handle Customer Profile
					if ( 'new_check' == $this->what_payment_customer_used ) {
						// Create and save new customer profile
						$this->create_customer_profile( $order, 'check' );
					} else {
						$this->prepare_customer_id_for_request( $order, WC_PayTrace::get_post( 'paytrace-used-check' ) );
					}
				} else {
					// Handle Customer Profile
					if ( 'new_cc' == $this->what_payment_customer_used ) {
						// Create and save new customer profile
						$this->create_customer_profile( $order, 'card' );
					} else {
						$this->prepare_customer_id_for_request( $order, WC_PayTrace::get_post( 'paytrace-used-cc' ) );
					}
				}

				// Now that we have the info need for future payment, mark the order pre-ordered
				WC_Pre_Orders_Order::mark_order_as_pre_ordered( $order );

				// Empty cart
				WC_Compat_PayTrace::empty_cart();

				// Redirect to thank you page
				return array(
					'result'   => 'success',
					'redirect' => $this->get_return_url( $order ),
				);
			}
			catch ( Exception $e ) {
				WC_Compat_PayTrace::wc_add_notice( $e->getMessage(), 'error' );
			}
		} else {
			// Process order normally
			return parent::process_payment( $order->id );
		}
	}

	/**
	 * Charge the payment on order release
	 *
	 * @since 1.2
	 *
	 * @param WC_Order $order
	 */
	public function process_pre_order_release_payment( \WC_Order $order ) {
		try {
			$this->process_profile_payment_request( $order );

			// Payment complete
			$order->payment_complete();
		}
		catch ( Exception $e ) {
			$order->add_order_note( $e->getMessage(), 'error' );
		}
	}

	/**
	 * Returns true, if order contains Subscription
	 *
	 * @since 1.4
	 *
	 * @param WC_Order $order
	 *
	 * @return bool
	 */
	public function order_contains_subscription( WC_Order $order ) {
		if (
			2 === WC_PayTrace::get_subscriptions_version()
			&& (
				wcs_order_contains_subscription( $order )
				|| wcs_order_contains_renewal( $order )
			)
		) {
			return true;
		}

		return false;
	}

	/**
	 * Returns true, if order contains Pre-Order
	 *
	 * @since 1.4
	 *
	 * @param WC_Order $order
	 *
	 * @return bool
	 */
	public function order_contains_pre_order( WC_Order $order ) {
		if (
			WC_PayTrace::is_pre_orders_active()
			&& WC_Pre_Orders_Order::order_contains_pre_order( $order )
		) {
			return true;
		}

		return false;
	}
}