<?php

/*
 * Common functions. 
 */
if ( ! defined( 'ABSPATH' ) ) {
	exit ; // Exit if accessed directly.
}

require_once 'efw-layout-functions.php' ;
include_once 'efw-post-functions.php' ;

if ( ! function_exists( 'efw_check_is_array' ) ) {

	/**
	 * Check if resource is array.
	 *
	 * @return bool
	 */
	function efw_check_is_array( $array ) {
		return ( is_array( $array ) && ! empty( $array ) ) ;
	}

}

if ( ! function_exists( 'efw_page_screen_ids' ) ) {

	/**
	 * Get page screen IDs.
	 *
	 * @return array
	 */
	function efw_page_screen_ids() {

		$wc_screen_id = sanitize_title( esc_html__( 'WooCommerce', 'woocommerce' ) ) ;

		return apply_filters( 'efw_page_screen_ids', array(
			'product',
			$wc_screen_id . '_page_efw_settings',
				) ) ;
	}

}

if ( ! function_exists( 'efw_get_allowed_setting_tabs' ) ) {

	/**
	 * Get setting tabs.
	 * 
	 * @return array
	 */
	function efw_get_allowed_setting_tabs() {

		return apply_filters( 'efw_settings_tabs_array', array() ) ;
	}

}

if ( ! function_exists( 'efw_get_settings_page_url' ) ) {

	/**
	 * Get Settings page URL.
	 * 
	 * @return array
	 */
	function efw_get_settings_page_url( $args = array() ) {

		$url = add_query_arg( array( 'page' => 'efw_settings' ), admin_url( 'admin.php' ) ) ;

		if ( efw_check_is_array( $args ) ) {
			$url = add_query_arg( $args, $url ) ;
		}

		return $url ;
	}

}

if ( ! function_exists( 'efw_get_wc_categories' ) ) {

	/**
	 * Get WC Categories.
	 *
	 * @return array
	 */
	function efw_get_wc_categories() {
		$categories    = array() ;
		$wc_categories = get_terms( 'product_cat' ) ;

		if ( ! efw_check_is_array( $wc_categories ) ) {
			return $categories ;
		}

		foreach ( $wc_categories as $category ) {
			$categories[ $category->term_id ] = $category->name ;
		}

		return $categories ;
	}

}

if ( ! function_exists( 'efw_get_wc_shipping_methods' ) ) {

	/**
	 * Get Active WC Shipping Methods.
	 *
	 * @return array
	 */
	function efw_get_wc_shipping_methods() {
		$shipping_methods    = array() ;
		$wc_shipping_methods = WC()->shipping()->get_shipping_methods() ;

		if ( ! efw_check_is_array( $wc_shipping_methods ) ) {
			return $shipping_methods ;
		}

		foreach ( $wc_shipping_methods as $shipping_method ) {
			$shipping_methods[ $shipping_method->id ] = $shipping_method->method_title ;
		}

		return $shipping_methods ;
	}

}

if ( ! function_exists( 'efw_get_wc_available_gateways' ) ) {

	/**
	 * Get WC Gateway Id.
	 *
	 * @return array
	 */
	function efw_get_wc_available_gateways( $active = false ) {
		$available_gateways = array() ;
		$wc_gateways        = WC()->payment_gateways->payment_gateways() ;

		if ( ! efw_check_is_array( $wc_gateways ) ) {
			return $available_gateways ;
		}

		foreach ( $wc_gateways as $gateway ) {

			$enabled = $active ? ( 'yes' == $gateway->enabled ) : true ;

			if ( $enabled ) {
				$available_gateways[ $gateway->id ] = $gateway->title ;
			}
		}

		return $available_gateways ;
	}

}

if ( ! function_exists( 'efw_get_wc_cart_subtotal' ) ) {

	/**
	 * Get WC cart Subtotal
	 *
	 * @return string/float
	 */
	function efw_get_wc_cart_subtotal( $cart_obj = null ) {
		if ( ! $cart_obj ) {
			$cart_obj = WC()->cart ;
		}

		if ( method_exists( $cart_obj, 'get_cart_contents_total' ) ) {
			$subtotal = ( 'incl' == get_option( 'woocommerce_tax_display_cart' ) ) ? $cart_obj->get_cart_contents_total() + $cart_obj->get_cart_contents_tax() : $cart_obj->get_cart_contents_total() ;
		} else {
			$subtotal = ( 'incl' == get_option( 'woocommerce_tax_display_cart' ) ) ? $cart_obj->cart_contents_total + $cart_obj->cart_contents_tax : $cart_obj->cart_contents_total ;
		}

		return $subtotal ;
	}

}

if ( ! function_exists( 'efw_get_wc_cart_total' ) ) {

	/**
	 * Get WC cart total.
	 *
	 * @return string/float
	 */
	function efw_get_wc_cart_total( $cart_obj = null ) {
		if ( ! $cart_obj ) {
			$cart_obj = WC()->cart ;
		}

		if ( method_exists( $cart_obj, 'get_cart_contents_total' ) ) {
			$total = $cart_obj->get_cart_contents_total() + $cart_obj->get_cart_contents_tax() + $cart_obj->get_shipping_total() + $cart_obj->get_shipping_tax() + $cart_obj->get_fee_total() + $cart_obj->get_fee_tax() ;
		} else {
			$total = $cart_obj->cart_contents_total + $cart_obj->cart_contents_tax + $cart_obj->shipping_total + $cart_obj->shipping_tax + $cart_obj->fee_total + $cart_obj->fee_tax ;
		}

		return $total ;
	}

}

if ( ! function_exists( 'efw_product_filter' ) ) {

	/**
	 * Check if Product is applicable for Product Fee.
	 *
	 * @return bool
	 */
	function efw_product_filter( $product_id ) {

		if ( '1' == get_option( 'efw_productfee_apply_for' ) ) {
			return true ;
		} elseif ( '2' == get_option( 'efw_productfee_apply_for' ) ) {
			$include_product = get_option( 'efw_productfee_include_products' ) ;
			$include_product = ! empty( $include_product ) ? $include_product : array() ;
			$include_product = efw_check_is_array( $include_product ) ? $include_product : explode( ',', $include_product ) ;
			if ( in_array( $product_id, $include_product ) ) {
				return true ;
			}
		} elseif ( '3' == get_option( 'efw_productfee_apply_for' ) ) {
			$exclude_product = get_option( 'efw_productfee_exclude_products' ) ;
			$exclude_product = ! empty( $exclude_product ) ? $exclude_product : array() ;
			$exclude_product = efw_check_is_array( $exclude_product ) ? $exclude_product : explode( ',', $exclude_product ) ;
			if ( ! in_array( $product_id, $exclude_product ) ) {
				return true ;
			}
		} elseif ( '4' == get_option( 'efw_productfee_apply_for' ) ) {
			$include_categories = get_option( 'efw_productfee_include_category' ) ;
			$include_categories = ! empty( $include_categories ) ? $include_categories : array() ;
			$include_categories = efw_check_is_array( $include_categories ) ? $include_categories : explode( ',', $include_categories ) ;
			$product            = wc_get_product( $product_id ) ;
			$productid          = empty( $product->get_parent_id() ) ? $product_id : $product->get_parent_id() ;
			$product_categories = get_the_terms( $productid, 'product_cat' ) ;
			if ( efw_check_is_array( $product_categories ) ) {
				foreach ( $product_categories as $terms ) {
					if ( in_array( $terms->term_id, $include_categories ) ) {
						return true ;
					}
				}
			}
		} else {
			$exclude_categories = get_option( 'efw_productfee_exclude_category' ) ;
			$exclude_categories = ! empty( $exclude_categories ) ? $exclude_categories : array() ;
			$exclude_categories = efw_check_is_array( $exclude_categories ) ? $exclude_categories : explode( ',', $exclude_categories ) ;
			$product            = wc_get_product( $product_id ) ;
			$productid          = empty( $product->get_parent_id() ) ? $product_id : $product->get_parent_id() ;
			$product_categories = get_the_terms( $productid, 'product_cat' ) ;
			if ( efw_check_is_array( $product_categories ) ) {
				foreach ( $product_categories as $terms ) {
					if ( ! in_array( $terms->term_id, $exclude_categories ) ) {
						return true ;
					}
				}
			}
		}

		return false ;
	}

}

if ( ! function_exists( 'efw_global_fee_value' ) ) {

	/**
	 * Global Fee Value for Product.
	 *
	 * @return float
	 */
	function efw_global_fee_value( $price ) {
		if ( '1' == get_option( 'efw_productfee_fee_type' ) ) {
			$fee_value = ( float ) get_option( 'efw_productfee_fixed_value' ) ;
		} else {
			$percent_value = ( float ) get_option( 'efw_productfee_percent_value' ) ;
			$fee_value     = ( $percent_value / 100 ) * ( float ) $price ;
		}

		return ( float ) $fee_value ;
	}

}

if ( ! function_exists( 'efw_get_fee_text' ) ) {

	/**
	 * Get Fee Text for Product.
	 *
	 * @return string
	 */
	function efw_get_fee_text( $product_id ) {
		if ( '1' == get_option( 'efw_productfee_fee_setup' ) ) {
			$fee_text = get_option( 'efw_productfee_fee_text' ) ;
		} else {
			if ( '1' == get_post_meta( $product_id, '_efw_text_from', true ) ) {
				$fee_text = get_option( 'efw_productfee_fee_text' ) ;
			} else {
				$fee_text = get_post_meta( $product_id, '_efw_fee_text', true ) ;
			}
		}

		return $fee_text ;
	}

}

if ( ! function_exists( 'efw_get_wp_user_roles' ) ) {

	/**
	 * Get WordPress User Roles
	 *
	 * @return array
	 */
	function efw_get_wp_user_roles() {
		global $wp_roles ;
		$user_roles = array() ;

		if ( ! isset( $wp_roles->roles ) || ! efw_check_is_array( $wp_roles->roles ) ) {
			return $user_roles ;
		}

		foreach ( $wp_roles->roles as $slug => $role ) {
			$user_roles[ $slug ] = $role[ 'name' ] ;
		}

		return $user_roles ;
	}

}

if ( ! function_exists( 'efw_get_user_roles' ) ) {

	/**
	 * Get User Roles
	 *
	 * @return array
	 */
	function efw_get_user_roles( $extra_options = array() ) {
		$user_roles = efw_get_wp_user_roles() ;

		$user_roles[ 'guest' ] = esc_html__( 'Guest', 'extra-fees-for-woocommerce' ) ;

		$user_roles = array_merge( $user_roles, $extra_options ) ;

		return $user_roles ;
	}

}
