<?php
/**
 * Toolbar user forms helper
 *
 * @package Toolset/Forms
 */

namespace OTGS\Toolset\CRED\Model\Forms\User;

use OTGS\Toolset\CRED\Controller\Forms\User\Main as UserFormMain;
use OTGS\Toolset\CRED\Model\FormEditorToolbar\Helper as BaseHelper;
use OTGS\Toolset\CRED\Controller\FieldsControl\Db as FieldsControlDb;

/**
 * Helper for the user forms content editor toolbar.
 *
 * Provides convenient methods for gathering relevant data, like fields for a given set of roles.
 *
 * @since 2.1
 */
class Helper extends BaseHelper {

	/**
	 * List of roles to get field for.
	 *
	 * @var array
	 *
	 * @since 2.1
	 */
	private $roles = array();

	/**
	 * Types conditions
	 *
	 * @var Toolset_Condition_Plugin_Types_Active
	 *
	 * @since 2.1
	 */
	private $di_toolset_types_condition = null;

	/**
	 * Constructor
	 *
	 * @param array                                 $roles Roles.
	 * @param Toolset_Condition_Plugin_Types_Active $di_toolset_types_condition For testing purposes.
	 */
	public function __construct( $roles = array(), $di_toolset_types_condition = null ) {
		$this->roles = $roles;
		$this->items = array(
			'basic' => array(),
			'meta'  => array(),
		);

		$this->di_toolset_types_condition = ( null === $di_toolset_types_condition )
			? new Toolset_Condition_Plugin_Types_Active()
			: $di_toolset_types_condition;
	}

	/**
	 * Populate and return the list of items.
	 *
	 * @return array
	 *
	 * @since 2.1
	 */
	public function populate_items() {
		$this->populate_basic_fields();
		$this->populate_types_fields_by_roles();
		$this->populate_non_types_fields();

		$this->add_label_option_to_fields();

		return $this->items;
	}

	/**
	 * Populate the list of basic fields.
	 *
	 * @since 2.1
	 */
	protected function populate_basic_fields() {
		$this->items['basic']['user_login'] = array(
			'label'             => __( 'Username', 'wp-cred' ),
			'shortcode'         => UserFormMain::SHORTCODE_NAME_FORM_FIELD,
			'autogeneratedItem' => true,
			'class'             => 'js-cred-toolbar-scaffold-item-autogeneratedUsername',
			'attributes' => array(
				\CRED_Form_Builder::SCAFFOLD_FIELD_ID => 'user_login',
				'field'                               => 'user_login',
				'class'                               => 'form-control',
				'output'                              => 'bootstrap',
			),
			'options'           => $this->get_value_and_url_options(),
		);

		$this->items['basic']['user_pass'] = array(
			'label'             => __( 'Password', 'wp-cred' ),
			'shortcode'         => UserFormMain::SHORTCODE_NAME_FORM_FIELD,
			'autogeneratedItem' => true,
			'class'             => 'js-cred-toolbar-scaffold-item-autogeneratedPassword',
			'attributes'        => array(
				\CRED_Form_Builder::SCAFFOLD_FIELD_ID => 'user_pass',
				'field'                               => 'user_pass',
				'class'                               => 'form-control',
				'output'                              => 'bootstrap',
			),
			'options'           => $this->get_value_and_url_options(
				array(
					'confirm_pass' => array(
						'label'             => __( 'Password confirmation', 'wp-cred' ),
						'type'              => 'radio',
						'options'           => array(
							'yes' => __( 'Force users to confirm their password', 'wp-cred' ),
							'no'  => __( 'Do not force users to confirm their password', 'wp-cred' ),
						),
						'defaultForceValue' => 'yes',
					),
				)
			),
		);

		$this->items['basic']['user_email'] = array(
			'label'      => __( 'Email', 'wp-cred' ),
			'shortcode'  => UserFormMain::SHORTCODE_NAME_FORM_FIELD,
			'attributes' => array(
				\CRED_Form_Builder::SCAFFOLD_FIELD_ID => 'user_email',
				'field'                               => 'user_email',
				'class'                               => 'form-control',
				'output'                              => 'bootstrap',
			),
			'options'    => $this->get_value_and_url_options(),
		);

		$this->items['basic']['nickname'] = array(
			'label'             => __( 'Nickname', 'wp-cred' ),
			'shortcode'         => UserFormMain::SHORTCODE_NAME_FORM_FIELD,
			'autogeneratedItem' => true,
			'class'             => 'js-cred-toolbar-scaffold-item-autogeneratedNickname',
			'attributes'        => array(
				\CRED_Form_Builder::SCAFFOLD_FIELD_ID => 'nickname',
				'field'                               => 'nickname',
				'class'                               => 'form-control',
				'output'                              => 'bootstrap',
			),
			'options'           => $this->get_value_and_url_options(),
		);

		$this->items['basic']['first_name'] = array(
			'label'      => __( 'First name', 'wp-cred' ),
			'shortcode'  => UserFormMain::SHORTCODE_NAME_FORM_FIELD,
			'attributes' => array(
				\CRED_Form_Builder::SCAFFOLD_FIELD_ID => 'first_name',
				'field'                               => 'first_name',
				'class'                               => 'form-control',
				'output'                              => 'bootstrap',
			),
			'options'    => $this->get_value_and_url_options(),
		);

		$this->items['basic']['last_name'] = array(
			'label'      => __( 'Last name', 'wp-cred' ),
			'shortcode'  => UserFormMain::SHORTCODE_NAME_FORM_FIELD,
			'attributes' => array(
				\CRED_Form_Builder::SCAFFOLD_FIELD_ID => 'last_name',
				'field'                               => 'last_name',
				'class'                               => 'form-control',
				'output'                              => 'bootstrap',
			),
			'options'    => $this->get_value_and_url_options(),
		);

		$this->items['basic']['description'] = array(
			'label'      => __( 'Biographical info', 'wp-cred' ),
			'shortcode'  => UserFormMain::SHORTCODE_NAME_FORM_FIELD,
			'attributes' => array(
				\CRED_Form_Builder::SCAFFOLD_FIELD_ID => 'description',
				'field'                               => 'description',
				'class'                               => 'form-control',
				'output'                              => 'bootstrap',
			),
			'options'    => $this->get_value_and_url_options(),
		);

	}

	/**
	 * Populate the list of Types fields.
	 *
	 * @since 2.1
	 */
	private function populate_types_fields_by_roles() {
		if ( empty( $this->roles ) ) {
			return;
		}

		// Only populate fields if Types is active.
		if ( ! $this->di_toolset_types_condition->is_met() ) {
			return;
		}

		$groups_by_roles           = array();
		$groups_included           = array();
		$user_fields_group_factory = \Toolset_Field_Group_User_Factory::get_instance();
		$groups_by_roles           = $user_fields_group_factory->get_groups_by_roles();

		foreach ( $this->roles as $user_role ) {
			$groups_by_role = toolset_getarr( $groups_by_roles, $user_role, array() );

			if ( empty( $groups_by_role ) ) {
				continue;
			}

			foreach ( $groups_by_role as $field_group ) {

				// AVoid adding the same group twice if assigned to more than one role.
				$group_slug = $field_group->get_slug();
				if ( in_array( $group_slug, $groups_included, true ) ) {
					continue;
				} else {
					$groups_included[] = $group_slug;
				}

				$fields_in_group = $field_group->get_field_definitions();

				foreach ( $fields_in_group as $field_definition ) {
					$field      = $field_definition->get_definition_array();
					$field_type = $field_definition->get_type()->get_slug();
					$this->items['meta'][ $field_definition->get_slug() ] = array(
						'groupName'    => $field_group->get_name(),
						'label'        => $field_definition->get_name(),
						'shortcode'    => UserFormMain::SHORTCODE_NAME_FORM_FIELD,
						'requiredItem' => ( isset( $field['data']['validate']['required']['active'] ) && $field['data']['validate']['required']['active'] ),
						'type'         => $field_type,
						'isRepetitive' => $field_definition->get_is_repetitive(),
						'metaKey'      => $field_definition->get_meta_key(),
						'attributes'   => array(
							\CRED_Form_Builder::SCAFFOLD_FIELD_ID => $field_definition->get_slug(),
							'field'      => $field_definition->get_slug(),
							'force_type' => 'field',
							'class'      => 'form-control',
							'output'     => 'bootstrap',
						),
						'icon_class'   => $field_definition->get_type()->get_icon_classes(),
						'options'      => in_array( $field_type, $this->field_types_without_value_and_url_options )
							? array()
							: $this->get_value_and_url_options(),
					);
				}
			}
		}

	}

	/**
	 * Populates non type fields
	 */
	private function populate_non_types_fields() {
		$fields_control_db = new FieldsControlDb();
		$cred_fields       = $fields_control_db->get_fields_per_post_type( CRED_USER_FORMS_CUSTOM_POST_NAME );

		foreach ( $cred_fields as $field ) {
			if ( toolset_getarr( $field, '_cred_ignore', false ) ) {
				continue;
			}
			$this->items['meta'][ $field['slug'] ] = array(
				'groupName'    => __( 'Non-Toolset fields under Forms control', 'wp-cred' ),
				'label'        => $field['slug'],
				'shortcode'    => UserFormMain::SHORTCODE_NAME_FORM_FIELD,
				'requiredItem' => ( isset( $field['data']['validate']['required']['active'] ) && $field['data']['validate']['required']['active'] ),
				'type'         => $field['type'],
				'isRepetitive' => false,
				'metaKey'      => $field['slug'],
				'attributes'   => array(
					\CRED_Form_Builder::SCAFFOLD_FIELD_ID => $field['slug'],
					'field'                               => $field['slug'],
					'force_type'                          => 'field',
					'class'                               => 'form-control',
					'output'                              => 'bootstrap',
				),
				'options'      => $this->get_value_and_url_options(),
			);
		}
	}
}
