<?php
/**
 * Plugin Name: WooCommerce PayTrace Payment Gateway
 * Plugin URI: https://woocommerce.com/products/paytrace-gateway/
 * Description: Allows you to use <a href="http://www.paytrace.com/">PayTrace</a> payment processor with the WooCommerce plugin.
 * Version: 2.7.1
 * Author: VanboDevelops
 * Author URI: http://www.vanbodevelops.com
 * Woo: 18723:75d0d10deafdfc281fd154423ef70922
 * WC requires at least: 3.0.0
 * WC tested up to: 6.5.0
 *
 *        Copyright: (c) 2012 - 2022 VanboDevelops
 *        License: GNU General Public License v3.0
 *        License URI: http://www.gnu.org/licenses/gpl-3.0.html
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Required functions
 */
if ( ! function_exists( 'woothemes_queue_update' ) ) {
	require_once( 'woo-includes/woo-functions.php' );
}

/**
 * Plugin updates
 */
woothemes_queue_update( plugin_basename( __FILE__ ), '75d0d10deafdfc281fd154423ef70922', '18723' );

if ( ! is_woocommerce_active() ) {
	return;
}

class WC_Paytrace {
	
	/**
	 * Text domain string.
	 */
	const TEXT_DOMAIN = 'woocommerce-gateway-paytrace';
	/**
	 * The plugin version
	 */
	const VERSION = '2.7.1';
	/**
	 * The files and folders version
	 * Should be changes every time there is a new class file added or one deleted
	 * @since 2.0
	 */
	const FILES_VERSION = '1.4.1';
	/**
	 * WC Logger object
	 * @var object
	 */
	private static $log;
	/**
	 * Plugin URL
	 * @var string
	 */
	private static $plugin_url;
	/**
	 * Plugin Path
	 * @var string
	 */
	private static $plugin_path;
	/**
	 * Is WC Subscriptions active
	 * @var bool
	 */
	private static $is_subscriptions_active;
	/**
	 * Hold WC Subscriptions main version
	 * @var bool
	 */
	private static $is_subscriptions_version;
	/**
	 * Is WC Pre-Orders active
	 * @var bool
	 */
	private static $is_pre_orders_active;
	/**
	 * Do we have debug mode enabled
	 * @var bool
	 */
	private static $is_debug_enabled;
	const MIN_PHP_VERSION = '7.0.0';
	const MIN_WC_VERSION = '3.0.0';
	
	public function __construct() {
		do_action( 'wc_paytrace_pre_loaded' );
		
		// Add required files
		$this->load_autoloader();
		
		$this->load_updates();
		
		add_action( 'init', array( $this, 'load_text_domain' ) );
		add_action( 'init', array( $this, 'load_plugin_essentials' ) );
		
		// Add a 'Settings' link to the plugin action links
		add_filter( 'plugin_action_links_' . plugin_basename( __FILE__ ), array(
			$this,
			'settings_support_link',
		), 10, 4 );
		
		add_filter( 'woocommerce_payment_gateways', array( $this, 'add_payment_gateway' ) );
		
		do_action( 'wc_paytrace_loaded' );
	}
	
	/**
	 * Loads the plugin autoloader
	 *
	 * @since 2.0
	 */
	public function load_autoloader() {
		require_once( 'classes/class-wc-paytrace-autoloader.php' );
		
		$loader = new WC_Paytrace_Autoloader( self::plugin_path(), self::FILES_VERSION, 'classes' );
		
		spl_autoload_register( array( $loader, 'load_classes' ) );
	}
	
	/**
	 * Loads any version updates for the plugin
	 * @since 2.3.0
	 */
	public function load_updates() {
		$updates = new WC_Paytrace_Updates( 'paytrace' );
		$updates->hooks();
	}
	
	/**
	 * Loads localization
	 *
	 * @since 2.0
	 */
	public function load_text_domain() {
		$locale = is_admin() && function_exists( 'get_user_locale' ) ? get_user_locale() : get_locale();
		$locale = apply_filters( 'plugin_locale', $locale, self::TEXT_DOMAIN );
		
		unload_textdomain( self::TEXT_DOMAIN );
		load_textdomain( self::TEXT_DOMAIN, WP_LANG_DIR . '/' . self::TEXT_DOMAIN . '/' . self::TEXT_DOMAIN . '-' . $locale . '.mo' );
		load_plugin_textdomain( self::TEXT_DOMAIN, false, plugin_basename( dirname( __FILE__ ) ) . '/languages' );
	}
	
	/**
	 * Init localisations and files
	 */
	public function load_plugin_essentials() {
		// No load, if WC is not loaded
		if ( ! class_exists( 'WC_Payment_Gateway' ) ) {
			return;
		}
		
		if ( WC_Paytrace_Compat::is_wc_2_6() ) {
			// Include the token classes
			include_once( 'classes/tokens/class-wc-payment-token-paytrace-cc.php' );
			include_once( 'classes/tokens/class-wc-payment-token-paytrace-echeck.php' );
		}
		
		// Scripts
		$this->load_scripts();
		
		// Ajax
		$this->load_ajax();
		
		// Load the manage cards class
		$this->load_manage_cards();
		
		if ( is_admin() ) {
			// Admin
			$this->load_admin();
		}
	}
	
	/**
	 * Add the gateway to WooCommerce
	 *
	 * @since 1.2
	 *
	 * @param array $methods
	 *
	 * @return array
	 */
	function add_payment_gateway( $methods ) {
		$methods[] = self::get_gateway_class();
		
		return $methods;
	}
	
	/**
	 * Add 'Settings' link to the plugin actions links
	 *
	 * @since 1.2
	 *
	 * @param array $actions
	 *
	 * @return array associative array of plugin action links
	 */
	public function settings_support_link( $actions, $plugin_file, $plugin_data, $context ) {
		$gateway = 'paytrace';
		if ( ! WC_Paytrace_Compat::is_wc_2_6() ) {
			$gateway = self::get_gateway_class();
		}
		
		return array_merge(
			array( 'settings' => '<a href="' . WC_Paytrace_Compat::gateway_settings_page( $gateway ) . '">' . __( 'Settings', self::TEXT_DOMAIN ) . '</a>' ),
			$actions
		);
	}
	
	/**
	 * Get the correct gateway class name to load
	 *
	 * @since 1.2
	 * @return string Class name
	 */
	public static function get_gateway_class() {
		if ( self::is_pre_orders_active() || self::is_subscriptions_active() ) {
			$methods = 'WC_Paytrace_Gateway_Addons';
		} else {
			$methods = 'WC_Paytrace_Gateway';
		}
		
		return $methods;
	}
	
	/**
	 * Safely get POST variables
	 *
	 * @since 1.2
	 *
	 * @param string $name POST variable name
	 *
	 * @return string The variable value
	 */
	public static function get_post( $name ) {
		if ( isset( $_POST[ $name ] ) ) {
			return $_POST[ $name ];
		}
		
		return null;
	}
	
	/**
	 * Safely get GET variables
	 *
	 * @since 1.2
	 *
	 * @param string $name GET variable name
	 *
	 * @return string The variable value
	 */
	public static function get_get( $name ) {
		if ( isset( $_GET[ $name ] ) ) {
			return $_GET[ $name ];
		}
		
		return null;
	}
	
	/**
	 * Safely retrieve an array or object key/property
	 *
	 * @since 2.0
	 *
	 * @param string       $name    Name of the key/prop
	 * @param array|object $stack   The stack we are looking in
	 * @param string       $default Variable name
	 *
	 * @return mixed The variable value
	 */
	public static function get_field( $name, $stack, $default = '' ) {
		
		if ( is_array( $stack ) ) {
			if ( isset( $stack[ $name ] ) ) {
				return $stack[ $name ];
			}
		}
		
		if ( is_object( $stack ) ) {
			if ( isset( $stack->{$name} ) ) {
				return $stack->{$name};
			}
		}
		
		return $default;
	}
	
	/**
	 * Add debug log message
	 *
	 * @since   1.2
	 * @version 2.4.0
	 *
	 * @param string $message
	 */
	public static function add_debug_log( $message, $handle = 'paytrace', $level = 'debug' ) {
		if ( ! is_object( self::$log ) ) {
			self::$log = new WC_Paytrace_Debug( 'paytrace' );
		}
		
		self::$log->add_debug_log( $message, $handle, $level );
	}
	
	/**
	 * Check, if debug logging is enabled
	 *
	 * @deprecated 2.4.0 - Will be removed in future versions
	 *
	 * @since      1.3
	 *
	 * @return bool
	 */
	public static function is_debug_enabled() {
		if ( self::$is_debug_enabled ) {
			return self::$is_debug_enabled;
		} else {
			$paytrace_settings = get_option( 'woocommerce_paytrace_settings' );
			
			self::$is_debug_enabled = ( 'yes' == $paytrace_settings['debug'] );
			
			return self::$is_debug_enabled;
		}
	}
	
	/**
	 * Get the plugin url
	 *
	 * @since 1.2
	 * @return string
	 */
	public static function plugin_url() {
		if ( self::$plugin_url ) {
			return self::$plugin_url;
		}
		
		return self::$plugin_url = untrailingslashit( plugins_url( '/', __FILE__ ) );
	}
	
	/**
	 * Get the plugin path
	 *
	 * @since 1.2
	 * @return string
	 */
	public static function plugin_path() {
		if ( self::$plugin_path ) {
			return self::$plugin_path;
		}
		
		return self::$plugin_path = untrailingslashit( plugin_dir_path( __FILE__ ) );
	}
	
	/**
	 * Detect if WC Subscriptions is active
	 *
	 * @since 1.2
	 * @return bool True if active, False if not
	 */
	public static function is_subscriptions_active() {
		if ( is_bool( self::$is_subscriptions_active ) ) {
			return self::$is_subscriptions_active;
		}
		
		self::$is_subscriptions_active = false;
		if ( class_exists( 'WC_Subscriptions' ) || function_exists( 'wcs_order_contains_subscription' ) ) {
			self::$is_subscriptions_active = true;
		}
		
		return self::$is_subscriptions_active;
	}
	
	/**
	 * Get back the Subsctiptions version.
	 *
	 * @since 1.4
	 * @return bool Main Subscriptions version number (e.i. 1, 2, 3), False, if Subscriptions is not active
	 */
	public static function get_subscriptions_version() {
		if ( null !== self::$is_subscriptions_version ) {
			return self::$is_subscriptions_version;
		}
		
		self::$is_subscriptions_version = false;
		
		if ( function_exists( 'wcs_order_contains_subscription' ) ) {
			self::$is_subscriptions_version = 2;
		} elseif ( class_exists( 'WC_Subscriptions' ) ) {
			self::$is_subscriptions_version = 1;
		}
		
		return self::$is_subscriptions_version;
	}
	
	/**
	 * Detect if Pre-Orders is active
	 *
	 * @since 1.2
	 * @return bool True if actiove, False if not
	 */
	public static function is_pre_orders_active() {
		if ( is_bool( self::$is_pre_orders_active ) ) {
			return self::$is_pre_orders_active;
		}
		
		if ( class_exists( 'WC_Pre_Orders' ) ) {
			self::$is_pre_orders_active = true;
		} else {
			self::$is_pre_orders_active = false;
		}
		
		return self::$is_pre_orders_active;
	}
	
	/**
	 * Verify a form request action
	 *
	 * @since 1.2
	 *
	 * @param string $name
	 * @param string $action
	 *
	 * @throws Exception
	 */
	public static function verify_request( $name, $action ) {
		
		if ( ! wp_verify_nonce( self::get_post( $name ), $action ) ) {
			throw new Exception( __( 'Cannot verify the request, please try again.', self::TEXT_DOMAIN ) );
		}
	}
	
	/**
	 * Return the order number with stripped # or n° ( french translations )
	 *
	 * @since 1.2
	 *
	 * @param WC_Order $order
	 *
	 * @return string
	 */
	public static function get_order_number( WC_Order $order ) {
		return str_replace( array( '#', 'n°' ), '', $order->get_order_number() );
	}
	
	/**
	 * Loads plugin scripts
	 *
	 * @since 2.0
	 */
	public function load_scripts() {
		// Add needed scripts
		$scripts = new WC_Paytrace_Scripts();
		$scripts->hooks();
	}
	
	/**
	 * Loads Ajax processes
	 */
	public function load_ajax() {
		$ajax = new WC_Paytrace_Ajax();
		$ajax->hooks();
	}
	
	/**
	 * Loads Admin processes
	 */
	public function load_admin() {
		$admin = new \WcPaytrace\Admin\Admin();
	}
	
	/**
	 * Loads the manage cards functionality
	 * @since 2.0
	 */
	public function load_manage_cards() {
		$cards = new WC_Paytrace_Manage_Cards( $this );
		$cards->hooks();
	}
}

/**
 * Load the plugin main class
 */
add_action( 'plugins_loaded', 'wc_paytrace_load_plugin', 10 );
function wc_paytrace_load_plugin() {
	new WC_Paytrace();
}