<?php

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Wrapper for the Paytrace/Tokens.
 * Unifies the tokens and profiles, so we can get the information in a consistent manner.
 *
 * @since  2.0
 * @author VanboDevelops
 *
 *        Copyright: (c) 2017 VanboDevelops
 *        License: GNU General Public License v3.0
 *        License URI: http://www.gnu.org/licenses/gpl-3.0.html
 */
class WC_Paytrace_Token {
	
	/**
	 * @var WC_Payment_Token_Paytrace_CC|WC_Payment_Token_Paytrace_eCheck|WC_Payment_Token
	 */
	protected $profile;
	/**
	 * @var int The ID of the legacy profile
	 */
	protected $id;
	/**
	 * Is it a WC_Payment_Token
	 * @var bool
	 */
	protected $is_wc_token = false;
	protected $allowed_token_types = array(
		'Paytrace_eCheck',
		'Paytrace_CC',
	);
	
	/**
	 * Constructor
	 *
	 * @param WC_Payment_Token|array $token
	 * @param null|int               $id The ID is only passed when we load legacy token or we load token wrapper. Don't pass ID, if you load a WC Token
	 */
	public function __construct( $token, $id = null ) {
		$this->set_profile( $token );
		
		if ( class_exists( 'WC_Payment_Token' ) ) {
			$this->is_wc_token = $this->profile instanceof WC_Payment_Token && null === $id;
		}
		
		if ( null !== $id ) {
			$this->set_id( (int) $id );
		}
	}

	/**
	 * Returns true, if the token is WC_Payment_Token
	 *
	 * @since 2.0
	 *
	 * @return bool
	 */
	public function is_wc_token() {
		return $this->is_wc_token;
	}

	/**
	 * Show the token display name
	 *
	 * @since 2.1
	 *
	 * @return string
	 */
	public function get_display_name() {
		if ( $this->is_wc_token ) {
			return $this->profile->get_display_name();
		}

		if ( 'Paytrace_eCheck' == $this->get_type() ) {
			$name = sprintf(
				__( 'eCheck ending in %1$s', WC_Paytrace::TEXT_DOMAIN ),
				$this->get_last4()
			);
		} else {
			$expiry_year = $this->get_expiry_year();
			if ( strlen( $expiry_year ) == 4 ) {
				$expiry_year -= 2000;
			}
			$name = sprintf(
				__( 'Card ending in %1$s (expires %2$s/%3$s)', WC_Paytrace::TEXT_DOMAIN ),
				$this->get_last4(),
				$this->get_expiry_month(),
				$expiry_year
			);
		}

		return $name;
	}
	
	/**
	 * Returns the token type
	 *
	 * @since 2.0
	 *
	 * @return string
	 */
	public function get_type() {
		if ( $this->is_wc_token ) {
			return $this->profile->get_type();
		}
		
		// If we set the type on the object load
		if ( isset( $this->profile['type'] ) && in_array( $this->profile['type'], $this->allowed_token_types ) ) {
			return $this->profile['type'];
		}
		
		// Check for the eCheck data to determine if we have an eCheck
		if ( ( isset( $this->profile['tr_last4'] ) && isset( $this->profile['dda_last4'] ) )
		     || isset( $this->profile['routing_last4'] )
		) {
			return 'Paytrace_eCheck';
		}
		
		return 'Paytrace_CC';
	}
	
	/**
	 * Sets the profile
	 *
	 * @since 2.0
	 *
	 * @param WC_Payment_Token_Paytrace_CC|WC_Payment_Token_Paytrace_eCheck|WC_Payment_Token $token
	 */
	public function set_profile( $token ) {
		$this->profile = $token;
	}
	
	/**
	 * Returns the token/profile ID
	 *
	 * @since 2.0
	 *
	 * @return int
	 */
	public function get_id() {
		if ( $this->is_wc_token ) {
			return $this->profile->get_id();
		}
		
		return $this->id;
	}
	
	/**
	 * Sets the ID for the legacy profile
	 *
	 * @since 2.0
	 *
	 * @param $id
	 */
	public function set_id( $id ) {
		$this->id = $id;
	}
	
	/**
	 * Returns the Expiry month
	 *
	 * @since 2.0
	 *
	 * @return string
	 */
	public function get_expiry_month() {
		if ( $this->is_wc_token ) {
			return $this->profile->get_expiry_month();
		}
		
		$return = '';
		if ( isset( $this->profile['exp_month'] ) ) {
			$return = $this->profile['exp_month'];
		} elseif ( isset( $this->profile['expiry_month'] ) ) {
			$return = $this->profile['expiry_month'];
		}
		
		return $return;
	}
	
	/**
	 * Returns the Expiry year
	 *
	 * @since 2.0
	 *
	 * @return int
	 */
	public function get_expiry_year() {
		if ( $this->is_wc_token ) {
			$expiry_year = $this->profile->get_expiry_year();
			
			// We need the year as YY, so if it is YYYY, trim it
			if ( strlen( $expiry_year ) == 4 ) {
				$expiry_year -= 2000;
			}
			
			return $expiry_year;
		}
		
		$return = '';
		if ( isset( $this->profile['exp_year'] ) ) {
			$return = $this->profile['exp_year'];
		} elseif ( isset( $this->profile['expiry_year'] ) ) {
			$return = $this->profile['expiry_year'];
		}
		
		return $return;
	}
	
	/**
	 * Returns the user id
	 *
	 * @since 2.0
	 *
	 * @return string
	 */
	public function get_user_id() {
		if ( $this->is_wc_token ) {
			return $this->profile->get_user_id();
		}
		
		return isset( $this->profile['user_id'] ) ? $this->profile['user_id'] : 0;
	}
	
	/**
	 * Returns the gateway ID 'paytrace'
	 *
	 * @since 2.0
	 *
	 * @return string
	 */
	public function get_gateway_id() {
		if ( $this->is_wc_token ) {
			return $this->profile->get_gateway_id();
		}
		
		return isset( $this->profile['gateway_id'] ) ? $this->profile['gateway_id'] : 'paytrace';
	}
	
	/**
	 * Returns the card type
	 *
	 * @since 2.0
	 *
	 * @return string
	 */
	public function get_card_type() {
		if ( $this->is_wc_token ) {
			return $this->profile->get_card_type();
		}
		
		return isset( $this->profile['card_type'] ) ? $this->profile['card_type'] : '';
	}
	
	/**
	 * Returns the last 4 digits of the card
	 *
	 * @since 2.0
	 *
	 * @return string
	 */
	public function get_last4() {
		if ( $this->is_wc_token ) {
			return $this->profile->get_last4();
		}
		
		$last4 = '';
		if ( isset( $this->profile['dda_last4'] ) ) {
			$last4 = $this->profile['dda_last4'];
		} elseif ( isset( $this->profile['last4'] ) ) {
			$last4 = $this->profile['last4'];
		}
		
		return $last4;
	}
	
	/**
	 * Returns the last4 of the routing number
	 *
	 * @since 2.0
	 *
	 * @return mixed
	 */
	public function get_routing_last4() {
		if ( $this->is_wc_token ) {
			return $this->profile->get_routing_last4();
		}
		
		$routing = '';
		if ( isset( $this->profile['tr_last4'] ) ) {
			$routing = $this->profile['tr_last4'];
		} elseif ( isset( $this->profile['routing_last4'] ) ) {
			$routing = $this->profile['routing_last4'];
		}
		
		return $routing;
	}
	
	/**
	 * Returns the last4 of the routing number
	 *
	 * @since 2.0
	 *
	 * @return mixed
	 */
	public function get_profile_id() {
		if ( $this->is_wc_token ) {
			return $this->profile->get_profile_id();
		}
		
		return $this->profile['profile_id'];
	}
	
	/**
	 * Returns the billing ID or the token
	 *
	 * @since 2.0
	 *
	 * @return string
	 */
	public function get_token() {
		if ( $this->is_wc_token ) {
			return $this->profile->get_token();
		}
		
		$return = '';
		if ( isset( $this->profile['customer_id'] ) ) {
			$return = $this->profile['customer_id'];
		} elseif ( isset( $this->profile['token'] ) ) {
			$return = $this->profile['token'];
		}
		
		return $return;
	}
	
	/**
	 * Returns whether the card is default method
	 *
	 * @since 2.0
	 *
	 * @return bool
	 */
	public function get_is_default() {
		if ( $this->is_wc_token ) {
			return $this->profile->get_is_default();
		}
		
		return isset( $this->profile['is_default'] ) ? $this->profile['is_default'] : false;
	}
	
	/**
	 * Delete the WC_Payment_Token
	 *
	 * @since 2.0
	 *
	 * @return bool
	 */
	public function delete() {
		if ( $this->is_wc_token ) {
			return $this->profile->delete();
		}
		
		return true;
	}
}