<?php

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Handles the Paytrace Customer Tokens.
 * Saves the Paytrace Tokens if WC > 2.6, or saves user meta profiles, if WC < 2.6
 *
 * @since  2.0
 * @author VanboDevelops
 *
 *        Copyright: (c) 2017 VanboDevelops
 *        License: GNU General Public License v3.0
 *        License URI: http://www.gnu.org/licenses/gpl-3.0.html
 */
class WC_Paytrace_Order {
	
	/**
	 * WC_Paytrace_Order constructor.
	 *
	 * @param $order
	 */
	public function __construct( \WC_Order $order ) {
		$this->order      = $order;
		$this->gateway_id = 'paytrace';
	}
	
	/**---------------------------------
	 * GETTERS
	 * -----------------------------------*/
	
	/**
	 * Returns the Paytrace billing id
	 *
	 * @since 2.0
	 *
	 * @return mixed
	 */
	public function get_customer_id() {
		return \WC_Paytrace_Compat::get_meta( $this->order, '_paytrace_customer_id', true );
	}
	
	/**
	 * Returns the Paytrace payment Type
	 *
	 * @since 2.0
	 *
	 * @return mixed
	 */
	public function get_payment_type() {
		return \WC_Paytrace_Compat::get_meta( $this->order, '_paytrace_payment_type', true );
	}
	
	/**
	 * Returns the Paytrace transaction id
	 *
	 * @since 2.0
	 *
	 * @return string
	 */
	public function get_transaction_id() {
		return \WC_Paytrace_Compat::get_meta( $this->order, '_paytrace_transaction_id', true );
	}
	
	/**
	 * Returns the payment captured data
	 *
	 * @since 2.0
	 *
	 * @return string
	 */
	public function get_is_payment_captured() {
		return \WC_Paytrace_Compat::get_meta( $this->order, '_wc_paytrace_is_payment_captured', true );
	}
	
	/**
	 * Returns the amount captured
	 *
	 * @since 2.0
	 *
	 * @return float
	 */
	public function get_order_amount_captured() {
		return \WC_Paytrace_Compat::get_meta( $this->order, '_wc_paytrace_order_amount_captured', true );
	}
	
	/**
	 * Returns the amount authorized in the transaction
	 *
	 * @since 2.0
	 *
	 * @return float
	 */
	public function get_order_amount_authorized() {
		return \WC_Paytrace_Compat::get_meta( $this->order, '_wc_paytrace_order_amount_authorized', true );
		
	}
	
	/**---------------------------------------------------
	 * CREATE
	 * ---------------------------------------------------*/
	
	/**
	 * Saves Paytrace billing id to the order
	 *
	 * @since 2.0
	 *
	 * @param string $customer_id
	 *
	 * @return bool|int
	 */
	public function save_customer_id( $customer_id ) {
		return WC_Paytrace_Compat::update_meta( $this->order, '_paytrace_customer_id', wc_clean( $customer_id ), true );
	}
	
	/**
	 * Saves order payment type to the order
	 *
	 * @since 2.0
	 *
	 * @param string $type
	 *
	 * @return bool|int
	 */
	public function save_payment_type( $type ) {
		return WC_Paytrace_Compat::update_meta( $this->order, '_paytrace_payment_type', wc_clean( $type ), true );
	}
	
	/**
	 * Saves Paytrace transaction id to the order
	 *
	 * @since 2.0
	 *
	 * @param string $id
	 *
	 * @return bool|int
	 */
	public function save_transaction_id( $id ) {
		return WC_Paytrace_Compat::update_meta( $this->order, '_paytrace_transaction_id', wc_clean( $id ), true );
	}
	
	/**
	 * Marks the order the amount captured
	 *
	 * @since 2.0
	 *
	 * @param bool $is_captured
	 */
	public function save_is_payment_captured( $is_captured = false ) {
		WC_Paytrace_Compat::update_meta( $this->order, '_wc_paytrace_is_payment_captured', wc_clean( $is_captured ), true );
	}
	
	/**
	 * Marks the order payment as captured or not
	 *
	 * @since 2.0
	 *
	 * @param bool $amount (optional) If not present the order total will be saved
	 */
	public function save_order_amount_captured( $amount = false ) {
		if ( false === $amount ) {
			$amount = $this->order->get_total();
		}
		
		WC_Paytrace_Compat::update_meta( $this->order, '_wc_paytrace_order_amount_captured', wc_clean( $amount ), true );
	}
	
	/**
	 * Saves the amount authorized in the transaction
	 *
	 * @since 2.0
	 *
	 * @param bool $amount
	 */
	public function save_order_amount_authorized( $amount = false ) {
		if ( false === $amount ) {
			$amount = $this->order->get_total();
		}
		
		WC_Paytrace_Compat::update_meta( $this->order, '_wc_paytrace_order_amount_authorized', wc_clean( $amount ), true );
	}
	
	/**---------------------------------------------------
	 * DELETE
	 * ---------------------------------------------------*/
	
	/**
	 * Deletes the billing ID from the order
	 *
	 * @since 2.0
	 *
	 * @return bool|int
	 */
	public function delete_customer_id() {
		return WC_Paytrace_Compat::delete_meta( $this->order, '_paytrace_customer_id' );
	}
	
	/**
	 * Deletes the is payment captured mark
	 *
	 * @since 2.0
	 */
	public function delete_is_payment_captured() {
		return WC_Paytrace_Compat::delete_meta( $this->order, '_wc_paytrace_is_payment_captured' );
	}
	
	/**
	 * Deletes the amount captured value
	 *
	 * @since 2.0
	 */
	public function delete_order_amount_captured() {
		return WC_Paytrace_Compat::delete_meta( $this->order, '_wc_paytrace_order_amount_captured' );
	}
	
	/**
	 * Deletes the amount authorized meta
	 *
	 * @since 2.0
	 *
	 * @return bool|int
	 */
	public function delete_order_amount_authorized() {
		return WC_Paytrace_Compat::delete_meta( $this->order, '_wc_paytrace_order_amount_authorized' );
	}
	
	/**---------------------------------------------------
	 * Functional checks
	 * ---------------------------------------------------*/
	
	/**
	 * @since 2.4.1
	 * @return bool
	 */
	public function is_subscription() {
		if ( ! WC_Paytrace::is_subscriptions_active() ) {
			return false;
		}
		
		return wcs_is_subscription( $this->order );
	}
	
	/**
	 * @since 2.4.1
	 * @return bool
	 */
	public function contains_subscription() {
		if ( ! WC_Paytrace::is_subscriptions_active() ) {
			return false;
		}
		
		return wcs_order_contains_subscription( $this->order )
		       || wcs_order_contains_renewal( $this->order );
	}
	
	/**
	 * Returns true, if order contains Pre-Order
	 *
	 * @since 2.4.1
	 *
	 * @return bool
	 */
	public function contains_pre_order() {
		if ( ! WC_Paytrace::is_pre_orders_active() ) {
			return false;
		}
		
		return WC_Pre_Orders_Order::order_contains_pre_order( $this->order );
	}
	
	/**
	 * Returns true if the order is a pre-order and it requires tokenization(charged at release)
	 *
	 * @since 2.4.1
	 *
	 * @return bool
	 */
	public function is_pre_order_with_tokenization() {
		if ( ! WC_Paytrace::is_pre_orders_active() ) {
			return false;
		}
		
		return \WC_Pre_Orders_Order::order_contains_pre_order( $this->order )
		       && \WC_Pre_Orders_Order::order_requires_payment_tokenization( $this->order );
	}
}