<?php

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Handles the Paytrace Customer Tokens.
 * Saves the WC Tokens if WC > 2.6, or saves user meta profiles, if WC < 2.6
 *
 * @since  2.0
 * @author VanboDevelops
 *
 *        Copyright: (c) 2017 VanboDevelops
 *        License: GNU General Public License v3.0
 *        License URI: http://www.gnu.org/licenses/gpl-3.0.html
 */
class WC_Paytrace_Manage_Cards {
	
	/**
	 * @var WC_Paytrace
	 */
	protected $plugin;
	public static $running_customer_tokens_transfer = false;
	
	/**
	 * WC_Paytrace_Manage_Cards constructor.
	 *
	 * @param WC_Paytrace $plugin
	 */
	public function __construct( WC_Paytrace $plugin ) {
		$this->plugin = $plugin;
	}
	
	/**
	 * Returns the manage tokens URL for the plugin
	 *
	 * @since 2.0
	 *
	 * @param string $type
	 *
	 * @return string
	 */
	public static function manage_tokens_url( $type = 'card' ) {
		
		if ( \WC_Paytrace_Compat::is_wc_2_6() ) {
			return wc_get_account_endpoint_url( 'payment-methods' );
		}
		
		if ( 'check' == $type ) {
			return wc_get_page_permalink( 'myaccount' ) . '#paytrace-saved-checks';
		}
		
		return wc_get_page_permalink( 'myaccount' ) . '#paytrace-saved-cards';
	}
	
	/**
	 * Loads the class hooks
	 *
	 * @since 2.0
	 */
	public function hooks() {
		if ( WC_Paytrace_Compat::is_wc_2_6() ) {
			/**
			 * WC > 2.6 Tokens features
			 */
			add_filter( 'woocommerce_payment_methods_list_item', array(
				$this,
				'wc_get_account_saved_payment_methods_list_item'
			), 10, 2 );
			
			// Initially, check for the customer legacy profiles and transfer them to tokens
			add_filter( 'woocommerce_get_customer_payment_tokens', array(
				$this,
				'maybe_transfer_legacy_profiles_to_tokens'
			), 10, 3 );
			
			// Delete the payment method before
			add_action( 'woocommerce_paytrace_payment_token_delete', array(
				$this,
				'delete_payment_token'
			), 10, 2 );
			
			if ( WC_Paytrace_Compat::is_wc_3_1_2() ) {
				add_filter( 'woocommerce_credit_card_type_labels', array(
					$this,
					'add_payment_options_labels'
				) );
			} else {
				add_filter( 'wocommerce_credit_card_type_labels', array(
					$this,
					'add_payment_options_labels'
				) );
			}
		} else {
			// Manage Saved Cards
			add_action( 'template_redirect', array( $this, 'manage_saved_cards' ) );
			
			add_action( 'woocommerce_after_my_account', array( $this, 'display_saved_cards' ) );
		}
	}
	
	/**
	 * Checks for and transfers the customer legacy profiles to WC Payment Tokens
	 *
	 * @since 2.0
	 *
	 * @param $tokens
	 * @param $customer_id
	 * @param $gateway_id
	 *
	 * @return array|mixed
	 */
	public function maybe_transfer_legacy_profiles_to_tokens( $tokens, $customer_id, $gateway_id ) {
		// Bail, if we already transferred the profiles
		$transferred_legacy_profiles = get_user_meta( $customer_id, '_paytrace_legacy_profiles_transferred', true );
		if ( $transferred_legacy_profiles || true === WC_Paytrace_Customer_Tokens::$is_transfer_legacy_profiles_running ) {
			return $tokens;
		}
		
		new WC_Paytrace_Customer_Tokens( $customer_id );
		
		// After we transferred the profiles just get the tokens again
		$tokens = WC_Payment_Tokens::get_tokens( array(
			'user_id'    => $customer_id,
			'gateway_id' => $gateway_id,
		) );
		
		return $tokens;
	}
	
	/**
	 * Detect and process customer update and delete profile requests
	 *
	 * @since 2.0
	 *
	 * @return bool
	 */
	public function manage_saved_cards() {
		
		if ( ! is_user_logged_in() ) {
			return false;
		}
		
		// No need to continue, if an update is not submitted
		if ( null == WC_Paytrace::get_post( 'paytrace_delete_token' )
		     && null == WC_Paytrace::get_post( 'paytrace_update_card' )
		     && null == WC_Paytrace::get_post( 'paytrace_update_check' )
		) {
			return false;
		}
		
		/**
		 * @var \WC_Payment_Gateway|\WC_Paytrace_Gateway_Addons $gateway
		 */
		$gateway = \WcPaytrace\Helpers\Factories::get_gateway( 'paytrace' );
		
		try {
			if ( null != WC_Paytrace::get_post( 'paytrace_delete_token' ) ) {
				// First verify the request was send from the manage card form
				WC_Paytrace::verify_request( 'paytrace-nonce', 'paytrace_delete_token' );
				
				$gateway->delete_customer_profile( (int) WC_Paytrace::get_post( 'paytrace_delete_token' ) );
				
				wc_add_notice( __( 'Token deleted successfully', WC_Paytrace::TEXT_DOMAIN ), 'success' );
			}
			
			if ( null != WC_Paytrace::get_post( 'paytrace_update_card' ) ) {
				if ( WC_Paytrace::get_post( 'paytrace-card-number' ) &&
				     WC_Paytrace::get_post( 'paytrace-card-expiry' ) &&
				     WC_Paytrace::get_post( 'paytrace-name-on-card' )
				) {
					// First verify the request was send from the manage card form
					WC_Paytrace::verify_request( 'paytrace-nonce', 'paytrace-update-card' );
					
					$gateway->update_customer_profile( (int) WC_Paytrace::get_post( 'paytrace_update_card' ) );
					
					wc_add_notice( __( 'Card updated successfully', WC_Paytrace::TEXT_DOMAIN ), 'success' );
				}
			} elseif ( null != WC_Paytrace::get_post( 'paytrace_update_check' ) ) {
				
				WC_Paytrace::add_debug_log( 'Updating check: ' . WC_Paytrace::get_post( 'paytrace_update_check' ) );
				
				if ( WC_Paytrace::get_post( 'paytrace-echeck-routing-number' ) &&
				     WC_Paytrace::get_post( 'paytrace-echeck-account-number' ) &&
				     WC_Paytrace::get_post( 'paytrace-name-on-echeck' )
				) {
					// First verify the request was send from the manage card form
					WC_Paytrace::verify_request( 'paytrace-nonce', 'paytrace-update-check' );
					
					WC_Paytrace::add_debug_log( 'Run the update profile: ' );
					
					$gateway->update_customer_profile( (int) WC_Paytrace::get_post( 'paytrace_update_check' ) );
					
					wc_add_notice( __( 'Check updated successfully', WC_Paytrace::TEXT_DOMAIN ), 'success' );
				}
			}
		}
		catch ( Exception $e ) {
			wc_add_notice( $e->getMessage(), 'error' );
		}
	}
	
	/**
	 * Delete the Paytrace Profile after we delete a token
	 *
	 * @since 2.0
	 *
	 * @param int                                                           $token_id
	 * @param WC_Payment_Token_Paytrace_CC|WC_Payment_Token_Paytrace_eCheck $token
	 *
	 * @throws Exception
	 */
	public function delete_payment_token( $token_id, $token ) {
		if ( 'paytrace' === $token->get_gateway_id() ) {
			/**
			 * @var \WC_Payment_Gateway|\WC_Paytrace_Gateway_Addons $gateway
			 */
			$gateway = \WcPaytrace\Helpers\Factories::get_gateway( 'paytrace' );
			
			try {
				// Delete profile
				$gateway->delete_customer_profile( $token->get_id(), $token->get_user_id() );
			}
			catch ( Exception $e ) {
				if ( function_exists( 'wc_add_notice' ) ) {
					wc_add_notice( $e->getMessage(), 'error' );
				}
			}
		}
	}
	
	/**
	 * Controls the output for credit cards on the my account page.
	 *
	 * @since 2.0
	 *
	 * @param  array             $item          Individual list item from woocommerce_saved_payment_methods_list
	 * @param  WC_Paytrace_Token $payment_token The payment token associated with this method entry
	 *
	 * @return array                           Filtered item
	 */
	function wc_get_account_saved_payment_methods_list_item( $item, $payment_token ) {
		if ( 'paytrace' !== $payment_token->get_gateway_id() ) {
			return $item;
		}
		
		if ( 'Paytrace_eCheck' !== $payment_token->get_type() ) {
			$card_type               = $payment_token->get_card_type();
			$item['method']['brand'] = ( ! empty( $card_type ) ? ucfirst( $card_type ) : esc_html__( 'Card', 'woocommerce' ) );
			$item['expires']         = $payment_token->get_expiry_month() . '/' . substr( $payment_token->get_expiry_year(), - 2 );
		} else {
			$item['method']['brand']   = esc_html__( 'eCheck', 'woocommerce' );
			$item['expires'] = '';
		}
		
		$item['method']['last4'] = $payment_token->get_last4();
		
		return $item;
	}
	
	/**
	 * Manage the customer saved credit cards
	 *
	 * @since 2.0
	 */
	public function display_saved_cards() {
		
		// Customer should be logged in
		if ( ! is_user_logged_in() ) {
			return;
		}
		
		/**
		 * @var \WC_Payment_Gateway|\WC_Paytrace_Gateway_Addons $gateway
		 */
		$gateway = \WcPaytrace\Helpers\Factories::get_gateway( 'paytrace' );
		
		// Don't show if the gateway is not available for use
		if ( ! $gateway->is_available() ) {
			return;
		}
		
		try {
			$customer_tokens = new WC_Paytrace_Customer_Tokens( get_current_user_id() );
			$tokens          = $customer_tokens->get_tokens();
			
			// Don't show if there are not profiles or the save customers is not enabled
			if ( empty( $tokens ) || false == $gateway->save_customers ) {
				return;
			}
			
			// Don't show if there are not profiles or the save customers is not enabled
			if ( 'no' == $gateway->get_option( $gateway->get_option( 'integration' ) . '_save_customers' ) ) {
				return;
			}
			
			wp_enqueue_script( 'paytrace-my-account' );
			
			$saved_cards  = array();
			$saved_checks = array();
			/**
			 * @var \WC_Paytrace_Token $token
			 */
			foreach ( $tokens as $n => $token ) {
				if ( 'Paytrace_eCheck' == $token->get_type() ) {
					$saved_checks[ $n ] = $token;
				} else {
					$saved_cards[ $n ] = $token;
				}
			}
			
			if ( $saved_cards ) {
				wc_get_template(
					'myaccount/myaccount-paytrace-saved-cards.php',
					array(
						'gateway'      => $gateway,
						'gateway_id'   => $gateway->id,
						'saved_cards'  => $saved_cards,
						'available_cc' => $gateway->available_cc,
						'card_options' => $gateway->card_options,
					),
					'',
					\WC_Paytrace::plugin_path() . '/templates/'
				);
			}
			
			if ( $saved_checks ) {
				wc_get_template(
					'myaccount/myaccount-paytrace-saved-checks.php',
					array(
						'saved_checks' => $saved_checks,
						'gateway_id'   => $gateway->id,
					),
					'',
					\WC_Paytrace::plugin_path() . '/templates/'
				);
			}
		}
		catch ( Exception $e ) {
		}
	}
	
	/**
	 * Adds the Paytrace card types to the WC card types
	 *
	 * @since 2.0
	 *
	 * @param $labels
	 *
	 * @return mixed
	 */
	public function add_payment_options_labels( $labels ) {
		/**
		 * @var \WC_Payment_Gateway|\WC_Paytrace_Gateway_Addons $gateway
		 */
		$gateway = \WcPaytrace\Helpers\Factories::get_gateway( 'paytrace' );
		
		foreach ( $gateway->card_options as $key => $string ) {
			if ( ! isset( $labels[ strtolower( $key ) ] ) ) {
				$labels[ strtolower( $key ) ] = $string;
			}
		}
		
		return $labels;
	}
}