<?php

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Handles the Paytrace Customer Tokens.
 * Saves the WC Tokens if WC > 2.6, or saves user meta profiles, if WC < 2.6
 *
 * @since  2.0
 * @author VanboDevelops
 *
 *        Copyright: (c) 2017 VanboDevelops
 *        License: GNU General Public License v3.0
 *        License URI: http://www.gnu.org/licenses/gpl-3.0.html
 */
class WC_Paytrace_Customer_Tokens {
	
	public static $is_transfer_legacy_profiles_running = false;
	public $gateway_id = 'paytrace';
	public $user_id;
	
	public function __construct( $user_id ) {
		$this->user_id = (int) $user_id;
		
		$this->maybe_transfer_legacy_profiles();
	}
	
	/**---------------------------------
	 * GETTERS
	 * -----------------------------------*/
	
	/**
	 * Returns the saved Paytrace Tokens/Profiles for the Customer
	 *
	 * @since 2.0
	 *
	 * @return array|mixed
	 */
	public function get_tokens() {
		if ( WC_Paytrace_Compat::is_wc_2_6() ) {
			$tokens = $this->get_customer_tokens();
		} else {
			$tokens = $this->get_user_saved_profiles();
		}
		
		return $tokens;
	}
	
	/**
	 * Returns the specific token/profile by the provided token ID.
	 *
	 * @since 2.0
	 *
	 * @param $token_id
	 *
	 * @throws Exception
	 *
	 * @return WC_Paytrace_Token|WC_Payment_Token_Paytrace_CC|WC_Payment_Token_Paytrace_eCheck
	 */
	public function get_token( $token_id ) {
		$tokens = $this->get_tokens();
		
		/**
		 * @var WC_Paytrace_Token $token
		 */
		foreach ( $tokens as $id => $token ) {
			if ( $token_id == $token->get_id() ) {
				return $tokens[ $id ];
			}
		}
		
		throw new Exception( __( 'The Token was not found.', WC_Paytrace::TEXT_DOMAIN ) );
	}
	
	/**
	 * Returns the token by the provided customer ID or token (for the WC_Payment_Token)
	 *
	 * @since 2.0
	 *
	 * @param string $customer_id
	 *
	 * @return WC_Paytrace_Token|false
	 */
	public function get_token_by_customer_id( $customer_id ) {
		$saved_cards = $this->get_tokens();
		
		/**
		 * @var WC_Paytrace_Token $token
		 */
		foreach ( $saved_cards as $id => $token ) {
			if ( $token->get_token() == $customer_id ) {
				return $saved_cards[ $id ];
			}
		}
		
		return false;
	}
	
	/**
	 * Pings the Paytrace vault to find the Customer ID and creates a WC Token from the Vault record
	 *
	 * @since 2.4.3
	 *
	 * @param string                                         $vault_customer_id
	 * @param \WcPaytrace\Integrations\Json\Integration_Json $integration
	 *
	 * @throws Exception
	 * @return int|false The ID of the created WC Token
	 */
	public function create_wc_token_from_paytrace_customer_id( $vault_customer_id, $integration ) {
		// We can do this only on the JSON API
		if ( 'json' != $integration->get_gateway()->get_option( 'integration' ) ) {
			throw new Exception( __( 'A Paytrace Vault profile export can only be performed with the JSON API.', WC_Paytrace::TEXT_DOMAIN ) );
		}
		
		// Get the profile from the Vault
		$lookup = $integration->export_profile( $vault_customer_id );
		
		// Error if there is no profile in the Vault, what are we charging then and why are we saving this value
		if ( ! is_array( $lookup ) || empty( $lookup ) ) {
			throw new Exception( __( 'Could not find the provided customer ID in the Paytrace Vault. Please double check that the "PayTrace Customer Profile ID" exists in the Paytrace Vault.', WC_Paytrace::TEXT_DOMAIN ) );
		}
		
		// Take the first customer returned, it should be only one anyway since the vault_customer_id is unique
		$looked_customer = array_shift( $lookup );
		
		if ( isset( $looked_customer->check ) ) {
			$check = WC_Paytrace::get_field( 'check', $looked_customer, false );
			
			// If the customer ID is for a check and there is no Check values ... can't charge it
			if ( ! $check ) {
				throw new Exception( __( 'Failed to generate WC Token for the Check in the "PayTrace Customer Profile ID" provided. Please double check that the "PayTrace Customer Profile ID" exists in the Paytrace system.', WC_Paytrace::TEXT_DOMAIN ) );
			}
			
			$profile = array(
				'routing_last4' => substr( WC_Paytrace::get_field( 'routing_number', $check, '' ), - 4 ),
				'last4'         => substr( WC_Paytrace::get_field( 'account_number', $check, '' ), - 4 ),
				'type'          => 'Paytrace_eCheck',
			);
		} else {
			$card = WC_Paytrace::get_field( 'credit_card', $looked_customer, false );
			
			// No card means there is nothing saved to the profile that we can charge
			if ( ! $card ) {
				throw new Exception( __( 'Failed to generate WC Token for the Card in the "PayTrace Customer Profile ID" provided. Please double check that the "PayTrace Customer Profile ID" exists in the Paytrace system.', WC_Paytrace::TEXT_DOMAIN ) );
			}
			
			$services  = $integration->get_api_services();
			$validator = $services->get_validator();
			
			$last4        = substr( WC_Paytrace::get_field( 'masked_number', $card, '' ), - 4 );
			$card_type    = $validator->get_card_type_from_first_six( WC_Paytrace::get_field( 'masked_number', $card, '' ) );
			$expiry_month = WC_Paytrace::get_field( 'expiration_month', $card, '' );
			$expiry_year  = WC_Paytrace::get_field( 'expiration_year', $card, '' );
			
			$profile = array(
				'last4'     => $last4,
				'exp_year'  => $expiry_year,
				'exp_month' => $expiry_month,
				'type'      => 'Paytrace_CC',
				'card_type' => $card_type,
			);
		}
		
		// Create a Paytrace Token to save
		$profile['token']      = $vault_customer_id;
		$profile['profile_id'] = $vault_customer_id;
		$new_token             = new \WC_Paytrace_Token( $profile, 0 );
		
		// Save the token and return its ID
		return $this->save_profile_or_token( $new_token );
	}
	
	/**
	 * Returns the WC Customer tokens for the Paytrace gateway
	 *
	 * @since 2.0
	 *
	 * @return array
	 */
	public function get_customer_tokens() {
		$tokens    = WC_Payment_Tokens::get_customer_tokens( $this->user_id, $this->gateway_id );
		$formatted = array();
		
		// Format the tokens with a wrapper
		foreach ( $tokens as $id => $token ) {
			$formatted[ $id ] = new WC_Paytrace_Token( $token );
		}
		
		return $formatted;
	}
	
	/**
	 * Returns all profiles saved to the user meta
	 *
	 * @since 2.0
	 *
	 * @return mixed
	 */
	public function get_user_saved_profiles() {
		$profiles  = get_user_meta( $this->user_id, '_paytrace_saved_profiles', false );
		$formatted = array();
		
		// Format the profiles with a wrapper, so we can use them as WC tokens
		foreach ( $profiles as $id => $profile ) {
			$formatted[ $id ] = new WC_Paytrace_Token( $profile, $id );
		}
		
		return $formatted;
	}
	
	/**
	 * Returns the Paytrace customer id
	 *
	 * @since 2.0
	 *
	 * @param \WC_Order $order
	 *
	 * @return mixed
	 */
	public function get_customer_id_from_order( \WC_Order $order ) {
		return \WC_Paytrace_Compat::get_meta( $order, '_paytrace_customer_id', true );
		
	}
	
	/**---------------------------------------------------
	 * Create method for creating new Profiles/Tokens
	 * ---------------------------------------------------*/
	
	/**
	 * Saves the Paytrace Token/Profile to the customer.
	 *
	 * @since 2.0
	 *
	 * @param \WC_Paytrace_Token $new_token
	 *
	 * @throws Exception
	 * @return bool|false|int
	 */
	public function save_profile_or_token( \WC_Paytrace_Token $new_token ) {
		if ( WC_Paytrace_Compat::is_wc_2_6() ) {
			return $this->create_wc_token( $new_token );
		}
		
		return $this->create_legacy_profile( $new_token );
	}
	
	/**
	 * Creates a WC Credit card token, saved to the customer, using the passed data.
	 *
	 * @since 2.0
	 *
	 * @param \WC_Paytrace_Token $token
	 *
	 * @throws Exception
	 * @return bool|int
	 */
	public function create_wc_token( \WC_Paytrace_Token $token ) {
		if ( 'Paytrace_eCheck' == $token->get_type() ) {
			$new_token = new WC_Payment_Token_Paytrace_eCheck();
			$new_token->set_routing_last4( $token->get_routing_last4() );
		} else {
			$new_token = new WC_Payment_Token_Paytrace_CC();
			$new_token->set_card_type( $token->get_card_type() );
			$new_token->set_expiry_month( $token->get_expiry_month() );
			// Year needs to be YYYY, so if it is YY add some to it :)
			$year = $token->get_expiry_year();
			if ( strlen( $year ) == 2 ) {
				$year += 2000;
			}
			$new_token->set_expiry_year( $year );
		}
		
		$new_token->set_last4( substr( $token->get_last4(), - 4 ) );
		$new_token->set_profile_id( $token->get_profile_id() );
		$new_token->set_token( $token->get_token() );
		$new_token->set_gateway_id( $token->get_gateway_id() );
		$new_token->set_user_id( $this->user_id );
		
		if ( ! $new_token->validate() ) {
			throw new Exception( __( 'The token could not be validated and created in the store.', WC_Paytrace::TEXT_DOMAIN ) );
		}
		
		return $new_token->save();
	}
	
	/**
	 * Adds the Paytrace Profile to the user meta
	 *
	 * @since 2.0
	 *
	 * @param \WC_Paytrace_Token $new_token
	 *
	 * @return false|int
	 */
	public function create_legacy_profile( \WC_Paytrace_Token $new_token ) {
		if ( 'Paytrace_eCheck' == $new_token->get_type() ) {
			$profile = array(
				'profile_id'  => $new_token->get_profile_id(),
				'customer_id' => $new_token->get_token(),
				'tr_last4'    => substr( $new_token->get_routing_last4(), - 4 ),
				'dda_last4'   => substr( $new_token->get_last4(), - 4 ),
			);
		} else {
			$profile = array(
				'profile_id'  => $new_token->get_profile_id(),
				'customer_id' => $new_token->get_token(),
				'last4'       => substr( $new_token->get_last4(), - 4 ),
				'exp_year'    => $new_token->get_expiry_year(),
				'exp_month'   => $new_token->get_expiry_month(),
			);
		}
		
		return add_user_meta( $this->user_id, '_paytrace_saved_profiles', $profile );
	}
	
	/**---------------------------------------------------
	 * UPDATE method for updating existing Profiles/Tokens
	 * ---------------------------------------------------*/
	
	/**
	 * Updates the Paytrace Profile/Token
	 *
	 * @since 2.0
	 *
	 * @param \WC_Paytrace_Token $new_token
	 * @param \WC_Paytrace_Token $old_token
	 *
	 * @return bool|false|int
	 */
	public function update_profile_or_token( \WC_Paytrace_Token $new_token, \WC_Paytrace_Token $old_token ) {
		if ( WC_Paytrace_Compat::is_wc_2_6() ) {
			return $this->update_wc_token( $new_token, $old_token );
		}
		
		return $this->update_user_legacy_profile( $new_token, $old_token );
	}
	
	/**
	 * Creates a WC Credit card token, saved to the customer, using the passed data.
	 *
	 * @since 2.0
	 *
	 * @param \WC_Paytrace_Token $new_token
	 * @param \WC_Paytrace_Token $old_token
	 *
	 * @throws Exception
	 * @return bool|int
	 */
	public function update_wc_token( \WC_Paytrace_Token $new_token, \WC_Paytrace_Token $old_token ) {
		$old_token_id = $old_token->get_id();
		if ( empty( $old_token_id ) ) {
			throw new Exception( __( 'Token could not be updated. Token ID was not provided.', WC_Paytrace::TEXT_DOMAIN ) );
		}
		
		if ( ! empty( $new_data['routing_last4'] ) ) {
			$token   = new WC_Payment_Token_Paytrace_eCheck( $old_token_id );
			$profile = array(
				'routing_last4' => $new_token->get_routing_last4(),
			);
		} else {
			$token   = new WC_Payment_Token_Paytrace_CC( $old_token_id );
			$profile = array(
				'expiry_year'  => $new_token->get_expiry_year(),
				'expiry_month' => $new_token->get_expiry_month(),
				'card_type'    => $new_token->get_card_type(),
			);
		}
		
		$profile['last4']      = $new_token->get_last4();
		$profile['user_id']    = $this->user_id;
		$profile['profile_id'] = $new_token->get_profile_id();
		$profile['token']      = $new_token->get_token();
		
		// Run through the properties and set the new values.
		foreach ( $profile as $prop_key => $prop_value ) {
			if ( is_callable( array( $token, 'set_' . $prop_key ) ) ) {
				$token->{'set_' . $prop_key}( $prop_value );
			}
		}
		
		// Validate and save the token
		if ( ! $token->validate() ) {
			throw new Exception( __( 'Token could not be updated. Token validation failed.', WC_Paytrace::TEXT_DOMAIN ) );
		}
		
		return $token->save();
	}
	
	/**
	 * Adds the Paytrace Profile to the user meta
	 *
	 * @since 2.0
	 *
	 * @param \WC_Paytrace_Token $new_token
	 * @param \WC_Paytrace_Token $old_token
	 *
	 * @throws \Exception
	 *
	 * @return false|int
	 */
	public function update_user_legacy_profile( \WC_Paytrace_Token $new_token, \WC_Paytrace_Token $old_token ) {
		
		// We need an old profile to update
		if ( 0 == $old_token->get_id() ) {
			throw new Exception( __( 'Profile could not be updated. We did not receive the profile to update.', WC_Paytrace::TEXT_DOMAIN ) );
		}
		
		// We need at least the token to be passed
		$token_value = $new_token->get_token();
		if ( empty( $token_value ) ) {
			throw new Exception( __( 'Profile could not be updated. Profile customer ID was missing.', WC_Paytrace::TEXT_DOMAIN ) );
		}
		
		$old_profile = $this->build_legacy_profile_array( $old_token );
		$profile     = $this->build_legacy_profile_array( $new_token );
		
		return update_user_meta( $this->user_id, '_paytrace_saved_profiles', $profile, $old_profile );
	}
	
	/**
	 * Builds a legacy profile array
	 *
	 * @since 2.0
	 *
	 * @param \WC_Paytrace_Token $token
	 *
	 * @return array
	 */
	public function build_legacy_profile_array( $token ) {
		
		$profile['profile_id']  = $token->get_profile_id();
		$profile['customer_id'] = $token->get_token();
		
		$last4 = $token->get_routing_last4();
		if ( ! empty( $last4 ) ) {
			$profile['tr_last4']  = $token->get_routing_last4();
			$profile['dda_last4'] = $token->get_last4();
		} else {
			$profile['last4']     = $token->get_last4();
			$profile['exp_year']  = $token->get_expiry_year();
			$profile['exp_month'] = $token->get_expiry_month();
		}
		
		return $profile;
	}
	
	/**---------------------------------------------------
	 * DELETE methods for deleting existing Profiles/Tokens
	 * ---------------------------------------------------*/
	
	/**
	 * Updates the Paytrace Profile/Token
	 *
	 * @since 2.0
	 *
	 * @param WC_Paytrace_Token $token
	 * @param bool              $force
	 *
	 * @throws Exception
	 * @return bool|false|int
	 */
	public function delete_profile_token( $token, $force = false ) {
		
		if ( WC_Paytrace_Compat::is_wc_2_6() ) {
			// The token should be deleted directly through the class and via 'woocommerce_paytrace_payment_token_delete'
			if ( true === $force ) {
				$token->delete();
			}
			
			return true;
		}
		
		$profile = $this->build_legacy_profile_array( $token );
		
		return $this->delete_user_profile( $profile );
	}
	
	/**
	 * Creates a WC Credit card token, saved to the customer, using the passed data.
	 *
	 * @since 2.0
	 *
	 * @param $token_id
	 *
	 * @throws Exception
	 *
	 * @return bool|int
	 */
	public function delete_wc_token( $token_id ) {
		
		if ( empty( $token_id ) ) {
			throw new Exception( __( 'Token could not be deleted. Token ID was not provided.', WC_Paytrace::TEXT_DOMAIN ) );
		}
		
		// Does not matter which class deletes the token, the process is the sames
		$token = new WC_Payment_Token_Paytrace_CC( $token_id );
		
		return $token->delete();
	}
	
	/**
	 * Deletes the Paytrace billing profile from the user meta
	 *
	 * @since 2.0
	 *
	 * @param array $profile
	 *
	 * @throws Exception
	 *
	 * @return false|int
	 */
	public function delete_user_profile( $profile ) {
		// We need the profile to delete
		if ( empty( $profile ) ) {
			throw new Exception( __( 'Profile could not be deleted. Profile was not provided.', WC_Paytrace::TEXT_DOMAIN ) );
		}
		
		return delete_user_meta( $this->user_id, '_paytrace_saved_profiles', $profile );
	}
	
	/**---------------------------------
	 * HELPERS
	 * -----------------------------------*/
	
	/**
	 * Transfers Customer profiles, saved as user meta, to WC Tokens
	 *
	 * NOTE: The profiles will not be deleted from the user meta
	 * NOTE: We indicate that the transfer has happened by adding a user meta '_paytrace_legacy_profiles_transferred' to each customer.
	 *
	 * @since 2.0
	 *
	 * @param array $tokens
	 *
	 * @return bool
	 */
	public function transfer_profiles_to_tokens( $tokens ) {
		if ( ! is_array( $tokens ) || empty( $tokens ) ) {
			$this->set_transferred_to_true();
			
			return false;
		}
		
		self::$is_transfer_legacy_profiles_running = true;
		
		/**
		 * @var \WC_Paytrace_Token $token
		 */
		foreach ( $tokens as $token ) {
			try {
				$this->create_wc_token( $token );
			}
			catch ( Exception $e ) {
			}
		}
		
		// Add a note to the user meta that we already transferred their saved profiles
		$this->set_transferred_to_true();
		
		self::$is_transfer_legacy_profiles_running = false;
		
		return true;
	}
	
	/**
	 * Checks and transfers legacy profiles to WC > 2.6 tokens
	 *
	 * @since 2.0
	 *
	 * @return bool
	 */
	public function maybe_transfer_legacy_profiles() {
		// If it is WC > 2.6 always check, if we transferred the legacy user profiles to tokens
		if ( WC_Paytrace_Compat::is_wc_2_6() ) {
			$transferred_legacy_profiles = get_user_meta( $this->user_id, '_paytrace_legacy_profiles_transferred', true );
			// If we have profiles and we never transferred them to tokens
			if ( ! $transferred_legacy_profiles && false === self::$is_transfer_legacy_profiles_running ) {
				$tokens = $this->get_user_saved_profiles();
				
				return $this->transfer_profiles_to_tokens( $tokens );
			}
		}
		
		return true;
	}
	
	/**
	 * Sets the transferred value to true, so more transfers will be performed
	 *
	 * @since 2.0
	 */
	public function set_transferred_to_true() {
		// Add a note to the user meta that we already transferred their saved profiles
		update_user_meta( $this->user_id, '_paytrace_legacy_profiles_transferred', 'true' );
	}
}