<?php

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * WC Payment Token TC for TC tokens.
 * Unifies the tokens and profiles, so we can get the information in a consistent manner.
 *
 * @since  2.0
 * @author VanboDevelops
 *
 *        Copyright: (c) 2017 VanboDevelops
 *        License: GNU General Public License v3.0
 *        License URI: http://www.gnu.org/licenses/gpl-3.0.html
 */
class WC_Payment_Token_Paytrace_eCheck extends \WC_Payment_Token_eCheck {
	
	/** @protected string Token Type String. */
	protected $type = 'Paytrace_eCheck';
	protected $token_type = 'echeck';
	protected $routing_last4;
	
	/**
	 * Get type to display to user.
	 *
	 * @since  2.6.0
	 *
	 * @param  string $deprecated Deprecated since WooCommerce 3.0
	 *
	 * @return string
	 */
	public function get_display_name( $deprecated = '' ) {
		/* translators: 1: credit card type 2: last 4 digits 3: expiry month 4: expiry year */
		$display = sprintf(
			__( 'eCheck ending in %1$s', WC_Paytrace::TEXT_DOMAIN ),
			$this->get_last4()
		);
		
		return apply_filters( 'wc_paytrace_echeck_display_name', $display, $this );
	}
	
	/**
	 * Hook prefix
	 *
	 * @since 2.0
	 */
	protected function get_hook_prefix() {
		return 'woocommerce_payment_token_paytrace_echeck_get_';
	}
	
	/**
	 * Validate credit card payment tokens.
	 *
	 * These fields are required by all credit card payment tokens:
	 * expiry_month  - string Expiration date (MM) for the card
	 * expiry_year   - string Expiration date (YYYY) for the card
	 * last4         - string Last 4 digits of the card
	 *
	 * @since 2.0
	 * @return boolean True if the passed data is valid
	 */
	public function validate() {
		if ( ! $this->get_last4( 'edit' ) ) {
			return false;
		}
		
		if ( ! $this->get_routing_last4( 'edit' ) ) {
			return false;
		}
		
		return true;
	}
	
	/**
	 * Delete an object, set the ID to 0, and return result.
	 *
	 * @since  2.0
	 *
	 * @param  bool $force_delete
	 *
	 * @return bool result
	 */
	public function delete( $force_delete = false ) {
		if ( $this->data_store ) {
			
			// Run an action before we delete a Payment Token
			do_action( 'woocommerce_paytrace_payment_token_delete', $this->get_id(), $this );
			
			$this->data_store->delete( $this, array( 'force_delete' => $force_delete ) );
			$this->set_id( 0 );
			
			return true;
		}
		
		return false;
	}
	
	/**
	 * Returns the token type
	 * Types: card | echeck
	 *
	 * @since  2.0
	 *
	 * @param string $context
	 *
	 * @return mixed
	 */
	public function get_token_type( $context = 'view' ) {
		return $this->get_meta( 'token_type' );
	}
	
	/**
	 * Set the last four digits.
	 *
	 * @since 2.0
	 *
	 * @param string $last4
	 */
	public function set_token_type( $last4 ) {
		$this->add_meta_data( 'token_type', $last4, true );
	}
	
	/**
	 * Returns the routing last 4
	 *
	 * @since 2.0
	 *
	 * @param string $context
	 *
	 * @return mixed
	 */
	public function get_routing_last4( $context = 'view' ) {
		return $this->get_meta( 'routing_last4' );
	}
	
	/**
	 * Sets the routing last 4
	 *
	 * @since 2.0
	 *
	 * @param $last4
	 */
	public function set_routing_last4( $last4 ) {
		$this->add_meta_data( 'routing_last4', $last4, true );
	}
	
	/**
	 * Returns the Paytrace Profile id of the token
	 *
	 * @since 2.0
	 *
	 * @param string $context
	 *
	 * @return mixed
	 */
	public function get_profile_id( $context = 'view' ) {
		return $this->get_meta( 'profile_id' );
	}
	
	/**
	 * Sets the Paytrace Profile id of the token
	 *
	 * @since 2.0
	 *
	 * @param $profile_id
	 */
	public function set_profile_id( $profile_id ) {
		$this->add_meta_data( 'profile_id', $profile_id, true );
	}
}