<?php

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * WC Payment Token TC for TC tokens.
 * Unifies the tokens and profiles, so we can get the information in a consistent manner.
 *
 * @since  2.0
 * @author VanboDevelops
 *
 *        Copyright: (c) 2017 VanboDevelops
 *        License: GNU General Public License v3.0
 *        License URI: http://www.gnu.org/licenses/gpl-3.0.html
 */
class WC_Payment_Token_Paytrace_CC extends \WC_Payment_Token_CC {
	
	/** @protected string Token Type String. */
	protected $type = 'Paytrace_CC';
	protected $token_type = 'card';

	/**
	 * Returns the card type label
	 *
	 * @since 2.1.1
	 *
	 * @param $type
	 *
	 * @return string
	 */
	public function get_card_type_label( $type ) {
		$label = wc_get_credit_card_type_label( $type );
		$label = '' == $label ? _x( 'Card', 'label-unknown-card-type', WC_Paytrace::TEXT_DOMAIN ) : $label;

		return apply_filters( 'wc_paytrace_card_type_label', $label, $type );
	}
	
	/**
	 * Get type to display to user.
	 *
	 * @since  2.0
	 *
	 * @param  string $deprecated Deprecated since WooCommerce 3.0
	 *
	 * @return string
	 */
	public function get_display_name( $deprecated = '' ) {
		/* translators: 1: credit card type 2: last 4 digits 3: expiry month 4: expiry year */
		$display = sprintf(
			__( '%1$s ending in %2$s (expires %3$s/%4$s)', WC_Paytrace::TEXT_DOMAIN ),
			$this->get_card_type_label( $this->get_card_type() ),
			$this->get_last4(),
			$this->get_expiry_month(),
			substr( $this->get_expiry_year(), 2 )
		);
		
		return apply_filters( 'wc_paytrace_card_display_name', $display, $this );
	}
	
	/**
	 * Hook prefix
	 *
	 * @since 2.0
	 */
	protected function get_hook_prefix() {
		return 'woocommerce_payment_token_paytrace_cc_get_';
	}
	
	/**
	 * Validate credit card payment tokens.
	 *
	 * These fields are required by all credit card payment tokens:
	 * expiry_month  - string Expiration date (MM) for the card
	 * expiry_year   - string Expiration date (YYYY) for the card
	 * last4         - string Last 4 digits of the card
	 *
	 * @since 2.0
	 *
	 * @return boolean True if the passed data is valid
	 */
	public function validate() {
		$token = $this->get_prop( 'token', 'edit' );
		if ( empty( $token ) ) {
			return false;
		}
		
		if ( ! $this->get_last4( 'edit' ) ) {
			return false;
		}
		
		if ( ! $this->get_expiry_year( 'edit' ) ) {
			return false;
		}
		
		if ( ! $this->get_expiry_month( 'edit' ) ) {
			return false;
		}
		
		if ( 4 !== strlen( $this->get_expiry_year( 'edit' ) ) ) {
			return false;
		}
		
		if ( 2 !== strlen( $this->get_expiry_month( 'edit' ) ) ) {
			return false;
		}
		
		return true;
	}
	
	/**
	 * Returns the Paytrace Profile id of the token
	 *
	 * @since 2.0
	 *
	 * @param string $context
	 *
	 * @return mixed
	 */
	public function get_profile_id( $context = 'view' ) {
		return $this->get_meta( 'profile_id' );
	}
	
	/**
	 * Sets the Paytrace Profile id of the token
	 *
	 * @since 2.0
	 *
	 * @param $profile_id
	 */
	public function set_profile_id( $profile_id ) {
		$this->add_meta_data( 'profile_id', $profile_id, true );
	}

	/**
	 * Return if the method is set to be the default one
	 *
	 * @since 2.2
	 *
	 * @param string $context
	 *
	 * @return mixed
	 */
	public function get_is_default( $context = 'view' ) {
		if ( ! \WC_Paytrace_Compat::is_wc_3_0() ) {
			return $this->is_default();
		}

		return parent::get_is_default( $context );
	}

	/**
	 * Delete an object, set the ID to 0, and return result.
	 *
	 * @since  2.0
	 *
	 * @param  bool $force_delete
	 *
	 * @return bool result
	 */
	public function delete( $force_delete = false ) {
		if ( $this->data_store ) {
			
			// Run an action before we delete a Payment Token
			do_action( 'woocommerce_paytrace_payment_token_delete', $this->get_id(), $this );
			
			$this->data_store->delete( $this, array( 'force_delete' => $force_delete ) );
			$this->set_id( 0 );
			
			return true;
		}
		
		return false;
	}
}