<?php

namespace WcPaytrace\Integrations\Post;

use WcPaytrace\Abstracts\Integration;
use WcPaytrace\Abstracts\Process;
use WcPaytrace\Abstracts\Validator;
use WcPaytrace\Api\Post\Response;

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Integration class to handle the POST integration actions.
 * Payment, Refunds, Captures, Voids they will all be started and finished here.
 *
 * @since  2.0
 * @author VanboDevelops
 *
 *        Copyright: (c) 2017 VanboDevelops
 *        License: GNU General Public License v3.0
 *        License URI: http://www.gnu.org/licenses/gpl-3.0.html
 */
class Integration_Post extends Process {
	
	/**
	 * Integration_Post constructor.
	 *
	 * @param \WC_Paytrace_Gateway $gateway
	 */
	public function __construct( $gateway ) {
		parent::__construct( $gateway );
	}
	
	/**
	 * Returns profile delete request parameters
	 *
	 * @since 2.0
	 *
	 * @param \WC_Payment_Token $token
	 *
	 * @return array
	 */
	public function get_profile_delete_request( $token ) {
		$request = array(
			'CUSTID' => $token->get_token(),
		);
		
		return $request;
	}
	
	/**
	 * Creates profile from an order
	 * Returns the "Create profile request" parameters
	 *
	 * @param        $order
	 * @param string $payment_type
	 *
	 * @throws \Exception
	 * @return array|mixed
	 */
	public function get_profile_create_request( $order, $payment_type = 'card' ) {
		// Init the class
		$services  = $this->get_api_services();
		$validator = $services->get_validator();
		
		$request = array(
			'CUSTID' => uniqid() . '-' . get_current_user_id(),
		);
		
		$request = $this->add_billing_address_details_to_request( $request, $order );
		$request = $this->maybe_add_shipping_address_details_to_request( $request, $order );
		
		// Debug
		$this->log_request( $request, 'POST: Create Profile from order Request:' );
		
		$request = $this->add_card_or_echeck_details( $request, $validator, $payment_type );
		
		return $request;
	}
	
	/**
	 * Creates profile from submitted card/eCheck without an order
	 *
	 * @since 2.1
	 *
	 * @param        $user_id
	 * @param string $payment_type
	 *
	 * @throws \Exception
	 * @return mixed|\WcPaytrace\Api\Json\Response|Response
	 */
	public function create_profile( $user_id, $payment_type = 'card' ) {
		// Init the class
		$services  = $this->get_api_services();
		$validator = $services->get_validator();
		$request   = array(
			'CUSTID' => uniqid() . '-' . get_current_user_id(),
		);
		
		// Set the Billing address data
		$request['BNAME']     = get_user_meta( $user_id, 'billing_first_name', true ) . ' ' . get_user_meta( $user_id, 'billing_last_name', true );
		$request['BADDRESS']  = get_user_meta( $user_id, 'billing_address_1', true );
		$request['BZIP']      = get_user_meta( $user_id, 'billing_postcode', true );
		$request['BADDRESS2'] = get_user_meta( $user_id, 'billing_address_2', true );
		$request['BCITY']     = get_user_meta( $user_id, 'billing_city', true );
		$request['BSTATE']    = strlen( get_user_meta( $user_id, 'billing_state', true ) ) != 2 ? '' : get_user_meta( $user_id, 'billing_state', true );
		$request['BCOUNTRY']  = get_user_meta( $user_id, 'billing_country', true );
		$request['EMAIL']     = get_user_meta( $user_id, 'billing_email', true );
		
		// Debug
		$this->log_request( $request, 'POST: Profile from user Request:' );
		
		$request = $this->add_card_or_echeck_details( $request, $validator, $payment_type );
		
		// Run the create profile request
		$response = $services->vault()->create_profile( $request );
		
		return $response;
	}
	
	/**
	 * Returns the "Profile Update request" parameters
	 *
	 * @param \WC_Payment_Token $token
	 * @param Validator         $validator
	 *
	 * @throws \Exception
	 * @return array
	 */
	public function get_profile_update_request( $token, Validator $validator ) {
		
		$request = array(
			'CUSTID' => $token->get_token(),
		);
		
		if ( null != \WC_Paytrace::get_post( 'paytrace_update_check' ) ) {
			
			\WC_Paytrace::add_debug_log( 'Update type: echeck' );
			$name = \WC_Paytrace::get_post( 'paytrace-name-on-echeck' );
			$validator->set_echeck_details();
			
			$validator->validate_echeck_fields( $validator->echeck_routing, $validator->echeck_account );
			
			$request['TR']  = $validator->echeck_routing;
			$request['DDA'] = $validator->echeck_account;
		} else {
			\WC_Paytrace::add_debug_log( 'Update type: card' );
			$validator->set_credit_card_details();
			$name = \WC_Paytrace::get_post( 'paytrace-name-on-card' );
			
			$validator->validate_cc_number( $validator->cc_number, $validator->get_card_type_from_number( $validator->cc_number ) );
			
			$request['CC']      = $validator->cc_number;
			$request['EXPMNTH'] = $validator->cc_exp_month;
			$request['EXPYR']   = $validator->cc_exp_year;
		}
		
		$request['BNAME'] = $name;
		
		return $request;
	}
	
	/**
	 * Adds the card or echeck payment details to the request
	 *
	 * @since 2.0
	 *
	 * @param array     $params
	 * @param Validator $validator
	 * @param string    $payment_type
	 *
	 * @return mixed
	 */
	public function add_card_or_echeck_details( $params, $validator, $payment_type = 'card' ) {
		if ( 'check' == $payment_type ) {
			// Set fields
			$validator->set_echeck_details();
			// Validate fields
			$validator->validate_echeck_fields( $validator->echeck_account, $validator->echeck_routing );
			
			// Add to the params
			$params['DDA'] = $validator->echeck_account;
			$params['TR']  = $validator->echeck_routing;
		} else {
			$validator->set_credit_card_details();
			$validator->validate_card_fields( $validator->cc_number, $validator->cc_cvc, $validator->cc_exp_month, $validator->cc_exp_year, $validator->get_card_type_from_number( $validator->cc_number ) );
			$params['CC']      = $validator->cc_number;
			$params['EXPMNTH'] = $validator->cc_exp_month;
			$params['EXPYR']   = $validator->cc_exp_year;
			$params['CSC']     = $validator->cc_cvc;
		}
		
		return $params;
	}
	
	/**
	 * Adds the billing details to the request array
	 *
	 * @since 2.0
	 *
	 * @param array     $request
	 * @param \WC_Order $order
	 *
	 * @return array The request array
	 */
	public function add_billing_address_details_to_request( $request, $order ) {
		// Set the Billing address data
		$request['BNAME']     = \WC_Paytrace_Compat::get_order_billing_first_name( $order ) . ' ' . \WC_Paytrace_Compat::get_order_billing_last_name( $order );
		$request['BADDRESS']  = \WC_Paytrace_Compat::get_order_billing_address_1( $order );
		$request['BZIP']      = \WC_Paytrace_Compat::get_order_billing_postcode( $order );
		$request['BADDRESS2'] = \WC_Paytrace_Compat::get_order_billing_address_2( $order );
		$request['BCITY']     = \WC_Paytrace_Compat::get_order_billing_city( $order );
		$request['BSTATE']    = strlen( \WC_Paytrace_Compat::get_order_billing_state( $order ) ) != 2 ? '' : \WC_Paytrace_Compat::get_order_billing_state( $order );
		$request['BCOUNTRY']  = \WC_Paytrace_Compat::get_order_billing_country( $order );
		$request['EMAIL']     = \WC_Paytrace_Compat::get_order_billing_email( $order );
		
		return $request;
	}
	
	/**
	 * Adds the shipping details to the request
	 *
	 * @since 2.0
	 *
	 * @param $request
	 * @param $order
	 *
	 * @return mixed
	 */
	public function maybe_add_shipping_address_details_to_request( $request, $order ) {
		// Set the Shipping address data
		if ( '' != \WC_Paytrace_Compat::get_order_shipping_address_1( $order ) ) {
			$request['SNAME']     = \WC_Paytrace_Compat::get_order_shipping_first_name( $order ) . ' ' . \WC_Paytrace_Compat::get_order_shipping_last_name( $order );
			$request['SADDRESS']  = \WC_Paytrace_Compat::get_order_shipping_address_1( $order );
			$request['SZIP']      = \WC_Paytrace_Compat::get_order_shipping_postcode( $order );
			$request['SADDRESS2'] = \WC_Paytrace_Compat::get_order_shipping_address_2( $order );
			$request['SCITY']     = \WC_Paytrace_Compat::get_order_shipping_city( $order );
			// State cannot be more than two characters,
			// so just don't send it if it is longer than two characters
			$request['SSTATE']   = ( strlen( \WC_Paytrace_Compat::get_order_shipping_state( $order ) ) != 2 ) ? '' : \WC_Paytrace_Compat::get_order_shipping_state( $order );
			$request['SCOUNTRY'] = \WC_Paytrace_Compat::get_order_shipping_country( $order );
		}
		
		return $request;
	}
	
	/**
	 * @param \WC_Order $order
	 * @param null      $amount
	 * @param bool      $is_subscription
	 * @param bool      $is_paid_with_profile
	 *
	 * @throws \Exception
	 * @return array
	 */
	public function get_transaction_request( $order, $amount = null, $is_subscription = false, $is_paid_with_profile = false ) {
		// Init the class
		$services  = $this->get_api_services();
		$validator = $services->get_validator();
		
		// If we have a subs payment the amount should be send as a param
		$order_total = empty( $amount ) ? number_format( $order->get_total(), 2, '.', '' ) : number_format( $amount, 2, '.', '' );
		
		$request = array(
			'AMOUNT'  => $order_total,
			'INVOICE' => \WC_Paytrace::get_order_number( $order ) . '_' . date( 'YmdHis' ),
		);
		
		if ( $is_paid_with_profile ) {
			// Get the saved card
			$order_tokens = new \WC_Paytrace_Order( $order );
			$customer_id  = $order_tokens->get_customer_id();
			
			$customer_tokens = new \WC_Paytrace_Customer_Tokens( $order->get_user_id() );
			$token           = $customer_tokens->get_token_by_customer_id( $customer_id );
			
			if ( ! $customer_id || ! $token ) {
				throw new \Exception( __( 'Customer profile not found.', \WC_Paytrace::TEXT_DOMAIN ) );
			}
			
			$request['METHOD'] = 'Paytrace_eCheck' == $token->get_type() ? 'ProcessCheck' : 'ProcessTranx';
			$request['CUSTID'] = $customer_id;
			
			// If the CVC is required, we want to send it every time the customer performs manual transaction with a card
			// If we are processing automatic(scheduled) payment, we don't want to send the CVC
			if ( 'yes' == $this->get_gateway()->get_option( 'require_cvc_with_saved_cards' )
			     && null !== \WC_Paytrace::get_field( 'paytrace-card-expiry', $_POST, null )
			     && ( 'saved_card' == $this->get_gateway()->get_used_payment_type()
			          || 'new_card' == $this->get_gateway()->get_used_payment_type() )
			) {
				$validator->set_credit_card_details();
				
				if ( '' != $validator->cc_cvc || $this->get_gateway()->is_cvc_required() ) {
					$validator->validate_cvc_number( $validator->cc_cvc, $validator->get_card_type_from_number( $validator->cc_number ) );
				}
				
				if ( '' != $validator->cc_cvc ) {
					$request['CSC'] = $validator->cc_cvc;
				}
			}
		} else {
			// Set the cc or check fields
			if ( 'new_check' == $this->get_gateway()->get_used_payment_type() ) {
				// Card payments just run the set transaction type
				$request['METHOD'] = 'ProcessCheck';
				$request           = $this->add_card_or_echeck_details( $request, $validator, 'check' );
			} else {
				$request['METHOD'] = 'ProcessTranx';
				$request           = $this->add_card_or_echeck_details( $request, $validator, 'card' );
			}
		}
		
		if ( 'new_card' == $this->get_gateway()->get_used_payment_type() ) {
			// Attempt to set the card details
			$validator->set_credit_card_details();
			
			// If the customer gave us a CVC number, us it in the request
			if ( ! empty( $validator->cc_cvc ) ) {
				$request['CSC'] = $validator->cc_cvc;
			}
		}
		// Set the Billing address data
		$request = $this->add_billing_address_details_to_request( $request, $order );
		// Set the Shipping address data
		$request = $this->maybe_add_shipping_address_details_to_request( $request, $order );
		
		if ( $this->maybe_send_order_description() ) {
			// Get Order description
			$request['DESCRIPTION'] = $this->get_order_description( $order, $is_subscription );
		}
		
		return $request;
	}
	
	/**
	 * Returns the refund request parameters
	 *
	 * @param \WC_Order $order
	 * @param null      $amount
	 * @param string    $reason
	 *
	 * @throws \Exception
	 * @return array
	 */
	public function get_refund_request( \WC_Order $order, $amount = null, $reason = '' ) {
		$order_tokens = new \WC_Paytrace_Order( $order );
		
		// Get the saved card
		$customer_id    = $order_tokens->get_customer_id();
		$transaction_id = $order_tokens->get_transaction_id();
		$payment_type   = $order_tokens->get_payment_type();
		
		// Bail, if there is no reference ID
		if ( '' == $customer_id && '' == $transaction_id ) {
			throw new \Exception( __( 'Error: Missing Reference ID. The order does not have all required information to process a refund.', \WC_Paytrace::TEXT_DOMAIN ) );
		}
		
		$request = array(
			'AMOUNT' => $amount,
		);
		
		// Allows 3rd party to overwrite the behavior and use the "customer_id" for refunds
		if ( apply_filters( 'wc_paytrace_refund_with_transaction_id', true, $order, $amount ) && $transaction_id ) {
			// If there is no profile, we will use the transaction ID.
			$request['METHOD'] = 'card' == $payment_type ? 'ProcessTranx' : 'ProcessCheck';
			
			if ( 'card' == $payment_type ) {
				$request['TRANXID'] = $transaction_id;
			} else {
				$request['CHECKID'] = $transaction_id;
			}
		} else {
			$token = false;
			
			// If we have a profile saved we will use it in priority
			if ( '' != $customer_id ) {
				$customer_tokens = new \WC_Paytrace_Customer_Tokens( $order->get_user_id() );
				$token           = $customer_tokens->get_token_by_customer_id( $customer_id );
			}
			
			if ( ! $token ) {
				throw new \Exception( __( 'Error: Missing token. The order does not have all required information to process a refund.', \WC_Paytrace::TEXT_DOMAIN ) );
			}
			
			// Set the customer ID to be refunded
			$request['CUSTID'] = $token->get_token();
			$request['METHOD'] = 'Paytrace_eCheck' == $token->get_type() ? 'ProcessCheck' : 'ProcessTranx';
		}
		
		return $request;
	}
	
	/**
	 * Returns the refund request parameters
	 *
	 * @param \WC_Order $order
	 * @param float     $amount
	 *
	 * @throws \Exception
	 * @return array
	 */
	public function get_capture_request( \WC_Order $order, $amount ) {
		$order_tokens = new \WC_Paytrace_Order( $order );
		
		// Get the saved card
		$transaction_id = $order_tokens->get_transaction_id();
		$payment_type   = $order_tokens->get_payment_type();
		
		// If there is no profile, we will use the transaction ID.
		$request['METHOD'] = 'card' == $payment_type ? 'ProcessTranx' : 'ProcessCheck';
		
		if ( 'card' == $payment_type ) {
			$request['TRANXID'] = $transaction_id;
		} else {
			$request['CHECKID'] = $transaction_id;
		}
		
		$amount_authorized = $order_tokens->get_order_amount_authorized();
		if ( $amount != $amount_authorized ) {
			$request['AMOUNT'] = $amount;
		}
		
		return $request;
	}
	
	/**
	 * Returns the parameters for the email receipt request
	 *
	 * @since 2.3.0
	 *
	 * @param \WC_Order $order
	 *
	 * @return mixed
	 */
	public function get_email_receipt_request( \WC_Order $order ) {
		$order_tokens = new \WC_Paytrace_Order( $order );
		
		// Get the saved card
		$transaction_id = $order_tokens->get_transaction_id();
		$payment_type   = $order_tokens->get_payment_type();
		
		if ( 'card' == $payment_type ) {
			$request['TRANXID'] = $transaction_id;
		} else {
			$request['CHECKID'] = $transaction_id;
		}
		
		$request['EMAIL'] = \WC_Paytrace_Compat::get_order_billing_email( $order );
		
		return $request;
	}
}