<?php

namespace WcPaytrace\Integrations\Json;

use WcPaytrace\Abstracts\Integration;
use WcPaytrace\Abstracts\Process;
use WcPaytrace\Abstracts\Validator;

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Integration class to handle the POST integration actions.
 * Payment, Refunds, Captures, Voids they will all be started and finished here.
 *
 * @since  2.0
 * @author VanboDevelops
 *
 *        Copyright: (c) 2017 VanboDevelops
 *        License: GNU General Public License v3.0
 *        License URI: http://www.gnu.org/licenses/gpl-3.0.html
 */
class Integration_Json extends Process {
	
	/**
	 * Integration_Post constructor.
	 *
	 * @param \WC_Paytrace_Gateway $gateway
	 */
	public function __construct( $gateway ) {
		parent::__construct( $gateway );
	}
	
	/**
	 * Returns profile delete request parameters
	 *
	 * @since 2.0
	 *
	 * @param \WC_Payment_Token $token
	 *
	 * @return array
	 */
	public function get_profile_delete_request( $token ) {
		$request = array(
			'customer_id' => $token->get_token(),
		);
		
		return $request;
	}
	
	/**
	 * Creates profile from an order
	 * Returns the "Create profile request" parameters
	 *
	 * @param        $order
	 * @param string $payment_type
	 *
	 * @throws \Exception
	 *
	 * @return array|mixed
	 */
	public function get_profile_create_request( $order, $payment_type = 'card' ) {
		// Init the class
		$services  = $this->get_api_services();
		$validator = $services->get_validator();
		
		$request = array(
			'customer_id' => uniqid() . '-' . get_current_user_id(),
			'email'       => \WC_Paytrace_Compat::get_order_billing_email( $order ),
		);
		
		if ( '' != \WC_Paytrace_Compat::get_order_billing_phone( $order ) ) {
			$request['phone'] = \WC_Paytrace_Compat::get_order_billing_phone( $order );
		}
		
		$request = $this->add_billing_address_details_to_request( $request, $order );
		$request = $this->maybe_add_shipping_address_details_to_request( $request, $order );
		
		// Debug
		$this->log_request( $request, 'JSON: Create Profile from order Request:' );
		
		$request = $this->add_card_or_echeck_details( $request, $validator, $payment_type );
		
		return $request;
	}
	
	/**
	 * Creates profile from submitted card/eCheck without an order
	 *
	 * NOTE: The method name is confusing. We are creating a Paytrace profile from the user meta data,
	 * but we also have a process to create profile from the order meta data. The name should reflect that
	 *
	 * @since 2.1
	 *
	 * @param        $user_id
	 * @param string $payment_type
	 *
	 * @throws \Exception
	 *
	 * @return mixed|\WcPaytrace\Api\Json\Response|\WcPaytrace\Api\Json\Response
	 */
	public function create_profile( $user_id, $payment_type = 'card' ) {
		// Init the class
		$services  = $this->get_api_services();
		$validator = $services->get_validator();
		$request   = array(
			'customer_id' => uniqid() . '-' . get_current_user_id(),
			'email'       => get_user_meta( $user_id, 'billing_email', true ),
		);
		
		// Set the Billing address data
		$request['billing_address'] = array(
			'name'            => get_user_meta( $user_id, 'billing_first_name', true ) . ' ' . get_user_meta( $user_id, 'billing_last_name', true ),
			'street_address'  => get_user_meta( $user_id, 'billing_address_1', true ),
			'street_address2' => get_user_meta( $user_id, 'billing_address_2', true ),
			'city'            => get_user_meta( $user_id, 'billing_city', true ),
			'state'           => strlen( get_user_meta( $user_id, 'billing_state', true ) ) != 2 ? '' : get_user_meta( $user_id, 'billing_state', true ),
			'zip'             => get_user_meta( $user_id, 'billing_postcode', true ),
			'country'         => get_user_meta( $user_id, 'billing_country', true ),
		);
		
		// Debug
		$this->log_request( $request, 'Create Profile from user Request:' );
		
		$request = $this->add_card_or_echeck_details( $request, $validator, $payment_type );
		
		// Run the create profile request
		$response = $services->vault()->create_profile( $request );
		
		return $response;
	}
	
	/**
	 * Returns the "Profile Update request" parameters
	 *
	 * @param \WC_Payment_Token $token
	 * @param Validator         $validator
	 *
	 * @throws \Exception
	 * @return array
	 */
	public function get_profile_update_request( $token, Validator $validator ) {
		
		if ( null != \WC_Paytrace::get_post( 'paytrace_update_check' ) ) {
			
			\WC_Paytrace::add_debug_log( 'Update type: echeck' );
			$name = \WC_Paytrace::get_post( 'paytrace-name-on-echeck' );
			$validator->set_echeck_details();
			
			$validator->validate_echeck_fields( $validator->echeck_routing, $validator->echeck_account );
			
			$request = array(
				'check' => array(
					'account_number' => $validator->echeck_account,
					'routing_number' => $validator->echeck_routing,
				),
			);
		} else {
			\WC_Paytrace::add_debug_log( 'Update type: card' );
			$validator->set_credit_card_details();
			$name = \WC_Paytrace::get_post( 'paytrace-name-on-card' );
			
			$validator->validate_card_number( $validator->cc_number, $validator->get_card_type_from_number( $validator->cc_number ) );
			
			$request = array(
				'credit_card' => array(
					'number'           => $validator->cc_number,
					'expiration_month' => $validator->cc_exp_month,
					'expiration_year'  => $validator->cc_exp_year,
				),
			);
		}
		
		$customer_id = get_current_user_id();
		
		$request['billing_address'] = array(
			'name'            => $name,
			'street_address'  => get_user_meta( $customer_id, 'billing_address_1', true ),
			'street_address2' => get_user_meta( $customer_id, 'billing_address_2', true ),
			'city'            => get_user_meta( $customer_id, 'billing_city', true ),
			'state'           => strlen( get_user_meta( $customer_id, 'billing_state', true ) ) != 2 ? '' : get_user_meta( $customer_id, 'billing_state', true ),
			'zip'             => get_user_meta( $customer_id, 'billing_postcode', true ),
			'country'         => get_user_meta( $customer_id, 'billing_country', true ),
		);
		
		$request['customer_id'] = $token->get_token();
		
		return $request;
	}
	
	/**
	 * Adds the card or echeck payment details to the request
	 *
	 * @since 2.0
	 *
	 * @param array     $request
	 * @param Validator $validator
	 * @param string    $payment_type
	 *
	 * @throws \Exception
	 *
	 * @return mixed
	 */
	public function add_card_or_echeck_details( $request, $validator, $payment_type = 'card' ) {
		if ( 'check' == $payment_type ) {
			return $this->add_echeck_details( $request, $validator );
		}
		
		if ( 'encryption' == $this->get_gateway()->get_security_type() ) {
			return $this->add_card_encryption_details( $request, $validator );
		}
		
		if ( 'protect' == $this->get_gateway()->get_security_type() ) {
			return $this->add_card_protect_details( $request, $validator );
		}
		
		return $this->add_card_details( $request, $validator );
	}
	
	/**
	 * Adds the card details to the request
	 *
	 * @since 2.7.0
	 *
	 * @param array     $request
	 * @param Validator $validator
	 *
	 * @throws \Exception
	 *
	 * @return array
	 */
	protected function add_card_details( $request, $validator ) {
		$validator->set_credit_card_details();
		$validator->validate_card_fields( $validator->cc_number, $validator->cc_cvc, $validator->cc_exp_month, $validator->cc_exp_year, $validator->get_card_type_from_number( $validator->cc_number ) );
		
		$request['credit_card'] = array(
			'number'           => $validator->cc_number,
			'expiration_month' => $validator->cc_exp_month,
			'expiration_year'  => $validator->cc_exp_year,
		);
		
		// If the customer gave us a CVC number, use it in the request
		if ( ! empty( $validator->cc_cvc ) ) {
			$request['csc'] = $validator->cc_cvc;
		}
		
		return $request;
	}
	
	/**
	 * Adds the protect.js details to the request
	 *
	 * @since 2.7.0
	 *
	 * @param array     $request
	 * @param Validator $validator
	 *
	 * @throws \Exception
	 *
	 * @return array
	 */
	protected function add_card_protect_details( $request, $validator ) {
		$request['hpf_token'] = \WC_Paytrace::get_post( 'paytrace_hpf_token' );
		$request['enc_key']   = \WC_Paytrace::get_post( 'paytrace_enc_key' );
		
		return $request;
	}
	
	/**
	 * Adds the card client-side encryption details to the request
	 *
	 * @since 2.7.0
	 *
	 * @param array     $request
	 * @param Validator $validator
	 *
	 * @throws \Exception
	 *
	 * @return array
	 */
	protected function add_card_encryption_details( $request, $validator ) {
		$validator->set_credit_card_details();
		$validator->validate_card_fields( $validator->cc_number, $validator->cc_cvc, $validator->cc_exp_month, $validator->cc_exp_year, $validator->get_card_type_from_number( $validator->cc_number ) );
		
		$request['credit_card'] = array(
			'encrypted_number' => $validator->cc_number,
			'expiration_month' => $validator->cc_exp_month,
			'expiration_year'  => $validator->cc_exp_year,
		);
		
		// If the customer gave us a CVC number, use it in the request
		if ( ! empty( $validator->cc_cvc ) ) {
			$request['encrypted_csc'] = $validator->cc_cvc;
		}
		
		return $request;
	}
	
	/**
	 * Adds the card or echeck payment details to the request
	 *
	 * @since 2.7.0
	 *
	 * @param array     $request
	 * @param Validator $validator
	 *
	 * @throws \Exception
	 *
	 * @return mixed
	 */
	protected function add_echeck_details( $request, $validator ) {
		// Set fields
		$validator->set_echeck_details();
		// Validate fields
		$validator->validate_echeck_fields( $validator->echeck_account, $validator->echeck_routing );
		
		$request['check'] = array(
			'account_number' => $validator->echeck_account,
			'routing_number' => $validator->echeck_routing,
		);
		
		return $request;
	}
	
	/**
	 * Adds the billing details to the request array
	 *
	 * @since 2.0
	 *
	 * @param array     $request
	 * @param \WC_Order $order
	 *
	 * @return array The request array
	 */
	public function add_billing_address_details_to_request( $request, $order ) {
		// Set the Billing address data
		$request['billing_address'] = array(
			'name'            => \WC_Paytrace_Compat::get_order_billing_first_name( $order ) . ' ' . \WC_Paytrace_Compat::get_order_billing_last_name( $order ),
			'street_address'  => \WC_Paytrace_Compat::get_order_billing_address_1( $order ),
			'street_address2' => \WC_Paytrace_Compat::get_order_billing_address_2( $order ),
			'city'            => \WC_Paytrace_Compat::get_order_billing_city( $order ),
			'state'           => strlen( \WC_Paytrace_Compat::get_order_billing_state( $order ) ) != 2 ? '' : \WC_Paytrace_Compat::get_order_billing_state( $order ),
			'zip'             => \WC_Paytrace_Compat::get_order_billing_postcode( $order ),
			'country'         => \WC_Paytrace_Compat::get_order_billing_country( $order ),
		);
		
		return $request;
	}
	
	/**
	 * Adds the shipping details to the request
	 *
	 * @since 2.0
	 *
	 * @param $request
	 * @param $order
	 *
	 * @return mixed
	 */
	public function maybe_add_shipping_address_details_to_request( $request, $order ) {
		// Set the Shipping address data
		if ( '' != \WC_Paytrace_Compat::get_order_shipping_address_1( $order ) ) {
			$request['shipping_address'] = array(
				'name'            => \WC_Paytrace_Compat::get_order_shipping_first_name( $order ) . ' ' . \WC_Paytrace_Compat::get_order_shipping_last_name( $order ),
				'street_address'  => \WC_Paytrace_Compat::get_order_shipping_address_1( $order ),
				'street_address2' => \WC_Paytrace_Compat::get_order_shipping_address_2( $order ),
				'city'            => \WC_Paytrace_Compat::get_order_shipping_city( $order ),
				'state'           => strlen( \WC_Paytrace_Compat::get_order_shipping_state( $order ) ) != 2 ? '' : \WC_Paytrace_Compat::get_order_shipping_state( $order ),
				'zip'             => \WC_Paytrace_Compat::get_order_shipping_postcode( $order ),
				'country'         => \WC_Paytrace_Compat::get_order_shipping_country( $order ),
			);
		}
		
		return $request;
	}
	
	/**
	 * @param \WC_Order $order
	 * @param null      $amount
	 * @param bool      $is_subscription
	 * @param bool      $is_paid_with_profile
	 *
	 * @throws \Exception
	 * @return array
	 */
	public function get_transaction_request( $order, $amount = null, $is_subscription = false, $is_paid_with_profile = false ) {
		// Init the class
		$services  = $this->get_api_services();
		$validator = $services->get_validator();
		
		// If we have a subs payment the amount should be send as a param
		$order_total = empty( $amount ) ? number_format( $order->get_total(), 2, '.', '' ) : number_format( $amount, 2, '.', '' );
		
		$request = array(
			'amount'     => $order_total,
			'invoice_id' => \WC_Paytrace::get_order_number( $order ) . '_' . date( 'YmdHis' ),
			'email'      => \WC_Paytrace_Compat::get_order_billing_email( $order ),
		);
		
		if ( $is_paid_with_profile ) {
			// Get the saved card
			$order_tokens = new \WC_Paytrace_Order( $order );
			$customer_id  = $order_tokens->get_customer_id();
			
			$customer_tokens = new \WC_Paytrace_Customer_Tokens( $order->get_user_id() );
			$token           = $customer_tokens->get_token_by_customer_id( $customer_id );
			
			if ( ! $customer_id || ! $token ) {
				throw new \Exception( __( 'Customer profile not found.', \WC_Paytrace::TEXT_DOMAIN ) );
			}
			
			$request['authorization_type'] = $this->get_gateway()->get_transaction_authorization_type( 'Paytrace_eCheck' == $token->get_type() ? 'check' : 'card', $order );
			$request['transaction_type']   = 'Paytrace_eCheck' == $token->get_type() ? 'checks' : 'transactions';
			$request['customer_id']        = $token->get_token();
			
			// If the CVC is required, we want to send it every time the customer performs manual transaction with a card
			// If we are processing automatic(scheduled) payment, we don't want to send the CVC
			if ( 'yes' == $this->get_gateway()->get_option( 'require_cvc_with_saved_cards' )
			     && null !== \WC_Paytrace::get_field( 'paytrace-card-expiry', $_POST, null )
			     && ( 'saved_card' == $this->get_gateway()->get_used_payment_type()
			          || 'new_card' == $this->get_gateway()->get_used_payment_type() )
			) {
				$validator->set_credit_card_details();
				
				$cc_type = null;
				if ( '' === $this->get_gateway()->get_security_type() ) {
					$cc_type = $validator->get_card_type_from_number( $validator->cc_number );
				}
				
				if ( '' != $validator->cc_cvc || $this->get_gateway()->is_cvc_required() ) {
					$validator->validate_cvc_number( $validator->cc_cvc, $cc_type );
				}
				
				if ( '' != $validator->cc_cvc ) {
					$csc_field_name = 'csc';
					if ( 'encryption' === $this->get_gateway()->get_security_type() ) {
						$csc_field_name = 'encrypted_' . $csc_field_name;
					}
					
					$request[ $csc_field_name ] = $validator->cc_cvc;
				}
			}
		} else {
			// Set the cc or check fields
			if ( 'new_check' == $this->get_gateway()->get_used_payment_type() ) {
				// Card payments just run the set transaction type
				$request = $this->add_card_or_echeck_details( $request, $validator, 'check' );
				
				$request['authorization_type'] = $this->get_gateway()->get_transaction_authorization_type( 'check', $order );
				$request['transaction_type']   = 'checks';
			} else {
				$request = $this->add_card_or_echeck_details( $request, $validator, 'card' );
				
				$request['authorization_type'] = $this->get_gateway()->get_transaction_authorization_type( 'card', $order );
				$request['transaction_type']   = 'transactions';
			}
		}
		
		// Set the Billing address data
		$request = $this->add_billing_address_details_to_request( $request, $order );
		// Set the Shipping address data
		$request = $this->maybe_add_shipping_address_details_to_request( $request, $order );
		
		if ( $this->maybe_send_order_description() ) {
			// Get Order description
			$desc                   = $this->get_order_description( $order, $is_subscription );
			$request['description'] = $desc;
		}
		
		return $request;
	}
	
	/**
	 * Returns the refund request parameters
	 *
	 * @param \WC_Order $order
	 * @param null      $amount
	 * @param string    $reason
	 *
	 * @throws \Exception
	 * @return array
	 */
	public function get_refund_request( \WC_Order $order, $amount = null, $reason = '' ) {
		$order_tokens = new \WC_Paytrace_Order( $order );
		
		// Get the saved card
		$customer_id    = $order_tokens->get_customer_id();
		$transaction_id = $order_tokens->get_transaction_id();
		$payment_type   = $order_tokens->get_payment_type();
		
		// Bail, if there is no reference ID
		if ( '' == $customer_id && '' == $transaction_id ) {
			throw new \Exception( __( 'Error: Missing Reference ID. The order does not have all required information to process a refund.', \WC_Paytrace::TEXT_DOMAIN ) );
		}
		
		$request = array(
			'amount'  => $amount,
			'invoice' => uniqid( 'refund-' ),
		);
		
		// Are we refunding checks or cards
		if ( 'card' == $payment_type ) {
			$request['transaction_type'] = 'transactions';
		} else {
			$request['transaction_type'] = 'checks';
		}
		
		// Allows 3rd party to overwrite the behavior and use the "customer_id" for refunds
		if ( apply_filters( 'wc_paytrace_refund_with_transaction_id', true, $order, $amount ) && $transaction_id ) {
			// If there is no profile, we will use the transaction ID.
			if ( 'card' == $payment_type ) {
				$request['transaction_id'] = $transaction_id;
			} else {
				$request['check_transaction_id'] = $transaction_id;
			}
		} else {
			// If we have a profile saved we will use it in priority
			$token = false;
			if ( '' != $customer_id ) {
				$customer_tokens = new \WC_Paytrace_Customer_Tokens( $order->get_user_id() );
				$token           = $customer_tokens->get_token_by_customer_id( $customer_id );
			}
			
			if ( ! $token ) {
				throw new \Exception( __( 'Error: Missing token. The order does not have all required information to process a refund.', \WC_Paytrace::TEXT_DOMAIN ) );
			}
			
			// Set the customer ID to be refunded
			$request['customer_id'] = $token->get_token();
		}
		
		$request = $this->add_billing_address_details_to_request( $request, $order );
		
		return $request;
	}
	
	/**
	 * @param \WC_Order $order
	 * @param float     $amount
	 *
	 * @return array
	 * @throws \Exception
	 */
	public function get_capture_request( \WC_Order $order, $amount ) {
		$order_tokens = new \WC_Paytrace_Order( $order );
		
		// Get the saved card
		$transaction_id = $order_tokens->get_transaction_id();
		$payment_type   = $order_tokens->get_payment_type();
		
		if ( 'card' == $payment_type ) {
			$request['transaction_type'] = 'transactions';
			$request['transaction_id']   = $transaction_id;
		} else {
			$request['transaction_type']     = 'checks';
			$request['check_transaction_id'] = $transaction_id;
		}
		
		$request = $this->add_billing_address_details_to_request( $request, $order );
		
		$amount_authorized = $order_tokens->get_order_amount_authorized();
		if ( $amount != $amount_authorized ) {
			$request['amount'] = $amount;
		}
		
		return $request;
	}
	
	/**
	 * Exports a Paytrace profile
	 *
	 * @param        $customer_id
	 * @param string $customer_email
	 *
	 * @throws \Exception
	 * @return bool|mixed|\WcPaytrace\Api\Json\Response
	 */
	public function export_profile( $customer_id, $customer_email = '' ) {
		
		if ( '' == $customer_id ) {
			return false;
		}
		
		$services = $this->get_api_services();
		
		$props = array(
			'customer_id' => $customer_id,
			'include_bin' => true,
		);
		
		if ( '' != $customer_email ) {
			$props['email'] = $customer_email;
		}
		
		try {
			
			$props = apply_filters( 'wc_paytrace_export_profile_parameters', $props, $customer_id, $customer_email );
			
			// Debug
			$this->log_request( $props, 'Export Profile Request:' );
			
			$exported = $services->vault()->export_profile( $props );
			
			// Debug
			$this->log_request( $exported->get_accepted_response_body(), 'Export Profile Response:' );
			
			if ( $exported->did_error_occur() ) {
				return false;
			}
			
			$customers = $exported->get_customers();
			
			return $customers;
		}
		catch ( \Exception $e ) {
			return false;
		}
	}
	
	public function export_transaction( $transaction_id ) {
		if ( '' == $transaction_id ) {
			return false;
		}
		
		$services = $this->get_api_services();
		
		$props = array(
			'transaction_id' => $transaction_id,
			'include_bin'    => true,
		);
		
		try {
			
			$props = apply_filters( 'wc_paytrace_export_transaction_parameters', $props, $transaction_id );
			
			// Debug
			$this->log_request( $props, 'Export Profile Request:' );
			
			$exported = $services->transaction()->export_transaction( $props );
			
			// Debug
			$this->log_request( $exported->get_accepted_response_body(), 'Export Transaction Response:' );
			
			if ( $exported->did_error_occur() ) {
				return false;
			}
			
			$transactions = $exported->get_transactions();
			
			return $transactions;
		}
		catch ( \Exception $e ) {
			return false;
		}
	}
	
	/**
	 * Returns the parameters for the email receipt request
	 *
	 * @since 2.3.0
	 *
	 * @param \WC_Order $order
	 *
	 * @return mixed
	 */
	public function get_email_receipt_request( \WC_Order $order ) {
		$order_tokens = new \WC_Paytrace_Order( $order );
		
		// Get the saved card
		$transaction_id = $order_tokens->get_transaction_id();
		$payment_type   = $order_tokens->get_payment_type();
		
		if ( 'card' == $payment_type ) {
			$request['transaction_type'] = 'transactions';
			$request['transaction_id']   = $transaction_id;
		} else {
			$request['transaction_type']     = 'checks';
			$request['check_transaction_id'] = $transaction_id;
		}
		
		$request['email'] = \WC_Paytrace_Compat::get_order_billing_email( $order );
		
		return $request;
	}
}