<?php

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * WC_Paytrace_Gateway Class
 *
 * WC_Paytrace_Gateway manages the main functionality of the PayTrace extension.
 * Managed: Credit Card and Check payments, creation and management of customer payment profiles.
 *
 * @package PayTrace Gateway
 * @author  VanboDevelops
 */
class WC_Paytrace_Gateway extends WC_Payment_Gateway {
	
	/**
	 * Holds what type of payment the customer chose.
	 *
	 * Values: 'new_check', 'saved_check', 'new_card', 'saved_card'
	 * @var string
	 */
	public $used_payment_type;
	/**
	 * @var array
	 */
	public $available_cc;
	public $support_check;
	public $save_customers;
	public $save_card_text;
	/**
	 * @var array
	 */
	public $card_options;
	public $check_pass;
	public $user_name;
	public $password;
	public $testmode;
	public $json_use_encryption_js;
	public $debug;
	public $proxy;
	public $ach_payment_onhold;
	public $send_order_description;
	/**
	 * @var bool Should a transaction receipt be sent from Paytrace to the customer
	 */
	public $send_transaction_receipt;
	public $trans_type;
	public static $loaded;
	public $separated_forms;
	public $separated_forms_label_card;
	public $separated_forms_description_card;
	public $separated_forms_label_check;
	public $separated_forms_description_check;
	/**
	 * The integration class we are using to run processes
	 * @since 2.3.0
	 * @var \WcPaytrace\Integrations\Json\Integration_Json|\WcPaytrace\Integrations\Post\Integration_Post
	 */
	public $integration_class;
	
	public function __construct() {
		
		$this->id                 = 'paytrace';
		$this->icon               = apply_filters( 'woocommerce_paytrace_icon', plugin_dir_url( __FILE__ ) . '../paytrace_cc.png' );
		$this->has_fields         = true;
		$this->method_title       = __( 'PayTrace', WC_Paytrace::TEXT_DOMAIN );
		$this->method_description = __( "PayTrace Gateway allows you to accept credit/debit cards, Check/ACH Checks. Supports Subscription and Pre-order payments, stores safely cards and Check to PayTrace secure vault.", WC_Paytrace::TEXT_DOMAIN
		);
		
		$this->card_options = apply_filters(
			'paytrace_card_options', array(
				'VISA'     => __( 'Visa', WC_Paytrace::TEXT_DOMAIN ),
				'MC'       => __( 'Master Card', WC_Paytrace::TEXT_DOMAIN ),
				'AMEX'     => __( 'American Express', WC_Paytrace::TEXT_DOMAIN ),
				'DISCOVER' => __( 'Discover', WC_Paytrace::TEXT_DOMAIN ),
				'JCB'      => __( 'JCB', WC_Paytrace::TEXT_DOMAIN ),
			)
		);
		
		$this->check_pass = false;
		
		$this->supports = array(
			// Subscriptions
			'subscriptions',
			'subscription_cancellation',
			'subscription_reactivation',
			'subscription_suspension',
			'subscription_amount_changes',
			'subscription_payment_method_change_customer',
			'subscription_payment_method_change_admin',
			'subscription_date_changes',
			'multiple_subscriptions',
			
			// WC core support
			'products',
			
			// Refunds
			'refunds',
			
			// Tokenization
			'add_payment_method',
			'tokenization',
			
			// Pre-Orders
			'pre-orders',
		);
		
		// Load the form fields.
		$this->init_form_fields();
		
		// Load the settings.
		$this->init_settings();
		
		// Define user set variables
		$this->enabled                  = $this->get_option( 'enabled', 'no' );
		$this->title                    = $this->get_option( 'title' );
		$this->description              = $this->get_option( 'description' );
		$this->testmode                 = $this->get_option( 'testmode', 'no' );
		$this->user_name                = $this->get_option( 'user_name' );
		$this->password                 = $this->get_option( 'password' );
		$this->trans_type               = $this->get_option( 'trans_type', 'Sale' );
		$this->available_cc             = $this->get_option( 'available_cc' );
		$this->debug                    = 'yes' == $this->get_option( 'debug', 'no' );
		$this->save_customers           = 'yes' == $this->get_option( 'save_customers', 'yes' );
		$this->save_card_text           = $this->get_option( 'save_card_text' );
		$this->support_check            = 'yes' == $this->get_option( 'support_check', 'yes' );
		$this->ach_payment_onhold       = 'yes' == $this->get_option( 'ach_payment_onhold', 'yes' );
		$this->send_order_description   = 'yes' == $this->get_option( 'send_order_description', 'yes' );
		$this->send_transaction_receipt = 'yes' == $this->get_option( 'send_transaction_receipt', 'yes' );
		
		// Only JSON integration has this option
		$this->json_use_encryption_js = 'encryption' === $this->get_security_type();
		
		// Form separation variables
		$this->separated_forms                  = 'separate' == $this->get_option( 'separated_forms' );
		$this->separated_forms_label_card       = $this->get_option( 'separated_forms_label_card' );
		$this->separated_forms_description_card = $this->get_option( 'separated_forms_description_card' );
		if ( $this->support_check ) {
			$this->separated_forms_label_check       = $this->get_option( 'separated_forms_label_check' );
			$this->separated_forms_description_check = $this->get_option( 'separated_forms_description_check' );
		}
		
		// The title will be the card type setting since we are not using the general one
		if ( $this->separated_forms ) {
			$this->title = $this->separated_forms_label_card;
		}
		
		// Add Refunds support
		if ( $this->save_customers ) {
			$this->supports = array_merge(
				$this->supports,
				array(
					'refunds',
				)
			);
		}
		
		// Disable if currency is not supported
		if ( ! $this->is_valid_currency() ) {
			$this->enabled = 'no';
		}
		
		// Run hooks
		$this->hooks();
	}
	
	/**
	 * Returns the integration class to use for the plugin processes
	 * @since 2.3.0
	 * @return \WcPaytrace\Integrations\Json\Integration_Json|\WcPaytrace\Integrations\Post\Integration_Post
	 */
	public function get_integration_class() {
		if ( null == $this->integration_class ) {
			if ( 'post' == $this->get_option( 'integration' ) ) {
				$integration_class = '\\WcPaytrace\\Integrations\\Post\\Integration_Post';
			} else {
				$integration_class = '\\WcPaytrace\\Integrations\\Json\\Integration_Json';
			}
			
			// Allow extension of the integration classes. The plugin will automatically load the class returned by the filter
			$integration_class = apply_filters( 'wc_gateway_integration_class_paytrace', $integration_class, $this->get_option( 'integration' ) );
			
			$this->integration_class = new $integration_class( $this );
		}
		
		return $this->integration_class;
	}
	
	/**
	 * Loads the hooks of the gateway
	 *
	 * @since 2.0
	 */
	public function hooks() {
		if ( true === self::$loaded ) {
			return;
		}
		
		// Save options
		add_action( 'woocommerce_update_options_payment_gateways_' . $this->id, array(
			$this,
			'process_admin_options',
		) );
		
		// This is a custom action that will run the capture_payment process
		add_action( 'wc_paytrace_capture_payment_for_order', array( $this, 'capture_payment' ), 10, 2 );
		
		do_action( 'wc_paytrace_plugin_loaded', $this );
		
		self::$loaded = true;
	}
	
	/**
	 * Check if the store currency is supported by the gateway
	 */
	function is_valid_currency() {
		if ( ! in_array( get_woocommerce_currency(), $this->get_supported_currency() ) ) {
			return false;
		}
		
		return true;
	}
	
	/**
	 * Returns the plugin supported currency
	 * @since 2.3.0
	 * @return array|mixed
	 */
	public function get_supported_currency() {
		$currency = apply_filters( 'wc_paytrace_allowed_currency', array( 'USD' ) );
		
		if ( ! is_array( $currency ) ) {
			$currency = array( $currency );
		}
		
		return $currency;
	}
	
	/**
	 * Check if this gateway is enabled and available in the user's country
	 */
	public function is_available() {
		if ( 'yes' == $this->enabled ) {
			// Check if the required fields are set
			if ( ! $this->user_name ) {
				return false;
			}
			if ( ! $this->password ) {
				return false;
			}
			
			// Subs and Pre-Orders require Paytrace Vault
			$order = false;
			if ( WC_Paytrace::get_get( 'order_id' ) ) {
				$order = wc_get_order();
			}
			if ( ! $this->save_customers
			     && ( $this->order_or_cart_contains_subscription( $order ) || $this->order_or_cart_contains_pre_order( $order ) )
			) {
				return false;
			}
		}
		
		return 'yes' == $this->enabled;
	}
	
	/**
	 * Admin Panel Options
	 */
	public function admin_options() {
		wp_enqueue_script( 'paytrace-admin' );
		parent::admin_options();
		?>
		<style>
					.paytrace_warning {
						display: none;
						color: #fff;
						margin: 0;
						background: #dc3232;
						box-shadow: 0 1px 1px 0 rgba(0, 0, 0, 0.1);
						padding: 15px 10px;
					}
		</style>
		<?php
	} // End admin_options()
	
	/**
	 * Initialise Gateway Settings Form Fields
	 **/
	function init_form_fields() {
		
		// Get the integration type
		$options = get_option( 'woocommerce_' . $this->id . '_settings', array() );
		if ( ! empty( $options['integration'] ) ) {
			$integration = $options['integration'];
		} else {
			$integration = 'post';
		}
		
		// Get the integration specific settings
		if ( 'json' == $integration ) {
			$client = new \WcPaytrace\Integrations\Json\Settings( $this );
		} else {
			$client = new \WcPaytrace\Integrations\Post\Settings( $this );
		}
		
		$this->form_fields = array(
			'integration' => array(
				'title'             => __( 'Integration Type', WC_Paytrace::TEXT_DOMAIN ),
				'type'              => 'select',
				'description'       => __( 'Choose the API integration you want to connect and Save Settings.', WC_Paytrace::TEXT_DOMAIN ) . '<br/> <span class="paytrace_warning"><span>Integration Type changed!</span><span>Save the change before you continue.</span ></span>',
				'options'           => array(
					'json' => 'JSON API',
					'post' => 'Post API',
				),
				'class'             => 'chosen_select',
				'css'               => 'min-width:350px;',
				'default'           => 'json',
				'custom_attributes' => array(
					'data-initial-type' => $integration,
				),
			),
			
			'integration_type_settings_end' => array(
				'title'       => '<hr/>',
				'type'        => 'title',
				'description' => '',
			),
			
			'enabled' => array(
				'title'    => __( 'Enable/Disable', WC_Paytrace::TEXT_DOMAIN ),
				'type'     => 'checkbox',
				'label'    => __( 'Enable PayTrace', WC_Paytrace::TEXT_DOMAIN ),
				'desc_tip' => true,
				'default'  => 'no',
			),
			
			'title'                => array(
				'title'       => __( 'Method Title', WC_Paytrace::TEXT_DOMAIN ),
				'type'        => 'text',
				'description' => __( 'This controls the title which the user sees during checkout.', WC_Paytrace::TEXT_DOMAIN ),
				'default'     => __( 'Credit/Debit Card', WC_Paytrace::TEXT_DOMAIN ),
				'class'       => 'show_if_separated_forms_together',
			),
			'description'          => array(
				'title'       => __( 'Description', WC_Paytrace::TEXT_DOMAIN ),
				'type'        => 'textarea',
				'description' => __(
					'This controls the description which the user sees during checkout. Will show above the card fields.', WC_Paytrace::TEXT_DOMAIN
				),
				'default'     => __( "Pay with credit or debit card.", WC_Paytrace::TEXT_DOMAIN ),
				'class'       => 'show_if_separated_forms_together',
			
			),
			'general_settings_end' => array(
				'title'       => '<hr/>',
				'type'        => 'title',
				'description' => '',
			),
		);
		
		// Add the integration settings
		$this->form_fields += $client->get_settings();
		
		$this->form_fields += array(
			'integration_settings_end' => array(
				'title'       => '<hr/>',
				'type'        => 'title',
				'description' => '',
			),
			
			'debug_and_testing' => array(
				'title'       => __( 'Debug Settings', WC_Paytrace::TEXT_DOMAIN ),
				'type'        => 'title',
				'description' => '',
			),
			
			'debug' => array(
				'title'       => __( 'Debug', WC_Paytrace::TEXT_DOMAIN ),
				'type'        => 'checkbox',
				'label'       => __( 'Enable Debug Logging', WC_Paytrace::TEXT_DOMAIN ),
				'default'     => 'no',
				'description' => sprintf(
					__( 'Debug log will provide you with most of the data and events generated by the payment process. Logged inside %s' ),
					'<code>' . wc_get_log_file_path( 'paytrace' ) . '</code>'
				),
			),
		);
		
		// Allow 3rd party to add settings to the gateway
		$this->form_fields = apply_filters( 'wc_paytrace_settings', $this->form_fields, $this );
	}
	
	/**
	 * Filter the gateway icon according to the accepted cards option
	 *
	 * @version 2.4.0 - Utilize the get_payment_type_images
	 *
	 * @return string The Card images in a html format
	 */
	function get_icon() {
		if ( $this->available_cc ) {
			$icon = $this->get_payment_type_images( 'card' );
			$icon .= $this->get_payment_type_images( 'check' );
		} else {
			$icon = '<img src="' . esc_url( WC_HTTPS::force_https_url( $this->icon ) ) . '" alt="' . esc_attr( $this->title ) . '" />';
		}
		
		return $icon;
	}
	
	/**
	 * Returns the payment icons based on the payment type
	 *
	 * @since 2.4.0
	 *
	 * @param $type
	 *
	 * @return string
	 */
	public function get_payment_type_images( $type ) {
		
		if ( empty( $this->available_cc ) ) {
			return '';
		}
		
		$html = '';
		if ( 'check' == $type ) {
			if ( $this->support_check && file_exists( WC_Paytrace::plugin_path() . '/assets/images/echeck.svg' ) ) {
				$html .= '<img src="' . esc_url( WC_HTTPS::force_https_url( WC_Paytrace::plugin_url() . '/assets/images/echeck.svg' ) ) . '" alt="echeck" class="paytrace-icon paytrace-icon-echeck" />';
			} elseif ( $this->support_check && file_exists( WC_Paytrace::plugin_path() . '/assets/images/echeck.png' ) ) {
				$html .= '<img src="' . esc_url( WC_HTTPS::force_https_url( WC_Paytrace::plugin_url() . '/assets/images/echeck.png' ) ) . '" alt="echeck" class="paytrace-icon paytrace-icon-echeck" />';
			}
		} elseif ( 'card' == $type ) {
			foreach ( $this->available_cc as $card ) {
				$card_slug = strtolower( $card );
				
				if ( file_exists( WC_Paytrace::plugin_path() . '/assets/images/' . $card_slug . '.svg' ) ) {
					$html .= '<img src="' . esc_url( WC_HTTPS::force_https_url( WC_Paytrace::plugin_url() . '/assets/images/' . $card_slug . '.svg' ) ) . '" alt="' . esc_attr( $card_slug ) . '" class="paytrace-icon paytrace-icon-' . $card_slug . '" />';
				} elseif ( file_exists( WC_Paytrace::plugin_path() . '/assets/images/' . $card_slug . '.png' ) ) {
					$html .= '<img src="' . esc_url( WC_HTTPS::force_https_url( WC_Paytrace::plugin_url() . '/assets/images/' . $card_slug . '.png' ) ) . '" alt="' . esc_attr( $card_slug ) . '" class="paytrace-icon paytrace-icon-' . $card_slug . '" />';
				}
			}
		}
		
		return apply_filters( 'wc_get_payment_type_images_paytrace', $html, $type );
	}
	
	/**
	 * Validate payment fields
	 *
	 * @return bool
	 */
	function validate_fields() {
		try {
			do_action( 'wc_validate_fields_paytrace' );
			
			// Check payment info before it is send to the gateway
			$this->validate_payment_information();
			
			// Note the credit card fields check was passed
			$this->check_pass = true;
			
			return true;
		}
		catch ( Exception $e ) {
			wc_add_notice( $e->getMessage(), 'error' );
			
			return false;
		}
	}
	
	/**
	 * Show PayTrace payment fields
	 */
	function payment_fields() {
		$description = $this->get_description();
		if ( $description ) {
			echo wpautop( wptexturize( $description ) );
		}
		
		if ( 'encryption' === $this->get_security_type() ) {
			wp_enqueue_script( 'paytrace-e2ee' );
		} elseif ( 'protect' === $this->get_security_type() ) {
			wp_enqueue_script( 'paytrace-protect' );
		}
		
		$show_save_customer = false;
		$order              = wc_get_order( WC_Paytrace::get_get( 'order_id' ) );
		
		if ( $this->save_customers && ! is_add_payment_method_page() ) {
			$show_save_customer = $this->maybe_show_save_customer_checkbox( $order );
		}
		
		$form = new WC_Paytrace_Payment_Form( $this );
		$form->payment_fields( $show_save_customer );
	}
	
	/**
	 * Processes the payment
	 *
	 * @param int $order_id
	 *
	 * @return array
	 */
	function process_payment( $order_id ) {
		try {
			if ( ! $this->check_pass ) {
				// Check payment info before it is send to the gateway
				$this->validate_payment_information();
			}
			
			$order = wc_get_order( $order_id );
			
			\WC_Paytrace::add_debug_log( 'process_payment: initiate for order: ' . $order_id );
			
			$integration = $this->get_integration_class();
			$integration->process_payment( $order );
			
			$this->send_transaction_receipt_action( $order );
			
			// Empty cart
			wc_empty_cart();
			
			// Redirect to the thank you page
			return array(
				'result'   => 'success',
				'redirect' => $this->get_return_url( $order ),
			);
		}
		catch ( Exception $e ) {
			\WC_Paytrace::add_debug_log( 'process_payment: something went wrong. Error: ' . $e->getMessage() );
			
			wc_add_notice( $e->getMessage(), 'error' );
			
			if ( 'protect' == $this->get_security_type() ) {
				WC()->session->refresh_totals = true;
			}
		}
	}
	
	/**
	 * Process automatic refunds
	 *
	 * @since 1.3
	 *
	 * @param int    $order_id
	 * @param null   $amount
	 * @param string $reason
	 *
	 * @return bool|WP_Error
	 */
	public function process_refund( $order_id, $amount = null, $reason = '' ) {
		try {
			$order = wc_get_order( $order_id );
			
			$integration = $this->get_integration_class();
			
			return $integration->process_refund( $order, $amount, $reason );
		}
		catch ( Exception $ex ) {
			return new WP_Error( 'paytrace-error', $ex->getMessage() );
		}
	}
	
	/**
	 * Adds Email Receipt action to the Order actions
	 *
	 * @since 2.3.0
	 *
	 * @param $actions
	 *
	 * @return mixed
	 */
	public function add_order_email_receipt_action( $actions ) {
		/**
		 * @var WC_Order $theorder
		 */
		global $theorder;
		
		$method = WC_Paytrace_Compat::get_prop( $theorder, 'payment_method' );
		if ( $this->id != $method ) {
			return $actions;
		}
		
		$tokens         = new WC_Paytrace_Order( $theorder );
		$transaction_id = $tokens->get_transaction_id();
		
		if ( '' == $transaction_id || ! $this->send_transaction_receipt ) {
			return $actions;
		}
		
		$actions['paytrace_email_receipt'] = __( 'Email Paytrace Transaction Receipt', WC_Paytrace::TEXT_DOMAIN );
		
		return $actions;
	}
	
	/**
	 * Sends a transaction receipt request to Paytrace
	 *
	 * @since 2.3.0
	 *
	 * @param \WC_Order $order
	 */
	public function send_transaction_receipt_action( $order ) {
		// Allow for additional check and permissions on a per order basis
		if ( $this->send_transaction_receipt && apply_filters( 'wc_paytrace_should_send_transaction_receipt', true, $order ) ) {
			$integration = $this->get_integration_class();
			$integration->send_transaction_receipt( $order );
		}
	}
	
	/**
	 * Adds capture payment action to the order actions
	 *
	 * @param $actions
	 *
	 * @return mixed
	 */
	public function add_order_capture_action( $actions ) {
		
		/**
		 * @var \WC_Order $theorder
		 */
		global $theorder;
		
		$method = WC_Paytrace_Compat::get_prop( $theorder, 'payment_method' );
		if ( $this->id != $method ) {
			return $actions;
		}
		
		$tokens                 = new WC_Paytrace_Order( $theorder );
		$captured               = $tokens->get_is_payment_captured();
		$allowed_order_statuses = \WcPaytrace\Admin\Capture::get_capture_allowed_order_statuses();
		
		if ( false != $captured || ! in_array( $theorder->get_status(), $allowed_order_statuses ) ) {
			return $actions;
		}
		
		$paytrace_order    = new WC_Paytrace_Order( $theorder );
		$authorized_amount = wc_format_decimal( $paytrace_order->get_order_amount_authorized() );
		$total             = $theorder->get_total();
		if ( ! empty( $authorized_amount ) && $total > $authorized_amount ) {
			$total = $authorized_amount;
		}
		
		$actions['paytrace_capture_payment'] = __( 'Capture Payment (' . get_woocommerce_currency_symbol( WC_Paytrace_Compat::get_order_currency( $theorder ) ) . $total . ')', WC_Paytrace::TEXT_DOMAIN );
		
		return $actions;
	}
	
	/**
	 * @param WC_Order   $order
	 * @param float|null $amount
	 *
	 * @return bool
	 */
	public function capture_payment( $order, $amount = null ) {
		$paytrace_order = new WC_Paytrace_Order( $order );
		$is_captured    = $paytrace_order->get_is_payment_captured();
		
		// Bail, if we captured the amount already
		if ( false != $is_captured ) {
			return false;
		}
		
		try {
			/**
			 * !!! IMPORTANT: Only one capture is allowed per authorization
			 * Add a field to enter a price and give the explanation that the amount to capture is one and done
			 */
			
			// Get the saved card
			$transaction_id = $paytrace_order->get_transaction_id();
			
			// Bail, if there is no reference ID
			if ( '' == $transaction_id ) {
				throw new \Exception( __( 'Missing Transaction ID. The order does not have all required information to process process a capture.', \WC_Paytrace::TEXT_DOMAIN ) );
			}
			
			// If no amount, use the order total
			if ( null === $amount ) {
				$amount = $order->get_total();
			}
			
			// Get the initially authorized amount.
			$authorized_amount = wc_format_decimal( $paytrace_order->get_order_amount_authorized() );
			if ( empty( $authorized_amount ) ) {
				$authorized_amount = $order->get_total();
			}
			
			// Can't capture more than the initially authorized amount
			if ( $authorized_amount < $amount ) {
				throw new \Exception( sprintf( __( "You can't capture more than the initially authorized amount: %s", \WC_Paytrace::TEXT_DOMAIN ), get_woocommerce_currency_symbol() . $authorized_amount ) );
			}
			
			$integration = $this->get_integration_class();
			
			$response = $integration->process_capture( $order, $amount );
			
			if ( $response->did_error_occur() ) {
				throw new \Exception( $response->get_error() );
			}
			
			if ( $response->was_capture_approved() ) {
				switch ( (int) $response->get_response_code() ) {
					case '112' :
					case '124' :
						// Debug log
						\WC_Paytrace::add_debug_log( 'Capture completed.' );
						
						// Add order note
						$order->add_order_note( sprintf( __( 'Amount captured %s. Transaction ID: %s.', \WC_Paytrace::TEXT_DOMAIN ),
							get_woocommerce_currency_symbol() . wc_format_decimal( $amount ),
							$response->get_transaction_id()
						) );
						
						$paytrace_order->save_transaction_id( $response->get_transaction_id() );
						$paytrace_order->save_is_payment_captured( true );
						$paytrace_order->save_order_amount_captured( $amount );
						
						do_action( 'wc_paytrace_capture_approved_response_processed', $response, $order, $amount );
						
						return true;
						
						break;
					default :
						// Add order note
						$error_message = $response->get_error();
						
						do_action( 'wc_paytrace_capture_declined_response_processed', $response, $order, $amount );
						
						throw new Exception( $error_message );
						break;
				}
			} else {
				do_action( 'wc_paytrace_capture_declined_response_processed', $response, $order, $amount );
				
				throw new \Exception( $response->get_response() );
			}
		}
		catch ( Exception $e ) {
			$order->add_order_note( sprintf( __( 'Capture failed. Error Message: %s', WC_Paytrace::TEXT_DOMAIN ), $e->getMessage() ) );
		}
	}
	
	/**
	 * Add payment method via account screen.
	 * @since 2.1
	 */
	public function add_payment_method() {
		if ( ! is_user_logged_in() ) {
			wc_add_notice( __( 'There was a problem adding the card.', WC_Paytrace::TEXT_DOMAIN ), 'error' );
			
			return;
		}
		try {
			$integration = $this->get_integration_class();
			
			$type = 'new_check' == $this->get_used_payment_type() ? 'check' : 'card';
			
			$response = $integration->create_profile( get_current_user_id(), $type );
			
			// Init the class
			$services  = $integration->get_api_services();
			$validator = $services->get_validator();
			
			if ( $response->did_error_occur() ) {
				// Show error to the customer
				throw new \Exception(
					sprintf(
						__(
							'Error occurred while creating the payment profile.'
							. ' Error message: %s', \WC_Paytrace::TEXT_DOMAIN
						),
						$response->get_error()
					)
				);
			}
			
			if ( $response->was_customer_request_approved() ) {
				switch ( (int) $response->get_response_code() ) :
					case '160' : // Profile created
						// Profile was created, save the info to the customer user data
						
						if ( 'check' == $type ) {
							$validator->set_echeck_details();
							$profile = array(
								'routing_last4' => substr( $validator->echeck_routing, - 4 ),
								'last4'         => substr( $validator->echeck_account, - 4 ),
								'type'          => 'Paytrace_eCheck',
							);
						} else {
							$validator->set_credit_card_details();
							$last4     = substr( $validator->cc_number, - 4 );
							$exp_month = $validator->cc_exp_month;
							$exp_year  = $validator->cc_exp_year;
							$card_type = $validator->get_card_type_from_number( $validator->cc_number );
							
							// TODO: How to get the last4 and card type in protect.js
							if ( 'encryption' === $this->get_security_type() ) {
								$lookup = $integration->export_profile( $response->get_customer_id() );
								
								if ( ! empty( $lookup ) && is_array( $lookup ) ) {
									$looked_customer = array_shift( $lookup );
									$card            = WC_Paytrace::get_field( 'credit_card', $looked_customer, false );
									if ( $card ) {
										$last4     = substr( WC_Paytrace::get_field( 'masked_number', $card, '' ), - 4 );
										$card_type = $validator->get_card_type_from_first_six( WC_Paytrace::get_field( 'masked_number', $card, '' ) );
									}
								}
							}
							
							if ( '' !== $this->get_security_type() ) {
								$lookup = $integration->export_profile( $response->get_customer_id() );
								if ( ! empty( $lookup ) && is_array( $lookup ) ) {
									$looked_customer = array_shift( $lookup );
									$card            = \WC_Paytrace::get_field( 'credit_card', $looked_customer, false );
									if ( $card ) {
										$last4     = substr( \WC_Paytrace::get_field( 'masked_number', $card, '' ), - 4 );
										$card_type = $validator->get_card_type_from_first_six( \WC_Paytrace::get_field( 'masked_number', $card, '' ) );
										if ( 'protect' == $this->get_security_type() ) {
											$exp_month = \WC_Paytrace::get_field( 'expiration_month', $card, '' );
											$exp_year  = substr( \WC_Paytrace::get_field( 'expiration_year', $card, '' ), - 2 );
										}
									}
								}
							}
							
							$profile = array(
								'last4'     => $last4,
								'exp_year'  => $exp_year,
								'exp_month' => $exp_month,
								'type'      => 'Paytrace_CC',
								'card_type' => $card_type,
							);
						}
						
						$profile['token']      = $response->get_customer_id();
						$profile['profile_id'] = $response->get_profile_id();
						$new_token             = new \WC_Paytrace_Token( $profile, 0 );
						
						$customer_tokens = new \WC_Paytrace_Customer_Tokens( get_current_user_id() );
						$customer_tokens->save_profile_or_token( $new_token );
						
						break;
					default : // Any other code did not create customer
						
						// Show error to the customer
						throw new \Exception(
							sprintf(
								__(
									'Creating your payment profile failed.'
									. ' Error message: %s', \WC_Paytrace::TEXT_DOMAIN
								),
								$response->get_response()
							)
						);
						
						break;
				endswitch;
			}
		}
		catch ( Exception $e ) {
			wc_add_notice( $e->getMessage(), 'error' );
			
			return;
		}
		
		return array(
			'result'   => 'success',
			'redirect' => wc_get_endpoint_url( 'payment-methods' ),
		);
	}
	
	/**
	 * Update a customer profile from both PayTrace and store systems.
	 *
	 * @param $update_card_id
	 *
	 * @throws Exception
	 */
	function update_customer_profile( $update_card_id ) {
		
		$integration = $this->get_integration_class();
		
		$integration->process_profile_update( $update_card_id );
	}
	
	/**
	 * Delete a customer profile from both PayTrace and store systems.
	 *
	 * @param string $delete_card_id The ID of the save, in the database, profile
	 * @param int    $user_id        The user the token belongs to
	 *
	 * @throws Exception
	 */
	function delete_customer_profile( $delete_card_id, $user_id = 0 ) {
		
		$integration = $this->get_integration_class();
		
		$integration->process_profile_delete( $delete_card_id, $user_id );
	}
	
	/**
	 * Validate the received payment information,
	 * before it is passed to the payment gateway.
	 *
	 * @throws Exception
	 * @return void
	 */
	function validate_payment_information() {
		$integration = $this->get_integration_class();
		
		$validator = $integration->get_api_services()->get_validator();
		$validator->validate_submitted_fields();
	}
	
	/**
	 * Returns the payment type used during checkout
	 *
	 * TODO: Move the the Integration or Process classes
	 *
	 * @since 2.0
	 *
	 * @return string
	 */
	public function get_used_payment_type() {
		
		// Short it, if we already computed the used payment type
		if ( null !== $this->used_payment_type ) {
			return $this->used_payment_type;
		}
		
		if ( \WC_Paytrace::get_post( 'paytrace_type_choice' ) == 'check' ) {
			$this->used_payment_type = 'new_check';
			if ( $this->save_customers
			     && ( 'new' !== $this->get_submitted_token_value( 'check' ) && is_user_logged_in() )
			) {
				$this->used_payment_type = 'saved_check';
			}
		} else {
			$this->used_payment_type = 'new_card';
			if ( $this->save_customers
			     && ( 'new' !== $this->get_submitted_token_value( 'card' ) && is_user_logged_in() )
			) {
				$this->used_payment_type = 'saved_card';
			}
		}
		
		return $this->used_payment_type;
	}
	
	/**
	 * Retrieves the submitted token value.
	 * Checks the token type before it retrieves the value
	 *
	 * @param string $type
	 * @param string $default
	 *
	 * @return string
	 */
	public function get_submitted_token_value( $type = 'card', $default = '' ) {
		$allowed = array(
			'card',
			'check',
		);
		
		$type = in_array( $type, $allowed ) ? $type : 'card';
		
		return \WC_Paytrace::get_field( 'wc-paytrace-payment-' . $type . '-token', $_POST, $default );
	}
	
	/**
	 * Validate Password Field.
	 *
	 * Make sure the data is escaped correctly, etc.
	 * We are not showing the password value to the front end,
	 * so we will overwrite the password validation, so we can update the password only when it is not empty.
	 * If left empty the password will be saved with the old value.
	 *
	 * @since 1.2
	 *
	 * @param mixed $key
	 *
	 * @return string
	 */
	public function validate_password_field( $key, $value = '' ) {
		$text      = $this->get_option( $key );
		$gen_field = $this->plugin_id . $this->id . '_' . $key;
		
		if ( isset( $_POST[ $gen_field ] ) && '' !== $_POST[ $gen_field ] ) {
			$text = wc_clean( stripslashes( trim( $_POST[ $gen_field ] ) ) );
			
			// If we saved a new password, reset the json access token
			delete_transient( 'wc_paytrace_json_access_token' );
		}
		
		return $text;
	}
	
	/**
	 * Generate Text Input HTML.
	 * Modify the text html to remove the password value from the front end.
	 *
	 * @since 1.2
	 *
	 * @param mixed $key
	 * @param mixed $data
	 *
	 * @return string
	 */
	public function generate_text_html( $key, $data ) {
		$field    = $this->plugin_id . $this->id . '_' . $key;
		$defaults = array(
			'title'             => '',
			'disabled'          => false,
			'class'             => '',
			'css'               => '',
			'placeholder'       => '',
			'type'              => 'text',
			'desc_tip'          => false,
			'description'       => '',
			'custom_attributes' => array(),
		);
		
		$data  = wp_parse_args( $data, $defaults );
		$value = $this->get_option( $key );
		
		// Passwords will not have the exact password displayed,
		// so lets add a placeholder letting the user know that the password is set or not.
		$is_password_field = 'password' == $data['type'];
		if ( $is_password_field ) {
			$data['placeholder'] = _x( 'Password is not set', 'password field placeholder', WC_Paytrace::TEXT_DOMAIN );
			if ( '' != $value ) {
				$data['placeholder'] = _x( 'Password is set', 'password field placeholder', WC_Paytrace::TEXT_DOMAIN );
			}
			$value = '';
		}
		
		ob_start();
		?>
		<tr valign="top">
			<th scope="row" class="titledesc">
				<label for="<?php echo esc_attr( $field ); ?>">
					<?php echo wp_kses_post( $data['title'] ); ?>
					<?php echo $this->get_tooltip_html( $data ); ?>
				</label>
			</th>
			<td class="forminp">
				<fieldset>
					<legend class="screen-reader-text">
						<span><?php echo wp_kses_post( $data['title'] ); ?></span></legend>
					<input class="input-text regular-input <?php echo esc_attr( $data['class'] ); ?>"
					       type="<?php echo esc_attr( $data['type'] ); ?>"
					       name="<?php echo esc_attr( $field ); ?>"
					       id="<?php echo esc_attr( $field ); ?>"
					       style="<?php echo esc_attr( $data['css'] ); ?>"
					       value="<?php echo esc_attr( $value ); ?>"
					       placeholder="<?php echo esc_attr( $data['placeholder'] ); ?>"
						<?php disabled( $data['disabled'], true ); ?> <?php echo $this->get_custom_attribute_html( $data ); ?> />
					<?php echo $this->get_description_html( $data ); ?>
				</fieldset>
			</td>
		</tr>
		<?php
		return ob_get_clean();
	}
	
	/**
	 * Determines if we are to show the save card checkbox
	 *
	 * @param $order
	 *
	 * @since 2.0
	 *
	 * @return bool
	 */
	public function maybe_show_save_customer_checkbox( $order ) {
		$show_save_customer = true;
		if ( $this->order_or_cart_contains_subscription( $order ) || $this->is_change_payment_method( $order ) ) {
			// We always save customers in Subscriptions or Pre-orders
			// No need to show the option
			$show_save_customer = false;
		} elseif (
			$this->order_or_cart_contains_pre_order( $order )
		) {
			$show_save_customer = false;
		}
		
		return $show_save_customer;
	}
	
	/**
	 * Returns true if the order or cart contains subscription
	 *
	 * @since 2.0
	 *
	 * @param WC_Order|bool $order
	 *
	 * @return bool
	 */
	public function order_or_cart_contains_subscription( $order = false ) {
		if ( ! class_exists( 'WC_Subscriptions_Cart' ) ) {
			return false;
		}
		
		if ( ! WC_Paytrace::is_subscriptions_active() ) {
			return false;
		}
		
		if ( false === $order || ! $order instanceof WC_Order ) {
			// If no order, check the Cart
			return WC_Subscriptions_Cart::cart_contains_subscription();
		}
		
		$paytrace_order = new WC_Paytrace_Order( $order );
		
		// Order contains Pre-orders
		return $paytrace_order->contains_subscription();
	}
	
	/**
	 * Returns true if the order or cart contains subscription
	 *
	 * @since 2.0
	 *
	 * @param WC_Order|bool $order
	 *
	 * @return bool
	 */
	public function order_or_cart_contains_pre_order( $order = false ) {
		if ( ! class_exists( 'WC_Pre_Orders_Cart' ) ) {
			return false;
		}
		
		if ( ! WC_Paytrace::is_pre_orders_active() ) {
			return false;
		}
		
		if ( false === $order || ! $order instanceof WC_Order ) {
			// If no order, check the Cart
			return WC_Pre_Orders_Cart::cart_contains_pre_order();
		}
		
		$paytrace_order = new WC_Paytrace_Order( $order );
		
		// Order contains Pre-orders
		return $paytrace_order->contains_pre_order();
	}
	
	/**
	 * Returns the payment transaction type.
	 * If pay_by is echeck, Authorization needs to be 'Hold' and Sale is 'Sale'
	 * If pay_by is card, Authorization needs to be 'Authorization' and Sale is 'Sale'
	 *
	 * @since 2.0
	 *
	 * @param string    $pay_by
	 * @param \WC_Order $order
	 *
	 * @return mixed|string
	 */
	public function get_transaction_authorization_type( $pay_by = 'card', $order = false ) {
		if ( 'check' == $pay_by ) {
			return ( 'Authorization' == $this->trans_type ) ? 'Hold' : 'Sale';
		}
		
		return apply_filters( 'wc_paytrace_transaction_authorization_type', $this->trans_type, $order );
	}
	
	/**
	 * Returns true, if order contains Subscription
	 *
	 * @since 1.4
	 *
	 * @param WC_Order $order
	 *
	 * @return bool
	 */
	public function order_contains_subscription( WC_Order $order ) {
		return false;
	}
	
	/**
	 * Returns true, if order contains Pre-Order
	 *
	 * @since 1.4
	 *
	 * @param WC_Order $order
	 *
	 * @return bool
	 */
	public function order_contains_pre_order( WC_Order $order ) {
		return false;
	}
	
	/**
	 * @param $subscription
	 *
	 * @return bool
	 */
	public function is_change_payment_method( $subscription ) {
		return false;
	}
	
	/**
	 * @return bool
	 */
	public function is_cvc_required() {
		return (bool) apply_filters( 'wc_paytrace_is_cvc_required', false );
	}
	
	public function get_security_type() {
		if ( 'post' === $this->get_option( 'integration' ) ) {
			return '';
		}
		
		return $this->get_option( 'json_security_type', 'protect' );
	}
} // End PayTrace class