<?php
/*
 * Class to ensure compatibility in the transition of WC from v2.4.0 to v3.0.0
 *
 * Version 1.9.0
 * Author: VanboDevelops
 * Author URI: http://www.vanbodevelops.com
 *
 *	Copyright: (c) 2012 - 2019 VanboDevelops
 *	License: GNU General Public License v3.0
 *	License URI: http://www.gnu.org/licenses/gpl-3.0.html
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

class WC_Paytrace_Compat {
	
	/**
	 * Is WC 2.4+
	 * @var bool
	 */
	public static $is_wc_2_4;
	/**
	 * Is WC 2.5+
	 * @var bool
	 */
	public static $is_wc_2_5;
	/**
	 * Is WC 2.6+
	 * @var bool
	 */
	public static $is_wc_2_6;
	/**
	 * Is WC 3.0+
	 * @var bool
	 */
	public static $is_wc_3_0;
	/**
	 * @var bool
	 */
	public static $is_wc_3_1_2;
	/**
	 * @var bool
	 */
	public static $is_wc_4_0;
	
	public static function equal_or_grt( $version ) {
		return version_compare( self::get_wc_version_constant(), $version, '>=' );
	}
	
	public static function is_grt( $version ) {
		return version_compare( self::get_wc_version_constant(), $version, '>' );
	}
	
	/**
	 * Detect, if we are using WC 2.4+
	 *
	 * @since 1.3.2
	 * @return bool
	 */
	public static function is_wc_2_4() {
		if ( is_bool( self::$is_wc_2_4 ) ) {
			return self::$is_wc_2_4;
		}
		
		return self::$is_wc_2_4 = self::equal_or_grt( '2.4.0' );
	}
	
	public static function is_wc_2_5() {
		if ( is_bool( self::$is_wc_2_5 ) ) {
			return self::$is_wc_2_5;
		}
		
		return self::$is_wc_2_5 = self::equal_or_grt( '2.5.0' );
	}
	
	public static function is_wc_2_6() {
		if ( is_bool( self::$is_wc_2_6 ) ) {
			return self::$is_wc_2_6;
		}
		
		return self::$is_wc_2_6 = self::equal_or_grt( '2.6.0' );
	}
	
	public static function is_wc_3_0() {
		if ( is_bool( self::$is_wc_3_0 ) ) {
			return self::$is_wc_3_0;
		}
		
		return self::$is_wc_3_0 = self::equal_or_grt( '3.0.0' );
	}
	
	public static function is_wc_3_1_2() {
		if ( is_bool( self::$is_wc_3_1_2 ) ) {
			return self::$is_wc_3_1_2;
		}
		
		return self::$is_wc_3_1_2 = self::equal_or_grt( '3.1.2' );
	}
	
	public static function is_wc_4_0() {
		if ( is_bool( self::$is_wc_4_0 ) ) {
			return self::$is_wc_4_0;
		}
		
		return self::$is_wc_4_0 = self::equal_or_grt( '4.0.0' );
	}
	
	/**
	 * Get the Gateway settings page
	 *
	 * @param string $class_name
	 *
	 * @return string Formatted URL
	 */
	public static function gateway_settings_page( $class_name ) {
		$page    = 'wc-settings';
		$tab     = 'checkout';
		$section = strtolower( $class_name );
		
		return admin_url( 'admin.php?page=' . $page . '&tab=' . $tab . '&section=' . $section );
	}
	
	/**
	 * Get the WC logger object
	 *
	 * @return \WC_Logger
	 */
	public static function get_wc_logger() {
		if ( self::is_wc_3_0() ) {
			return wc_get_logger();
		} else {
			return new WC_Logger();
		}
	}
	
	/**
	 * Empty the user cart session
	 *
	 * TODO: Remove, no longer used
	 */
	public static function empty_cart() {
		if ( self::is_wc_2_5() ) {
			wc_empty_cart();
		} else {
			WC()->cart->empty_cart();
		}
	}
	
	/**
	 * Get Order shipping total
	 *
	 * TODO: Remove, no longer used
	 *
	 * @param WC_Order $order
	 *
	 * @return double
	 */
	public static function get_total_shipping( WC_Order $order ) {
		if ( self::is_wc_3_0() ) {
			return $order->get_shipping_total();
		} else {
			return $order->get_total_shipping();
		}
	}
	
	/**
	 * Get My Account URL
	 *
	 * @return string Formatted URL string
	 */
	public static function get_myaccount_url() {
		return wc_get_page_permalink( 'myaccount' );
	}
	
	/**
	 * Get WC version constant.
	 *
	 * @return string|null
	 */
	public static function get_wc_version_constant() {
		if ( defined( 'WC_VERSION' ) && WC_VERSION ) {
			return WC_VERSION;
		}
		
		return null;
	}
	
	/**
	 * Get WC order currency
	 *
	 * @since   1.0.3
	 * @updated 1.6
	 *
	 * @param WC_Order $order
	 *
	 * @return int
	 */
	public static function get_order_currency( WC_Order $order ) {
		if ( self::is_wc_3_0() ) {
			return $order->get_currency();
		} else {
			return $order->get_order_currency();
		}
	}
	
	/**
	 * Returns the product from the order item.
	 *
	 * @since 1.6
	 *
	 * @param array|WC_Order_Item_Product $item
	 * @param bool|WC_Order               $order
	 *
	 * @return WC_Product
	 */
	public static function get_product_from_item( $item, $order = false ) {
		if ( self::is_wc_3_0() && is_callable( array( $item, 'get_product' ) ) ) {
			return $item->get_product();
		} else {
			return $order->get_product_from_item( $item );
		}
	}
	
	/**
	 * Returns the order item meta for gateway display
	 *
	 * @since 1.6
	 *
	 * @param $item
	 *
	 * @return string|WC_Order_Item_Meta
	 */
	public static function wc_display_item_meta( $item ) {
		if ( self::is_wc_3_0() ) {
			$item_meta = strip_tags( wc_display_item_meta( $item, array(
				'before'    => "",
				'separator' => ", ",
				'after'     => "",
				'echo'      => false,
				'autop'     => false,
			) ) );
		} else {
			$meta = new WC_Order_Item_Meta( $item );
			
			$item_meta = $meta->display( true, true );
		}
		
		return $item_meta;
	}
	
	/**
	 * Returns the item name
	 *
	 * @since 1.6
	 *
	 * @param array|WC_Order_item $item
	 *
	 * @return mixed
	 */
	public static function get_item_name( $item ) {
		if ( self::is_wc_3_0() && is_callable( array( $item, 'get_name' ) ) ) {
			$name = $item->get_name();
		} else {
			$name = $item['name'];
		}
		
		return $name;
	}
	
	/**
	 * Returns the item quantity
	 *
	 * @since 1.6
	 *
	 * @param array|WC_Order_item $item
	 *
	 * @return mixed
	 */
	public static function get_item_quantity( $item ) {
		if ( self::is_wc_3_0() && is_callable( array( $item, 'get_quantity' ) ) ) {
			$name = $item->get_quantity();
		} else {
			$name = $item['qty'];
		}
		
		return $name;
	}
	
	/**
	 * Returns an order property
	 *
	 * @since 1.6.1
	 *
	 * @param WC_Order $order
	 *
	 * @return mixed
	 */
	public static function get_order_prop( $order, $name ) {
		if ( self::is_wc_3_0() && is_callable( array( $order, 'get_' . $name ) ) ) {
			$method = 'get_' . $name;
			
			return $order->{$method}();
		}
		
		return $order->{$name};
	}
	
	/**
	 * Returns an order property
	 *
	 * @since 1.6.2
	 *
	 * @param object $object
	 *
	 * @return mixed
	 */
	public static function get_prop( $object, $name ) {
		if ( self::is_wc_3_0() && is_callable( array( $object, 'get_' . $name ) ) ) {
			$method = 'get_' . $name;
			
			return $object->{$method}();
		}
		
		return $object->{$name};
	}
	
	/**
	 * Returns the order ID.
	 *
	 * @since 1.6
	 *
	 * @param WC_Order $order
	 *
	 * @return mixed
	 */
	public static function get_order_id( $order ) {
		return self::get_order_prop( $order, 'id' );
	}
	
	/**
	 * Returns the order billing .
	 *
	 * @since 1.6
	 *
	 * @param WC_Order $order
	 *
	 * @return mixed
	 */
	public static function get_order_billing_first_name( $order ) {
		return self::get_order_prop( $order, 'billing_first_name' );
	}
	
	/**
	 * Returns the order billing .
	 *
	 * @since 1.6
	 *
	 * @param WC_Order $order
	 *
	 * @return mixed
	 */
	public static function get_order_billing_last_name( $order ) {
		return self::get_order_prop( $order, 'billing_last_name' );
	}
	
	/**
	 * Returns the order billing .
	 *
	 * @since 1.6
	 *
	 * @param WC_Order $order
	 *
	 * @return mixed
	 */
	public static function get_order_billing_company( $order ) {
		return self::get_order_prop( $order, 'billing_company' );
	}
	
	/**
	 * Returns the order billing .
	 *
	 * @since 1.6
	 *
	 * @param WC_Order $order
	 *
	 * @return mixed
	 */
	public static function get_order_billing_address_1( $order ) {
		return self::get_order_prop( $order, 'billing_address_1' );
	}
	
	/**
	 * Returns the order billing .
	 *
	 * @since 1.6
	 *
	 * @param WC_Order $order
	 *
	 * @return mixed
	 */
	public static function get_order_billing_address_2( $order ) {
		return self::get_order_prop( $order, 'billing_address_2' );
	}
	
	/**
	 * Returns the order billing .
	 *
	 * @since 1.6
	 *
	 * @param WC_Order $order
	 *
	 * @return mixed
	 */
	public static function get_order_billing_city( $order ) {
		return self::get_order_prop( $order, 'billing_city' );
	}
	
	/**
	 * Returns the order billing .
	 *
	 * @since 1.6
	 *
	 * @param WC_Order $order
	 *
	 * @return mixed
	 */
	public static function get_order_billing_postcode( $order ) {
		return self::get_order_prop( $order, 'billing_postcode' );
	}
	
	/**
	 * Returns the order billing .
	 *
	 * @since 1.6
	 *
	 * @param WC_Order $order
	 *
	 * @return mixed
	 */
	public static function get_order_billing_state( $order ) {
		return self::get_order_prop( $order, 'billing_state' );
	}
	
	/**
	 * Returns the order billing .
	 *
	 * @since 1.6
	 *
	 * @param WC_Order $order
	 *
	 * @return mixed
	 */
	public static function get_order_billing_country( $order ) {
		return self::get_order_prop( $order, 'billing_country' );
	}
	
	/**
	 * Returns the order billing .
	 *
	 * @since 1.6
	 *
	 * @param WC_Order $order
	 *
	 * @return mixed
	 */
	public static function get_order_billing_phone( $order ) {
		return self::get_order_prop( $order, 'billing_phone' );
	}
	
	/**
	 * Returns the order billing .
	 *
	 * @since 1.6
	 *
	 * @param WC_Order $order
	 *
	 * @return mixed
	 */
	public static function get_order_billing_email( $order ) {
		return self::get_order_prop( $order, 'billing_email' );
	}
	
	/**
	 * Returns the order shipping .
	 *
	 * @since 1.6
	 *
	 * @param WC_Order $order
	 *
	 * @return mixed
	 */
	public static function get_order_shipping_first_name( $order ) {
		return self::get_order_prop( $order, 'shipping_first_name' );
	}
	
	/**
	 * Returns the order shipping .
	 *
	 * @since 1.6
	 *
	 * @param WC_Order $order
	 *
	 * @return mixed
	 */
	public static function get_order_shipping_last_name( $order ) {
		return self::get_order_prop( $order, 'shipping_last_name' );
	}
	
	/**
	 * Returns the order shipping .
	 *
	 * @since 1.6
	 *
	 * @param WC_Order $order
	 *
	 * @return mixed
	 */
	public static function get_order_shipping_company( $order ) {
		return self::get_order_prop( $order, 'shipping_company' );
	}
	
	/**
	 * Returns the order shipping .
	 *
	 * @since 1.6
	 *
	 * @param WC_Order $order
	 *
	 * @return mixed
	 */
	public static function get_order_shipping_address_1( $order ) {
		return self::get_order_prop( $order, 'shipping_address_1' );
	}
	
	/**
	 * Returns the order shipping .
	 *
	 * @since 1.6
	 *
	 * @param WC_Order $order
	 *
	 * @return mixed
	 */
	public static function get_order_shipping_address_2( $order ) {
		return self::get_order_prop( $order, 'shipping_address_2' );
	}
	
	/**
	 * Returns the order shipping .
	 *
	 * @since 1.6
	 *
	 * @param WC_Order $order
	 *
	 * @return mixed
	 */
	public static function get_order_shipping_city( $order ) {
		return self::get_order_prop( $order, 'shipping_city' );
	}
	
	/**
	 * Returns the order shipping .
	 *
	 * @since 1.6
	 *
	 * @param WC_Order $order
	 *
	 * @return mixed
	 */
	public static function get_order_shipping_postcode( $order ) {
		return self::get_order_prop( $order, 'shipping_postcode' );
	}
	
	/**
	 * Returns the order shipping .
	 *
	 * @since 1.6
	 *
	 * @param WC_Order $order
	 *
	 * @return mixed
	 */
	public static function get_order_shipping_state( $order ) {
		return self::get_order_prop( $order, 'shipping_state' );
	}
	
	/**
	 * Returns the order shipping .
	 *
	 * @since 1.6
	 *
	 * @param WC_Order $order
	 *
	 * @return mixed
	 */
	public static function get_order_shipping_country( $order ) {
		return self::get_order_prop( $order, 'shipping_country' );
	}
	
	/**
	 * Returns the order customer note .
	 *
	 * @since 1.6
	 *
	 * @param WC_Order $order
	 *
	 * @return mixed
	 */
	public static function get_order_customer_note( $order ) {
		return self::get_order_prop( $order, 'customer_note' );
	}
	
	/**
	 * Return Cart URL
	 *
	 * @since 1.6.3
	 *
	 * @return string
	 */
	public static function wc_get_cart_url() {
		if ( self::is_wc_2_5() ) {
			return wc_get_cart_url();
		}
		
		return WC()->cart->get_cart_url();
	}
	
	/**
	 * Reduces stock levels
	 *
	 * @since 1.6.4
	 *
	 * @param WC_Order $order
	 */
	public static function wc_reduce_stock_levels( $order ) {
		if ( self::is_wc_3_0() ) {
			$order->reduce_order_stock();
		} else {
			wc_reduce_stock_levels( self::get_prop( $order, 'id' ) );
		}
	}
	
	/**
	 * Retrieves meta data from WC CRUD objects
	 *
	 * @since 1.6.4
	 *
	 * @param WC_Order|WC_Product|WC_Subscription $object
	 * @param string                              $meta_name
	 * @param bool                                $single
	 *
	 * @return mixed $value
	 */
	public static function get_meta( $object, $meta_name, $single = true ) {
		if ( self::is_wc_3_0() ) {
			return $object->get_meta( $meta_name, $single );
		}
		
		return get_post_meta( self::get_prop( $object, 'id' ), $meta_name, true );
	}
	
	/**
	 * Returns an order property
	 *
	 * @since 1.6.5
	 *
	 * @param WC_Product $product
	 *
	 * @return mixed
	 */
	public static function get_stock_quantity( $product ) {
		if ( self::is_wc_3_0() && is_callable( array( $product, 'get_stock_quantity' ) ) ) {
			return $product->get_stock_quantity();
		}
		
		return $product->stock;
	}
	
	/**
	 * Returns an order property
	 *
	 * @since 1.6.5
	 *
	 * @param WC_Product $product
	 *
	 * @return mixed
	 */
	public static function get_product_type( $product ) {
		if ( self::is_wc_3_0() && is_callable( array( $product, 'get_type' ) ) ) {
			return $product->get_type();
		}
		
		return $product->product_type;
	}
	
	/**
	 * Returns an order property
	 *
	 * @since 1.6.5
	 *
	 * @param WC_Product $product
	 *
	 * @return mixed
	 */
	public static function get_product_title( $product ) {
		if ( self::is_wc_3_0() && is_callable( array( $product, 'get_title' ) ) ) {
			return $product->get_name();
		}
		
		return get_the_title( self::get_prop( $product, 'id' ) );
	}
	
	/**
	 * Returns the order statuses for a paid order
	 *
	 * @since 1.7
	 *
	 * @return array|mixed
	 */
	public static function get_is_paid_statuses() {
		if ( self::is_wc_3_0() ) {
			return wc_get_is_paid_statuses();
		}
		
		return apply_filters( 'woocommerce_order_is_paid_statuses', array(
			'processing',
			'completed',
		) );
	}
	
	/**
	 * Update a WC object meta
	 *
	 * @since 1.9.0
	 *
	 * @param WC_Order|WC_Product|WC_Subscription $wc_object
	 * @param                                     $name
	 * @param                                     $value
	 * @param bool                                $unique
	 *
	 * @return bool|int
	 */
	public static function update_meta( $wc_object, $name, $value, $unique = true ) {
		if ( self::is_wc_3_0() ) {
			$wc_object->add_meta_data( $name, wc_clean( $value ), $unique );
			
			return $wc_object->save();
		} else {
			return update_post_meta( self::get_prop( $wc_object, 'id' ), $name, $value );
		}
	}
	
	/**
	 * Delete a WC object meta
	 *
	 * @since 1.9.0
	 *
	 * @param WC_Order|WC_Product|WC_Subscription $wc_object
	 * @param                                     $name
	 *
	 * @return bool|int
	 */
	public static function delete_meta( $wc_object, $name ) {
		if ( self::is_wc_3_0() ) {
			$wc_object->delete_meta_data( $name );
			
			return $wc_object->save();
		} else {
			return delete_post_meta( self::get_prop( $wc_object, 'id' ), $name );
		}
	}
}