<?php

namespace WcPaytrace\Api\Post;

use WcPaytrace\Api\Post\Services\Capture;
use WcPaytrace\Api\Post\Services\Receipt;
use WcPaytrace\Api\Post\Services\Refund;
use WcPaytrace\Api\Post\Services\Transaction;
use WcPaytrace\Api\Post\Services\Vault;

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Provides setup and routing for the specific service actions card/check payments, refunds, captures etc.
 * Will load the integration credentials and pass the request props to the specific service class.
 *
 * TODO: We can combine the Services classes
 *
 * @since  2.0
 * @author VanboDevelops
 *
 *        Copyright: (c) 2017 VanboDevelops
 *        License: GNU General Public License v3.0
 *        License URI: http://www.gnu.org/licenses/gpl-3.0.html
 */
class Services {
	
	/**
	 * @var \WcPaytrace\Api\Client
	 */
	public $client;
	/**
	 * @var Validator
	 */
	public $validator;
	public $username;
	public $password;
	public $test_mode;
	public $proxy;

	/**
	 * Services constructor.
	 *
	 * @param \WcPaytrace\Api\Client $client
	 */
	public function __construct( $client ) {
		$this->client = $client;
		$this->setup_credentials();
	}
	
	public function setup_credentials() {
		$this->username  = $this->client->get_user_name();
		$this->password  = $this->client->get_password();
		$this->test_mode = $this->client->is_testmode();
		$this->proxy     = $this->client->get_gateway()->get_option( 'proxy', '' );

		if ( '' == $this->username ) {
			throw new \Exception( __( 'Error while setting up payment. Customer ID is required.', \WC_Paytrace::TEXT_DOMAIN ) );
		}
		if ( '' == $this->password ) {
			throw new \Exception( __( 'Error while setting up payment. Gateway password is required', \WC_Paytrace::TEXT_DOMAIN ) );
		}
	}
	
	/**
	 * Loads the transaction API class
	 *
	 * @since 2.0
	 *
	 * @return Transaction
	 */
	public function transaction() {
		return new Transaction( $this, $this->get_validator() );
	}
	
	/**
	 * Loads the Vault API class
	 *
	 * @since 2.0
	 *
	 * @return Vault
	 */
	public function vault() {
		return new Vault( $this, $this->get_validator() );
	}
	
	/**
	 * Loads the Refund API class
	 *
	 * @since 2.0
	 *
	 * @return Refund
	 */
	public function refund() {
		return new Refund( $this, $this->get_validator() );
	}
	
	/**
	 * Loads the Capture API class
	 *
	 * @since 2.0
	 *
	 * @return Capture
	 */
	public function capture() {
		return new Capture( $this, $this->get_validator() );
	}

	/**
	 * Loads the email receipt class
	 *
	 * @since 2.3.0
	 * @return Receipt
	 */
	public function receipt() {
		return new Receipt( $this, $this->get_validator() );
	}
	
	/**
	 * Loads the Post service validator.
	 * The validator validates the submitted fields and data
	 *
	 * @since 2.0
	 *
	 * @return Validator
	 */
	public function get_validator() {
		if ( $this->validator ) {
			return $this->validator;
		}
		
		return $this->validator = new Validator( $this->client );
	}
}