<?php

namespace WcPaytrace\Api\Post;

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Description
 *
 * @since  2.0
 * @author VanboDevelops
 *
 *        Copyright: (c) 2017 VanboDevelops
 *        License: GNU General Public License v3.0
 *        License URI: http://www.gnu.org/licenses/gpl-3.0.html
 */
class Requests {
	
	public $request;
	/**
	 * @var \WcPaytrace\Api\Post\Validator
	 */
	public $validator;
	protected $post_url = "https://paytrace.com/api/default.pay";
	/**
	 * @var string
	 */
	protected $ERROR;
	protected $proxy;
	
	/**
	 * Send the PayTrace request and return its response body
	 *
	 * @since 2.0
	 *
	 * @param string $send_param_list
	 *
	 * @return mixed The response body
	 * @throws \Exception
	 */
	protected function send_request( $send_param_list ) {
		
		if ( ! empty( $this->proxy ) ) {
			// Set up CURL request if the customer uses proxy
			// Do this to be easier for the customer to use the proxy with PayTrace
			$l_Header = array(
				"MIME-Version: 1.0",
				"Content-type: application/x-www-form-urlencoded",
				"Contenttransfer-encoding: text"
			);
			$l_URL    = $this->post_url;
			$l_CURL   = curl_init();
			curl_setopt( $l_CURL, CURLOPT_URL, $l_URL );
			curl_setopt( $l_CURL, CURLOPT_VERBOSE, 1 );
			curl_setopt( $l_CURL, CURLOPT_PROXYTYPE, CURLPROXY_HTTP );
			curl_setopt( $l_CURL, CURLOPT_HTTPHEADER, $l_Header );
			// Set up proxy if one is needed by the hosting server
			if ( ! empty( $this->proxy ) ) {
				curl_setopt( $l_CURL, CURLOPT_PROXY, $this->proxy );
			}
			curl_setopt( $l_CURL, CURLOPT_SSL_VERIFYPEER, true );
			curl_setopt( $l_CURL, CURLOPT_SSL_VERIFYHOST, 2 );
			curl_setopt( $l_CURL, CURLOPT_POST, true );
			curl_setopt( $l_CURL, CURLOPT_POSTFIELDS, $send_param_list );
			curl_setopt( $l_CURL, CURLOPT_RETURNTRANSFER, true );
			curl_setopt( $l_CURL, CURLOPT_TIMEOUT, 30 );
			$response['body'] = curl_exec( $l_CURL );
			curl_close( $l_CURL );
		} else {
			$post_args = array(
				'headers'   => array(
					'MIME-Version' => '1.0',
					'Content-Type' => 'application/x-www-form-urlencoded',
				),
				'method'    => 'POST',
				'body'      => $send_param_list,
				'sslverify' => apply_filters( 'https_local_ssl_verify', true ),
				'timeout'   => 30,
			);
			$url       = $this->post_url;
			
			$response = wp_remote_post( $url, $post_args );
			
			if ( is_wp_error( $response ) ) {
				throw new \Exception( $response->get_error_message() );
			}
		}
		
		return $response;
	}
	
	/**
	 * Parses the request response and places all parameters into accessible class variables
	 *
	 * @since 2.0
	 *
	 * @param array $response The response string
	 *
	 * @return Response
	 */
	protected function parse_response( $response ) {
		$parsed = new Response( $response );
		
		return $parsed;
	}
	
	/**
	 * Generated the request string for the request
	 *
	 * @since 2.0
	 */
	protected function format_request() {
		if ( empty( $this->request ) ) {
			throw new \Exception( 'The parameter for the request to PayTrace are not set correctly' );
		}
		
		$l_parmlist = '';
		// Go through the set parameters and generate the request string
		foreach ( $this->request as $key => $value ) {
			$l_parmlist .= $key . '~' . $value . '|';
		}
		
		$l_parmlist = "ParmList=" . urlencode( $l_parmlist );
		
		return $l_parmlist;
	}
	
	/**
	 * Sets the username property to the request
	 *
	 * @since 2.0
	 *
	 * @param $username
	 */
	public function set_username( $username ) {
		$this->add_prop( 'UN', $username );
	}
	
	/**
	 * Sets the password property to the request
	 *
	 * @since 2.0
	 *
	 * @param $password
	 */
	public function set_password( $password ) {
		$this->add_prop( 'PSWD', $password );
	}
	
	/**
	 * Sets the method property to the request
	 *
	 * @since 2.0
	 *
	 * @param $value
	 */
	public function set_method( $value ) {
		$this->add_prop( 'METHOD', $value );
	}
	
	/**
	 * Sets the method property to the request
	 *
	 * @since 2.0
	 *
	 * @param $value
	 */
	public function set_terms( $value ) {
		$this->add_prop( 'TERMS', $value );
	}
	
	/**
	 * Sets the method property to the request
	 *
	 * @since 2.0
	 *
	 * @param $value
	 */
	public function set_test( $value ) {
		$this->add_prop( 'TEST', $value );
	}
	
	/**
	 * Bulk sets the provided properties
	 *
	 * @since 2.0
	 *
	 * @param $props
	 */
	public function set_props( $props ) {
		foreach ( $props as $key => $value ) {
			$call = 'set_' . strtolower( $key );
			if ( is_callable( array( $this, $call ) ) ) {
				$this->$call( $value );
			} else {
				$this->add_prop( $key, $value );
			}
		}
	}
	
	/**
	 * Add a parameter to the payment request
	 *
	 * @since 2.0
	 *
	 * @param string $key   The parameter name
	 * @param string $value The parameter value
	 */
	public function add_prop( $key, $value ) {
		if ( '' === $key ) {
			return;
		}
		
		$this->request[ strtoupper( $key ) ] = $this->format_property_value( $key, $value );
	}
	
	/**
	 * Formats the property value
	 *
	 * @since 2.0
	 *
	 * @param $key
	 * @param $value
	 *
	 * @return mixed|string
	 */
	public function format_property_value( $key, $value ) {
		// A restricted character in the prop value will break the request
		$value = $this->validator->remove_restricted_characters( $value );
		$value = $this->validator->format_property_text( $key, $value );
		$value = $this->validator->format_property_length( $key, $value );
		
		return $value;
	}
}