<?php

namespace WcPaytrace\Api\Json;

use WcPaytrace\Api\Json\Protect\Protect_Authorization;
use WcPaytrace\Api\Json\Services\Capture;
use WcPaytrace\Api\Json\Services\Receipt;
use WcPaytrace\Api\Json\Services\Refund;
use WcPaytrace\Api\Json\Services\Transaction;
use WcPaytrace\Api\Json\Services\Vault;

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Provides setup and routing for the specific service actions card/check payments, refunds, captures etc.
 * Will load the integration credentials and pass the request props to the specific service class.
 *
 * @since  2.0
 * @author VanboDevelops
 *
 *        Copyright: (c) 2017 VanboDevelops
 *        License: GNU General Public License v3.0
 *        License URI: http://www.gnu.org/licenses/gpl-3.0.html
 */
class Services {
	
	/**
	 * @var \WcPaytrace\Api\Client
	 */
	public $client;
	/**
	 * @var Validator
	 */
	public $validator;
	public $username;
	public $password;
	public $test_mode;
	public $access_token;
	
	/**
	 * Services constructor.
	 *
	 * @throws \Exception
	 *
	 * @param \WcPaytrace\Api\Client $client
	 */
	public function __construct( $client ) {
		$this->client = $client;
		$this->setup_credentials();
	}
	
	/**
	 * Sets and checks the API credentials
	 *
	 * @since 2.0
	 *
	 * @throws \Exception
	 */
	public function setup_credentials() {
		$this->username = $this->client->get_user_name();
		$this->password = $this->client->get_password();
		
		if ( '' == $this->username ) {
			throw new \Exception( __( 'Error while setting up payment. Customer ID is required.', \WC_Paytrace::TEXT_DOMAIN ) );
		}
		if ( '' == $this->password ) {
			throw new \Exception( __( 'Error while setting up payment. Gateway password is required', \WC_Paytrace::TEXT_DOMAIN ) );
		}
	}
	
	/**
	 * Returns the Transaction API class
	 *
	 * @since 2.0
	 *
	 * @return Transaction
	 */
	public function transaction() {
		return new Transaction( $this, $this->get_validator() );
	}
	
	/**
	 * Returns the OAuth API class
	 *
	 * @since 2.0
	 *
	 * @return OAuth2
	 */
	public function get_oauth2() {
		return new OAuth2( $this, $this->get_validator() );
	}
	
	/**
	 * Returns the Vault API class
	 *
	 * @since 2.0
	 *
	 * @return Vault
	 */
	public function vault() {
		return new Vault( $this, $this->get_validator() );
	}
	
	/**
	 * Returns the Refund API class
	 *
	 * @since 2.0
	 *
	 * @return Refund
	 */
	public function refund() {
		return new Refund( $this, $this->get_validator() );
	}
	
	/**
	 * Returns the Capture API class
	 *
	 * @since 2.0
	 *
	 * @return Capture
	 */
	public function capture() {
		return new Capture( $this, $this->get_validator() );
	}
	
	/**
	 * Loads the email receipt class
	 *
	 * @since 2.3.0
	 * @return Receipt
	 */
	public function receipt() {
		return new Receipt( $this, $this->get_validator() );
	}
	
	/**
	 * @return \WcPaytrace\Api\Json\Protect\Protect_Authorization
	 */
	public function protect_authorization() {
		return new Protect_Authorization( $this, $this->get_validator() );
	}
	
	/**
	 * Loads the Post service validator.
	 * The validator validates the submitted fields and data
	 *
	 * @since 2.0
	 *
	 * @return Validator
	 */
	public function get_validator() {
		if ( $this->validator ) {
			return $this->validator;
		}
		
		return $this->validator = new Validator( $this->client );
	}
	
	/**
	 * Returns an access token for the JSON API requests
	 *
	 * @since 2.0
	 *
	 * @return mixed|string
	 */
	public function get_access_token() {
		$access_token = get_transient( 'wc_paytrace_json_access_token' );
		
		if ( false === $access_token ) {
			$access_token = $this->get_oauth2()->request_token();
		}
		
		return $access_token;
	}
}