<?php

namespace WcPaytrace\Api\Json;

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * @since  2.0
 * @author VanboDevelops
 *
 *        Copyright: (c) 2017 VanboDevelops
 *        License: GNU General Public License v3.0
 *        License URI: http://www.gnu.org/licenses/gpl-3.0.html
 */
class Requests {
	
	public $allowed_transaction_types = array(
		'transactions',
		'checks',
	);
	public $allowed_authorization_types = array(
		'sale',
		'authorization',
		'hold',
	);
	public $request;
	/**
	 * @var \WcPaytrace\Api\Json\Validator
	 */
	public $validator;
	protected $base_url = "https://api.paytrace.com";
	protected $api_version_endpoint = '/v1';
	protected $action_endpoint = '';
	/**
	 * @var string
	 */
	protected $ERROR;
	/**
	 * @var Services
	 */
	protected $services;
	/**
	 * @var string
	 */
	protected $access_token;
	protected $request_method = 'POST';
	protected $allowed_request_methods = array(
		'POST',
		'GET',
	);
	private $integrator_id = '90000236vanb';
	
	public function __construct( Services $services, $validator ) {
		$this->services  = $services;
		$this->validator = $validator;
	}
	
	public function get_integrator_id() {
		return $this->integrator_id;
	}
	
	/**
	 * Send the PayTrace request and return its response body
	 *
	 * @since 2.0
	 *
	 * @return mixed The response body
	 * @throws \Exception
	 */
	protected function send() {
		$this->add_prop( 'integrator_id', $this->get_integrator_id() );
		$body = $this->format();
		
		$post_args = array(
			'headers'   => $this->get_request_headers(),
			'method'    => $this->get_request_method(),
			'body'      => $body,
			'sslverify' => apply_filters( 'https_local_ssl_verify', true ),
			'timeout'   => apply_filters( 'wc_paytrace_request_timeout', 30 ),
		);
		
		$response = wp_remote_post( $this->get_request_url(), $post_args );
		
		if ( is_wp_error( $response ) ) {
			throw new \Exception( $response->get_error_message() );
		}
		
		return $response;
	}
	
	/**
	 * Returns the URL that we are going to send the request to.
	 *
	 * @since 2.0
	 *
	 * @return string
	 */
	public function get_request_url() {
		return $this->get_base_url() . $this->get_api_version_endpoint() . $this->get_action_endpoint();
	}
	
	/**
	 * Returns the base url
	 *
	 * @since 2.0
	 *
	 * @return string
	 */
	public function get_base_url() {
		return $this->base_url;
	}
	
	/**
	 * Returns the API version endpoint "/vN".
	 *
	 * @since 2.0
	 *
	 * @return string
	 */
	public function get_api_version_endpoint() {
		return $this->api_version_endpoint;
	}
	
	/**
	 * Returns the request endpoint.
	 * In most cases this should be overwritten by the request specific class
	 *
	 * @since 2.0
	 *
	 * @return string
	 */
	public function get_action_endpoint() {
		return $this->action_endpoint;
	}
	
	/**
	 * Returns the request method
	 *
	 * @since 2.0
	 *
	 * @return string
	 */
	public function get_request_method() {
		return $this->request_method;
	}
	
	/**
	 * Sets the request method
	 *
	 * @since 2.0
	 *
	 * @param $method
	 *
	 * @return bool
	 */
	public function set_request_method( $method ) {
		if ( in_array( $method, $this->allowed_request_methods ) ) {
			return $this->request_method = $method;
		}
		
		return $this->request_method = 'POST';
	}
	
	/**
	 * Returns the request headers
	 *
	 * @since 2.0
	 *
	 * @return array
	 */
	public function get_request_headers() {
		return array(
			'Authorization' => 'Bearer ' . $this->get_access_token(),
			'Content-Type'  => 'application/json',
			'Cache-Control' => 'no-cache',
		);
	}
	
	/**
	 * Retrieves the oAuth2 access token
	 *
	 * @since 2.0
	 *
	 * @return mixed|Response
	 */
	public function get_access_token() {
		return $this->services->get_access_token();
	}
	
	/**
	 * Parses the request response and places all parameters into accessible class variables
	 *
	 * @since 2.0
	 *
	 * @param array $response The response string
	 *
	 * @return Response
	 */
	protected function parse_response( $response ) {
		$parsed = new Response( $response );
		
		return $parsed;
	}
	
	/**
	 * Generated the request string for the request
	 *
	 * @since 2.0
	 */
	protected function format() {
		if ( empty( $this->request ) ) {
			throw new \Exception( 'The parameter for the request to PayTrace are not set correctly' );
		}
		
		return json_encode( $this->request );
	}
	
	/**
	 * Sets the username property to the request
	 *
	 * @since 2.0
	 *
	 * @param string $username
	 */
	public function set_username( $username ) {
		$this->add_prop( 'username', $username );
	}
	
	/**
	 * Sets the password property to the request
	 *
	 * @since 2.0
	 *
	 * @param string $password
	 */
	public function set_password( $password ) {
		$this->add_prop( 'password', $password );
	}
	
	/**
	 * Sets the grant type prop
	 *
	 * @since 2.0
	 *
	 * @param string
	 */
	public function set_grant_type( $type ) {
		$this->add_prop( 'grant_type', $type );
	}
	
	/**
	 * Bulk sets the provided properties
	 *
	 * @since 2.0
	 *
	 * @param array $props
	 */
	public function set_props( $props ) {
		foreach ( $props as $key => $value ) {
			$this->add_prop( $key, $value );
		}
	}
	
	/**
	 * Add a parameter to the payment request
	 *
	 * @since 2.0
	 *
	 * @param string $key   The parameter name
	 * @param string $value The parameter value
	 */
	public function add_prop( $key, $value ) {
		if ( '' === $key ) {
			return;
		}
		
		$this->request[ $key ] = $this->format_property_value( $key, $value );
	}
	
	/**
	 * Formats the property value
	 *
	 * @since 2.0
	 *
	 * @param string $key
	 * @param string $value
	 *
	 * @return mixed|string
	 */
	public function format_property_value( $key, $value ) {
		if ( is_string( $value ) ) {
			$value = $this->validator->remove_restricted_characters( $value );
		}
		
		return $value;
	}
	
	/**
	 * Checks the the transaction type in the props and sets it according to the allowed values
	 *
	 * @since 2.0
	 *
	 * @param $props
	 *
	 * @return string
	 */
	public function get_transaction_type( $props ) {
		if ( ! isset( $props['transaction_type'] ) ) {
			return 'transactions';
		}
		
		$type = strtolower( $props['transaction_type'] );
		if ( ! in_array( $type, $this->allowed_transaction_types ) ) {
			return 'transactions';
		}
		
		return $type;
	}
	
	/**
	 * Checks the the authorization type in the props and sets it according to the allowed values
	 *
	 * @since 2.0
	 *
	 * @param $props
	 *
	 * @return string
	 */
	public function get_authorization_type( $props ) {
		if ( ! isset( $props['authorization_type'] ) ) {
			return 'sale';
		}
		
		$type = strtolower( $props['authorization_type'] );
		if ( ! in_array( $type, $this->allowed_authorization_types ) ) {
			return 'sale';
		}
		
		return $type;
	}
}