<?php

namespace WcPaytrace\Api\Json\Protect;

use WcPaytrace\Api\Json\Requests;

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * @since  2.7.0
 * @author VanboDevelops
 *
 *        Copyright: (c) 2022 VanboDevelops
 *        License: GNU General Public License v3.0
 *        License URI: http://www.gnu.org/licenses/gpl-3.0.html
 */
class Protect_Authorization extends Requests {
	
	/**
	 * Returns the endpoint for the request
	 *
	 * @since 2.0
	 *
	 * @return string
	 */
	public function get_action_endpoint() {
		return '/payment_fields/token/create';
	}
	
	/**
	 * Returns the request headers
	 *
	 * @since 2.0
	 *
	 * @return array
	 */
	public function get_request_headers() {
		return array(
			'Authorization' => 'Bearer ' . $this->get_access_token(),
			'Content-Type'  => 'application/json',
		);
	}
	
	/**
	 * Processes the capture request
	 *
	 * @since 2.7
	 *
	 * @return mixed|\WcPaytrace\Api\Json\Response
	 * @throws \Exception
	 */
	public function get_client_key() {
		$client_key = get_transient( 'wc_paytrace_protect_key' );
		
		if ( false !== $client_key ) {
			return $client_key;
		}
		
		$response = $this->parse_response( $this->send() );
		
		if ( ! $response->get_client_key() ) {
			$description = $response->get_error_description();
			$error       = $response->get_error();
			
			$message = __( 'Protect token was not obtained.', \WC_Paytrace::TEXT_DOMAIN );
			if ( '' != $error ) {
				$message .= ' ' . sprintf( __( 'Error: %s', \WC_Paytrace::TEXT_DOMAIN ), $error );
			}
			
			if ( '' != $description ) {
				$message .= ' ' . sprintf( __( 'Error Message: %s', \WC_Paytrace::TEXT_DOMAIN ), $description );
			}
			
			// TODO: This exception needs to be handled to an error to customer
			throw new \Exception( $message );
		}
		
		set_transient( 'wc_paytrace_protect_key', $response->get_client_key(), ( 2 * HOUR_IN_SECONDS ) - 600 );
		
		return $response->get_client_key();
	}
	
	protected function parse_response( $response ) {
		return new Protect_Response( $response );
	}
	
	protected function send() {
		$body = $this->format();
		
		$post_args = array(
			'headers'   => $this->get_request_headers(),
			'method'    => $this->get_request_method(),
			'body'      => $body,
			'sslverify' => apply_filters( 'https_local_ssl_verify', true ),
			'timeout'   => apply_filters( 'wc_paytrace_request_timeout', 30 ),
		);
		
		$response = wp_remote_post( $this->get_request_url(), $post_args );
		
		if ( is_wp_error( $response ) ) {
			throw new \Exception( $response->get_error_message() );
		}
		
		return $response;
	}
	
	public function format() {
		return $this->request;
	}
}