<?php

namespace WcPaytrace\Admin;

class Privacy_Policy extends \WC_Abstract_Privacy {
	
	/**
	 * Constructor
	 *
	 */
	public function __construct() {
		parent::__construct( __( 'Paytrace', \WC_Paytrace::TEXT_DOMAIN ) );
		
		$this->add_exporter( 'woocommerce-gateway-paytrace-order-data', __( 'WooCommerce Paytrace Order Data', \WC_Paytrace::TEXT_DOMAIN ), array(
			$this,
			'order_data_exporter'
		) );
		
		if ( function_exists( 'wcs_get_subscriptions' ) ) {
			$this->add_exporter( 'woocommerce-gateway-paytrace-subscriptions-data',
				__( 'WooCommerce Paytrace Subscriptions Data', \WC_Paytrace::TEXT_DOMAIN ), array(
					$this,
					'subscriptions_data_exporter'
				) );
		}
		
		$this->add_eraser( 'woocommerce-gateway-paytrace-order-data',
			__( 'WooCommerce Paytrace Data', \WC_Paytrace::TEXT_DOMAIN ), array(
				$this,
				'order_data_eraser'
			) );
	}
	
	/**
	 * Returns list of orders paid with Paytrace
	 *
	 * @param string $email_address
	 * @param int    $page
	 *
	 * @return array WC_Order
	 */
	protected function get_gateway_orders( $email_address, $page ) {
		$user = get_user_by( 'email', $email_address ); // Check if user has an ID in the DB to load stored personal data.
		
		$order_query = array(
			'payment_method' => array(
				'paytrace',
			),
			'limit'          => 10,
			'page'           => $page,
		);
		
		if ( $user instanceof \WP_User ) {
			$order_query['customer_id'] = (int) $user->ID;
		} else {
			$order_query['billing_email'] = $email_address;
		}
		
		return wc_get_orders( $order_query );
	}
	
	/**
	 * Gets the message of the privacy to display.
	 *
	 */
	public function get_privacy_message() {
		return wpautop( __( 'By using this extension, you may be storing personal data or sharing data with an external service.', \WC_Paytrace::TEXT_DOMAIN ) );
	}
	
	/**
	 * Handle exporting data for Orders.
	 *
	 * @param string $email_address E-mail address to export.
	 * @param int    $page          Pagination of data.
	 *
	 * @return array
	 */
	public function order_data_exporter( $email_address, $page = 1 ) {
		$done           = false;
		$data_to_export = array();
		
		$orders = $this->get_gateway_orders( $email_address, (int) $page );
		
		if ( 0 < count( $orders ) ) {
			foreach ( $orders as $order ) {
				$pt_order = new \WC_Paytrace_Order( $order );
				
				$data_to_export[] = array(
					'group_id'    => 'woocommerce_orders',
					'group_label' => __( 'Orders', \WC_Paytrace::TEXT_DOMAIN ),
					'item_id'     => 'order-' . $order->get_id(),
					'data'        => array(
						array(
							'name'  => __( 'Paytrace customer id', \WC_Paytrace::TEXT_DOMAIN ),
							'value' => $pt_order->get_customer_id(),
						),
					),
				);
			}
			
			$done = 10 > count( $orders );
		}
		
		return array(
			'data' => $data_to_export,
			'done' => $done,
		);
	}
	
	/**
	 * Handle exporting data for Subscriptions.
	 *
	 * @param string $email_address E-mail address to export.
	 * @param int    $page          Pagination of data.
	 *
	 * @return array
	 */
	public function subscriptions_data_exporter( $email_address, $page = 1 ) {
		$page           = (int) $page;
		$data_to_export = array();
		
		$meta_query = array(
			'relation' => 'AND',
			array(
				'key'     => '_payment_method',
				'value'   => array(
					'paytrace',
				),
				'compare' => 'IN',
			),
			array(
				'key'     => '_billing_email',
				'value'   => $email_address,
				'compare' => '=',
			),
		);
		
		$subscription_query = array(
			'posts_per_page' => 10,
			'page'           => $page,
			'meta_query'     => $meta_query,
		);
		
		$subscriptions = wcs_get_subscriptions( $subscription_query );
		
		$done = true;
		
		if ( 0 < count( $subscriptions ) ) {
			foreach ( $subscriptions as $subscription ) {
				$pt_sub = new \WC_Paytrace_Order( $subscription );
				
				$data_to_export[] = array(
					'group_id'    => 'woocommerce_subscriptions',
					'group_label' => __( 'Subscriptions', \WC_Paytrace::TEXT_DOMAIN ),
					'item_id'     => 'subscription-' . $subscription->get_id(),
					'data'        => array(
						array(
							'name'  => __( 'Paytrace customer id', \WC_Paytrace::TEXT_DOMAIN ),
							'value' => $pt_sub->get_customer_id(),
						),
					),
				);
			}
			
			$done = 10 > count( $subscriptions );
		}
		
		return array(
			'data' => $data_to_export,
			'done' => $done,
		);
	}
	
	/**
	 * Finds and erases order data by email address.
	 *
	 * @param string $email_address The user email address.
	 * @param int    $page          Page.
	 *
	 * @return array An array of personal data in name value pairs
	 */
	public function order_data_eraser( $email_address, $page ) {
		$orders = $this->get_gateway_orders( $email_address, (int) $page );
		
		$items_removed  = false;
		$items_retained = false;
		$messages       = array();
		
		foreach ( (array) $orders as $order ) {
			$order = wc_get_order( $order->get_id() );
			
			list( $removed, $retained, $msgs ) = $this->maybe_handle_order( $order );
			$items_removed  = $items_removed || $removed;
			$items_retained = $items_retained || $retained;
			$messages       = array_merge( $messages, $msgs );
			
			list( $removed, $retained, $msgs ) = $this->maybe_handle_subscription( $order );
			$items_removed  = $items_removed || $removed;
			$items_retained = $items_retained || $retained;
			$messages       = array_merge( $messages, $msgs );
		}
		
		// Tell core if we have more orders to work on still
		$done = count( $orders ) < 10;
		
		return array(
			'items_removed'  => $items_removed,
			'items_retained' => $items_retained,
			'messages'       => $messages,
			'done'           => $done,
		);
	}
	
	/**
	 * Handle eraser of data tied to Subscriptions
	 *
	 * @param \WC_Order $order
	 *
	 * @return array
	 */
	protected function maybe_handle_subscription( $order ) {
		if ( ! class_exists( 'WC_Subscriptions' ) ) {
			return array( false, false, array() );
		}
		
		$paytrace_order = new \WC_Paytrace_Order( $order );
		
		if ( ! $paytrace_order->contains_subscription() ) {
			return array( false, false, array() );
		}
		
		/**
		 * @var \WC_Subscription $subscription
		 */
		$subscription = current( wcs_get_subscriptions_for_order( $order->get_id() ) );
		
		$pt_sub      = new \WC_Paytrace_Order( $subscription );
		$customer_id = $pt_sub->get_customer_id();
		
		if ( empty( $customer_id ) ) {
			return array( false, false, array() );
		}
		
		if ( $subscription->has_status( apply_filters( 'wc_paytrace_privacy_eraser_subscription_statuses', array(
			'on-hold',
			'active'
		) ) )
		) {
			return array(
				false,
				true,
				array( sprintf( __( 'Order ID %d contains an active Subscription' ), $order->get_id() ) )
			);
		}
		
		$renewal_orders = $subscription->get_related_orders( 'ids', 'renewal' );
		
		foreach ( $renewal_orders as $renewal_order_id ) {
			$this->delete_order_data( wcs_get_subscription( $renewal_order_id ) );
		}
		
		$this->delete_order_data( wcs_get_subscription( $subscription->get_id() ) );
		
		return array(
			true,
			false,
			array( __( 'Paytrace Subscription Data Erased.', \WC_Paytrace::TEXT_DOMAIN ) )
		);
	}
	
	/**
	 * Handle eraser of data tied to Orders
	 *
	 * @param \WC_Order $order
	 *
	 * @return array
	 */
	protected function maybe_handle_order( $order ) {
		$pt_order = new \WC_Paytrace_Order( $order );
		
		$pt_order->delete_customer_id();
		
		return array(
			true,
			false,
			array( __( 'Paytrace personal data erased.', \WC_Paytrace::TEXT_DOMAIN ) )
		);
	}
	
	public function delete_order_data( $order ) {
		if ( ! $order instanceof \WC_Order ) {
			return false;
		}
		
		$pt_order = new \WC_Paytrace_Order( $order );
		$pt_order->delete_customer_id();
		
		return true;
	}
}