<?php

namespace WcPaytrace\Admin;

use WcPaytrace\Helpers\Factories;

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Description
 *
 * @since  2.5.0
 * @author VanboDevelops
 *
 *        Copyright: (c) 2019 VanboDevelops
 *        License: GNU General Public License v3.0
 *        License URI: http://www.gnu.org/licenses/gpl-3.0.html
 */
class Capture {
	
	/**
	 * Loads the capture action for the plugin
	 *
	 * This is loaded here and not in the gateway
	 * because we need it to load a little bit earlier for the action to be added to order edit screen
	 *
	 * @since 2.5.0
	 */
	public function hooks() {
		/**
		 * @var \WC_Payment_Gateway|\WC_Paytrace_Gateway_Addons $gateway
		 */
		$gateway = Factories::get_gateway( 'paytrace' );
		
		add_filter( 'woocommerce_order_actions', array( $gateway, 'add_order_capture_action' ) );
		add_filter( 'woocommerce_order_actions', array( $gateway, 'add_order_email_receipt_action' ) );
		add_action( 'woocommerce_order_action_paytrace_capture_payment', array(
			$gateway,
			'capture_payment',
		) );
		add_action( 'woocommerce_order_action_paytrace_email_receipt', array(
			$gateway,
			'send_transaction_receipt_action',
		) );
		
		if ( \WC_Paytrace_Compat::equal_or_grt( '4.0.0' ) ) {
			// The hook was moved down, so it is better to use
			add_action( 'woocommerce_admin_order_totals_after_total', array(
				$this,
				'order_meta_box_add_capture_field',
			) );
		} else {
			// The hook is only loaded when a refund is present, so we can't use it on 4.0+
			add_action( 'woocommerce_admin_order_totals_after_refunded', array(
				$this,
				'order_meta_box_add_capture_field',
			) );
		}
		
		add_action( 'woocommerce_order_item_add_action_buttons', array(
			$this,
			'order_meta_box_add_capture_payment_buttons',
		) );
	}
	
	/**
	 * Adds Capture buttons to the admin order edit screen
	 *
	 * @since 2.5.0
	 *
	 * @param \WC_Order $order
	 *
	 * @return bool
	 */
	public function order_meta_box_add_capture_payment_buttons( $order ) {
		$method = \WC_Paytrace_Compat::get_prop( $order, 'payment_method' );
		if ( 'paytrace' != $method ) {
			return false;
		}
		
		$paytrace_order         = new \WC_Paytrace_Order( $order );
		$is_captured            = $paytrace_order->get_is_payment_captured();
		$allowed_order_statuses = self::get_capture_allowed_order_statuses();
		
		if ( $paytrace_order->is_subscription() || $is_captured || ! in_array( $order->get_status(), $allowed_order_statuses ) ) {
			return false;
		}
		
		?>
		<button type="button" class="button wc-paytrace-capture-payment-init">
			<?php _e( 'Capture', \WC_Paytrace::TEXT_DOMAIN ); ?>
		</button>
		<span class="wc-paytrace-capture-payment-wrapper" style="display:none">
				<?php echo wc_help_tip( __( 'Enter the amount you want to capture above and press "Capture Payment" button.', \WC_Paytrace::TEXT_DOMAIN ) ); ?>
			<button type="button" class="button button-primary wc-paytrace-capture-payment">
				<?php _e( 'Capture Payment', \WC_Paytrace::TEXT_DOMAIN ); ?>
			</button>
			</span>
		<button type="button" class="button wc-paytrace-capture-cancel" style="display:none">
			<?php _e( 'Cancel Capture', \WC_Paytrace::TEXT_DOMAIN ); ?>
		</button>
		<?php
	}
	
	/**
	 * Adds the capture amount field to the admin order edit screen
	 *
	 * @since 2.5.0
	 *
	 * @param $order_id
	 *
	 * @return bool
	 */
	public function order_meta_box_add_capture_field( $order_id ) {
		$order = wc_get_order( $order_id );
		
		$method = \WC_Paytrace_Compat::get_prop( $order, 'payment_method' );
		if ( 'paytrace' != $method ) {
			return false;
		}
		
		$paytrace_order         = new \WC_Paytrace_Order( $order );
		$is_captured            = $paytrace_order->get_is_payment_captured();
		$allowed_order_statuses = self::get_capture_allowed_order_statuses();
		
		if ( $paytrace_order->is_subscription() || $is_captured || ! in_array( $order->get_status(), $allowed_order_statuses ) ) {
			return false;
		}
		
		// Get the initially authorized amount. We should not allow charges over this amount.
		$authorized_amount = $paytrace_order->get_order_amount_authorized();
		$total             = $order->get_total();
		if ( ! empty( $authorized_amount ) && $authorized_amount < $total ) {
			$total = $authorized_amount;
		}
		
		?>
		<tr class="wc-paytrace-capture-amount-wrapper" style="display: none;">
			<td class="label capture-total">
				<?php echo wc_help_tip( __( 'You can capture no more than the initially authorized amount. Capture action can be processed only once.', \WC_Paytrace::TEXT_DOMAIN ) ); ?>
				<?php _e( 'Capture', \WC_Paytrace::TEXT_DOMAIN ); ?>:
			</td>
			<?php // There are changes in the cells, so we needed to two versions ?>
			<?php echo \WC_Paytrace_Compat::is_wc_2_6() ? '<td width="1%"></td>' : ''; ?>
			<td class="total capture-total">
				<input type="text"
				       class="wc-paytrace-capture-amount wc_input_price"
				       name="wc-paytrace-capture-amount"
				       value="<?php echo esc_attr( $total ); ?>"
				/>

			</td>
			<?php echo \WC_Paytrace_Compat::is_wc_2_6() ? '' : '<td width="1%"></td>'; ?>
		</tr>
		<?php
	}
	
	/**
	 * Returns the allowed order statuses to perform capture of a transaction.
	 * We naturally assume that the status of an order should be a paid order status, not completed and not failed payment.
	 *
	 * @since 2.5.0
	 *
	 * @return mixed
	 */
	public static function get_capture_allowed_order_statuses() {
		return apply_filters( 'wc_paytrace_capture_allowed_order_statuses', array(
			'processing',
			'on-hold',
			'active',
		) );
	}
}