<?php

namespace WcPaytrace\Admin;

use WcPaytrace\Helpers\Factories;

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Class that represents admin notices.
 *
 * @since 2.3.0
 */
class Admin_Notices {
	
	/**
	 * Collection of all notices
	 * @var array
	 */
	public $notices = array();
	
	public function hooks() {
		add_action( 'admin_notices', array( $this, 'admin_notices' ) );
		add_action( 'wp_loaded', array( $this, 'hide_notices' ) );
	}
	
	/**
	 * Adds a notice to the collection
	 *
	 * @since 2.3.0
	 */
	public function add_notice( $slug, $type, $message, $dismissible = false ) {
		
		$map_class = array(
			'error'   => 'error',
			'notice'  => 'notice notice-warning',
			'warning' => 'notice notice-error',
		);
		
		$this->notices[ $slug ] = array(
			'class'       => $map_class[ $type ],
			'message'     => $message,
			'dismissible' => $dismissible,
		);
	}
	
	/**
	 * Loads the notices
	 * @since 2.3.0
	 */
	public function admin_notices() {
		if ( ! current_user_can( 'manage_woocommerce' ) ) {
			return;
		}
		
		$this->perform_checks();
		
		foreach ( (array) $this->notices as $notice_key => $notice ) {
			echo '<div class="' . esc_attr( $notice['class'] ) . '" style="position:relative;">';
			
			if ( $notice['dismissible'] ) {
				?>
				<a href="<?php echo esc_url( wp_nonce_url( add_query_arg( 'wc-paytrace-hide-notice', $notice_key ), 'wc_paytrace_hide_notice_nonce', '_wc_paytrace_notice_nonce' ) ); ?>" class="woocommerce-message-close notice-dismiss" style="position:absolute;right:1px;padding:9px;text-decoration:none;"></a>
				<?php
			}
			
			echo '<p>';
			echo wp_kses( $notice['message'], array( 'a' => array( 'href' => array() ) ) );
			echo '</p></div>';
		}
	}
	
	/**
	 * Performs the plugin checks
	 * @since 2.3.0
	 */
	public function perform_checks() {
		$show_recurring_notice = 'no' == get_option( 'wc_paytrace_show_notice_recurring' ) ? false : true;
		$show_ssl_notice       = 'no' == get_option( 'wc_paytrace_show_notice_valid_ssl' ) ? false : true;
		$show_phpver_notice    = 'no' == get_option( 'wc_paytrace_show_notice_phpver' ) ? false : true;
		$show_wcver_notice     = 'no' == get_option( 'wc_paytrace_show_notice_wcver' ) ? false : true;
		$show_curl_notice      = 'no' == get_option( 'wc_paytrace_show_notice_curl' ) ? false : true;
		$show_currency_notice  = 'no' == get_option( 'wc_paytrace_show_notice_invalid_currency' ) ? false : true;
		/**
		 * @var \WC_Payment_Gateway|\WC_Paytrace_Gateway_Addons $gateway
		 */
		$gateway        = Factories::get_gateway( 'paytrace' );
		
		// Bail, if the gateway is not enabled
		if ( false == $gateway ) {
			return;
		}
		
		$testmode       = 'yes' == $gateway->testmode ? true : false;
		$save_customers = $gateway->save_customers;
		
		// Bail, if the gateway is not enabled
		if ( 'yes' !== $gateway->enabled ) {
			return;
		}
		
		if ( $show_phpver_notice ) {
			if ( version_compare( phpversion(), \WC_Paytrace::MIN_PHP_VERSION, '<' ) ) {
				$message = __( 'WooCommerce Paytrace - The minimum PHP version required for this plugin is %1$s. You are running %2$s.', \WC_Paytrace::TEXT_DOMAIN );
				
				$this->add_notice( 'phpver', 'error', sprintf( $message, \WC_Paytrace::MIN_PHP_VERSION, phpversion() ), true );
				
				return;
			}
		}
		
		if ( $show_wcver_notice ) {
			if ( version_compare( WC_VERSION, \WC_Paytrace::MIN_WC_VERSION, '<' ) ) {
				$message = __( 'WooCommerce Paytrace - The minimum WooCommerce version required for this plugin is %1$s. You are running %2$s.', \WC_Paytrace::TEXT_DOMAIN );
				
				$this->add_notice( 'wcver', 'notice', sprintf( $message, \WC_Paytrace::MIN_WC_VERSION, WC_VERSION ), true );
				
				return;
			}
		}
		
		if ( $show_curl_notice ) {
			if ( ! function_exists( 'curl_init' ) ) {
				$this->add_notice( 'curl', 'notice', __( 'WooCommerce Paytrace - cURL is not installed.', \WC_Paytrace::TEXT_DOMAIN ), true );
			}
		}
		
		if ( $show_recurring_notice ) {
			if ( ! $save_customers && ( \WC_Paytrace::is_subscriptions_active() || \WC_Paytrace::is_pre_orders_active() ) ) {
				$message = sprintf( __( 'WooCommerce Paytrace - You have "%s" active, but you did not enable PayTrace "Payment Vault" from the %splugin settings%s. PayTrace payment option will not appear on the checkout page.' ), \WC_Paytrace::is_subscriptions_active() ? 'WC Subscriptions' : 'WC Pre-Orders', '<a href="' . $this->get_setting_link() . '">', '</a>' );
				$this->add_notice( 'recurring', 'warning', $message, true );
			}
		}
		
		if ( $show_ssl_notice ) {
			if ( ! wc_checkout_is_https() && ! $testmode ) {
				$this->add_notice( 'valid_ssl', 'notice', sprintf( __( 'Paytrace is enabled for live payments, but a SSL certificate is not detected. Your checkout may not be secure! Please ensure your server has a valid <a href="%1$s" target="_blank">SSL certificate</a>', \WC_Paytrace::TEXT_DOMAIN ), 'https://en.wikipedia.org/wiki/Transport_Layer_Security' ), true );
			}
		}
		
		if ( $show_currency_notice ) {
			
			if ( ! $gateway->is_valid_currency() ) {
				$this->add_notice( 'invalid_currency', 'warning', sprintf( __( 'WooCommerce Paytrace requires %s store currency to be set.', \WC_Paytrace::TEXT_DOMAIN ), implode( ', ', $gateway->get_supported_currency() ) ), true );
			}
		}
	}
	
	/**
	 * Hides any admin notices.
	 *
	 * @since 2.3.0
	 */
	public function hide_notices() {
		if ( isset( $_GET['wc-paytrace-hide-notice'] ) && isset( $_GET['_wc_paytrace_notice_nonce'] ) ) {
			if ( ! wp_verify_nonce( $_GET['_wc_paytrace_notice_nonce'], 'wc_paytrace_hide_notice_nonce' ) ) {
				wp_die( __( 'Action failed. Please refresh the page and retry.', \WC_Paytrace::TEXT_DOMAIN ) );
			}
			
			if ( ! current_user_can( 'manage_woocommerce' ) ) {
				wp_die( __( 'Cheatin&#8217; huh?', \WC_Paytrace::TEXT_DOMAIN ) );
			}
			
			$notice = wc_clean( $_GET['wc-paytrace-hide-notice'] );
			
			switch ( $notice ) {
				case 'phpver':
					update_option( 'wc_paytrace_show_notice_phpver', 'no' );
					break;
				case 'wcver':
					update_option( 'wc_paytrace_show_notice_wcver', 'no' );
					break;
				case 'curl':
					update_option( 'wc_paytrace_show_notice_curl', 'no' );
					break;
				case 'valid_ssl':
					update_option( 'wc_paytrace_show_notice_valid_ssl', 'no' );
					break;
				case 'recurring':
					update_option( 'wc_paytrace_show_notice_recurring', 'no' );
					break;
				case 'invalid_currency':
					update_option( 'wc_paytrace_show_notice_invalid_currency', 'no' );
					break;
			}
		}
	}
	
	/**
	 * Get setting link.
	 *
	 * @since 2.3.0
	 *
	 * @return string Setting link
	 */
	public function get_setting_link() {
		$use_id_as_section = function_exists( 'WC' ) ? version_compare( WC()->version, '2.6', '>=' ) : false;
		
		$section_slug = $use_id_as_section ? 'paytrace' : \WC_Paytrace::get_gateway_class();
		
		return admin_url( 'admin.php?page=wc-settings&tab=checkout&section=' . $section_slug );
	}
}