<?php

namespace WcPaytrace\Abstracts;

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Description
 *
 * @since  2.0
 * @author VanboDevelops
 *
 *        Copyright: (c) 2017 VanboDevelops
 *        License: GNU General Public License v3.0
 *        License URI: http://www.gnu.org/licenses/gpl-3.0.html
 */
abstract class Validator {
	
	public $echeck_routing;
	public $echeck_account;
	public $cc_type = '';
	public $cc_number = '';
	public $cc_exp_month = '';
	public $cc_exp_year = '';
	public $cc_cvc = '';
	/**
	 * @since 2.3.1
	 * @var array|string The restricted characters to remove
	 */
	public $restricted_characters = array( '|' );
	
	/**
	 * Paymenst_Fields_Validator constructor.
	 *
	 * @param \WcPaytrace\Api\Client $client
	 */
	public function __construct( $client ) {
		$this->client = $client;
	}
	
	/**
	 * Set the ACH routing and account number
	 *
	 * @since 1.3
	 */
	public function set_echeck_details() {
		// Set the ACH sensitive info
		$this->set_check_routing( \WC_Paytrace::get_post( 'paytrace-echeck-routing-number' ) );
		$this->set_check_account( \WC_Paytrace::get_post( 'paytrace-echeck-account-number' ) );
	}
	
	/**
	 * @since 2.4.3
	 *
	 * @param string $value The Check routing number
	 */
	public function set_check_routing( $value ) {
		$this->echeck_routing = $this->remove_spaces( $value );
	}
	
	/**
	 * @since 2.4.3
	 *
	 * @param string $value The Check account number
	 */
	public function set_check_account( $value ) {
		$this->echeck_account = $this->remove_spaces( $value );
	}
	
	/**
	 * Validates the payment information submitted during checkout
	 *
	 * @since 2.0
	 *
	 * @throws \Exception
	 */
	public function validate_submitted_fields() {
		// If Check is chosen
		if ( \WC_Paytrace::get_post( 'paytrace_type_choice' ) == 'check' ) {
			if ( 'yes' != $this->client->get_save_customers()
			     || ( $this->client->get_gateway()->get_submitted_token_value( 'check' ) === 'new' || ! is_user_logged_in() )
			) {
				$this->set_echeck_details();
				
				// Check and validate the posted card fields
				$this->validate_echeck_fields( $this->echeck_routing, $this->echeck_account );
			}
		} else {
			if ( 'yes' != $this->client->get_save_customers()
			     || ( $this->client->get_gateway()->get_submitted_token_value( 'card' ) === 'new' || ! is_user_logged_in() )
			) {
				// Set the card details
				$this->set_credit_card_details();
				
				// Check and validate the posted card fields
				$this->validate_card_fields(
					$this->cc_number,
					$this->cc_cvc,
					$this->cc_exp_month,
					$this->cc_exp_year,
					$this->get_card_type_from_number( $this->cc_number )
				);
			}
		}
	}
	
	/**
	 * Validate the Check fields
	 *
	 * @param mixed $echeck_routing
	 * @param mixed $echeck_account
	 *
	 * @throws \Exception
	 */
	public function validate_echeck_fields( $echeck_routing, $echeck_account ) {
		if ( empty( $echeck_routing ) ) {
			throw new \Exception( __( 'Check Routing Number Required.', \WC_Paytrace::TEXT_DOMAIN ) );
		} else {
			if ( ! is_numeric( $echeck_routing ) ) {
				throw new \Exception( __( 'Invalid Check Routing Number.', \WC_Paytrace::TEXT_DOMAIN ) );
			}
		}
		
		if ( empty( $echeck_account ) ) {
			throw new \Exception( __( 'Check Account Number Required.', \WC_Paytrace::TEXT_DOMAIN ) );
		} else {
			if ( ! is_numeric( $echeck_account ) ) {
				throw new \Exception( __( 'Invalid Check Account Number.', \WC_Paytrace::TEXT_DOMAIN ) );
			}
		}
	}
	
	/**
	 * Set credit card details into object variable
	 */
	public function set_credit_card_details() {
		$cc_exp_date = $this->format_exp_date( $this->remove_spaces( \WC_Paytrace::get_post( 'paytrace-card-expiry' ) ) );
		$this->set_card_expiry_month( $cc_exp_date['month'] );
		$this->set_card_expiry_year( $cc_exp_date['year'] );
		
		if ( $this->client->use_encryption() ) {
			$this->set_card_number( \WC_Paytrace::get_post( 'paytrace-card-number-encrypted' ) );
			$this->set_card_cvc( \WC_Paytrace::get_post( 'paytrace-card-cvc-encrypted' ) );
		} else {
			$this->set_card_number( \WC_Paytrace::get_post( 'paytrace-card-number' ) );
			$this->set_card_cvc( \WC_Paytrace::get_post( 'paytrace-card-cvc' ) );
			$this->set_card_type( $this->get_card_type_from_number( $this->cc_number ) );
		}
	}
	
	/**
	 * @since 2.4.3
	 *
	 * @param string $value The Card number
	 */
	public function set_card_number( $value ) {
		$this->cc_number = $this->remove_spaces( $value );
	}
	
	/**
	 * @since 2.4.3
	 *
	 * @param string $value The Card CVC
	 */
	public function set_card_cvc( $value ) {
		$this->cc_cvc = $this->remove_spaces( $value );
	}
	
	/**
	 * @since 2.4.3
	 *
	 * @param string $value The Card expiry month
	 */
	public function set_card_expiry_month( $value ) {
		$this->cc_exp_month = $value;
	}
	
	/**
	 * @since 2.4.3
	 *
	 * @param string $value The Card expiry year
	 */
	public function set_card_expiry_year( $value ) {
		if ( strlen( $value ) == 4 ) {
			$value -= 2000;
		}
		$this->cc_exp_year = $value;
	}
	
	/**
	 * @since 2.4.3
	 *
	 * @param string $value The Card type [visa,mc ...]
	 */
	public function set_card_type( $value ) {
		$this->cc_type = $value;
	}
	
	/**
	 * Format and return the expiration month and year
	 *
	 * @param $expiration_date
	 *
	 * @return array
	 */
	public function format_exp_date( $expiration_date ) {
		$cc_exp_date = str_replace( ' ', '', explode( '/', $expiration_date ) );
		
		return array(
			'month' => \WC_Paytrace::get_field( 0, $cc_exp_date, '' ),
			'year'  => \WC_Paytrace::get_field( 1, $cc_exp_date, '' ),
		);
	}
	
	/**
	 * Returns the card type from the card number
	 *
	 * @param        $cc_number
	 *
	 * @return string
	 */
	public function get_card_type_from_number( $cc_number ) {
		if ( preg_match( '/^4[0-9]{12}(?:[0-9]{3})?$/', $cc_number ) ) {
			$type = 'VISA';
		} elseif ( preg_match( '/^5[1-5]\d{14}$|^2(?:2(?:2[1-9]|[3-9]\d)|[3-6]\d\d|7(?:[01]\d|20))\d{12}$/', $cc_number ) ) {
			$type = 'MC';
		} elseif ( preg_match( '/^3[47][0-9]{13}$/', $cc_number ) ) {
			$type = 'AMEX';
		} elseif ( preg_match( '/^6(?:011|5[0-9]{2})[0-9]{12}$/', $cc_number ) ) {
			$type = 'DISCOVER';
		} elseif ( preg_match( '/^(?:2131|1800|35\d{3})\d{11}$/', $cc_number ) ) {
			$type = 'JCB';
		} else {
			$type = '';
		}
		
		return $type;
	}
	
	/**
	 * Matches the card number to a card type
	 *
	 * @since 2.2
	 *
	 * @param $number
	 *
	 * @return string
	 */
	public function get_card_type_from_first_six( $number ) {
		
		$number = substr( $number, 0, 6 );
		
		if ( preg_match( '/^4[0-9]{5}$/', $number ) ) {
			$type = 'VISA';
		} elseif ( preg_match( '/^5[1-5]\d{4}|^2(?:2(?:2[1-9]|[3-9]\d)|[3-6]\d\d|7(?:[01]\d|20))\d{2}/', $number ) ) {
			$type = 'MC';
		} elseif ( preg_match( '/^3[47][0-9]{4}/', $number ) ) {
			$type = 'AMEX';
		} elseif ( preg_match( '/^6[045][0-9]{4}|^622[0-9]{3}/', $number ) ) {
			$type = 'DISCOVER';
		} elseif ( preg_match( '/^35[0-9]{4}/', $number ) ) {
			$type = 'JCB';
		} else {
			$type = '';
		}
		
		return $type;
	}
	
	/**
	 * Check and validate the card fields
	 *
	 * @param null $cc_number    Credit card number
	 * @param null $cc_cvc       Card Security Code
	 * @param null $cc_exp_month Expiration month
	 * @param null $cc_exp_year  Expiration year
	 * @param null $cc_type      Card Type
	 *
	 * @throws \Exception
	 */
	public function validate_card_fields( $cc_number = null, $cc_cvc = null, $cc_exp_month = null, $cc_exp_year = null, $cc_type = null ) {
		$this->validate_card_number( $cc_number, $cc_type );
		
		if ( '' != $cc_cvc || $this->client->get_gateway()->is_cvc_required() ) {
			$this->validate_cvc_number( $cc_cvc );
		}
		
		$this->validate_card_expiry_date( $cc_exp_month, $cc_exp_year );
	}
	
	/**
	 * Runs checks against card number
	 *
	 * @param $cc_number
	 * @param $cc_type
	 *
	 * @throws \Exception
	 */
	public function validate_card_number( $cc_number, $cc_type = null ) {
		if ( '' !== $this->client->get_gateway()->get_security_type() ) {
			return true;
		}
		
		if ( empty( $cc_number ) ) {
			throw new \Exception( __( 'Card Number Required', \WC_Paytrace::TEXT_DOMAIN ) );
		}
		
		if ( ! is_numeric( $cc_number ) || ( strlen( $cc_number ) > 19 || strlen( $cc_number ) < 12 ) ) {
			throw new \Exception( __( 'Invalid Card Number', \WC_Paytrace::TEXT_DOMAIN ) );
		}
		
		$this->card_number_check( $cc_number, $cc_type );
		
		// Check the cc type: present and available
		if ( empty( $cc_type ) || ! in_array( $cc_type, $this->client->get_available_cc() ) ) {
			throw new \Exception( __( 'Invalid Card Number', \WC_Paytrace::TEXT_DOMAIN ) );
		}
		
		if ( ! $this->luhn_check( $cc_number ) ) {
			throw new \Exception( __( 'Invalid Card Number', \WC_Paytrace::TEXT_DOMAIN ) );
		}
	}
	
	/**
	 * Validates Card expiry date
	 *
	 * @param $cc_exp_month
	 * @param $cc_exp_year
	 *
	 * @throws \Exception
	 */
	public function validate_card_expiry_date( $cc_exp_month, $cc_exp_year ) {
		if ( 'protect' === $this->client->get_gateway()->get_security_type() ) {
			return true;
		}
		
		if ( empty( $cc_exp_month ) ) {
			throw new \Exception( __( 'Expiration month required.', \WC_Paytrace::TEXT_DOMAIN ) );
		}
		
		if ( empty( $cc_exp_year ) ) {
			throw new \Exception( __( 'Expiration year required.', \WC_Paytrace::TEXT_DOMAIN ) );
		}
		
		// Check exp date
		$current_year  = date( 'y' );
		$current_month = date( 'n' );
		if ( ! is_numeric( $cc_exp_year ) ||
		     $cc_exp_year < $current_year ||
		     $cc_exp_year > ( $current_year + 10 ) ||
		     ! is_numeric( $cc_exp_month ) ||
		     $cc_exp_month < 0 ||
		     $cc_exp_month > 12 ||
		     ( $cc_exp_year == $current_year && $cc_exp_month < $current_month )
		) {
			throw new \Exception( __( 'Invalid expiration date.', \WC_Paytrace::TEXT_DOMAIN ) );
		}
	}
	
	/**
	 * Validates CVC number
	 *
	 * @param int         $cc_cvc
	 * @param string|null $cc_type
	 *
	 * @throws \Exception
	 */
	public function validate_cvc_number( $cc_cvc, $cc_type = null ) {
		
		if ( '' !== $this->client->get_gateway()->get_security_type() ) {
			return true;
		}
		
		// Check credit card CV2 number
		if ( empty( $cc_cvc ) ) {
			throw new \Exception( __( 'CVC number required.', \WC_Paytrace::TEXT_DOMAIN ) );
		}
		
		// Should be numbers only
		if ( ! is_numeric( $cc_cvc ) ) {
			throw new \Exception( __( 'Invalid Card Verification Code. Only numbers are allowed.', \WC_Paytrace::TEXT_DOMAIN ) );
		}
		// Should be 4 digit for AMEX and 3 for all other types
		if ( ( $cc_type == 'AMEX' && strlen( $cc_cvc ) != 4 ) ||
		     ( in_array( $cc_type, array(
				     'VISA',
				     'MC',
				     'DISCOVER',
				     'JCB',
			     ) ) && strlen( $cc_cvc ) != 3 )
		) {
			throw new \Exception( __( 'Invalid Card Verification Code length.', \WC_Paytrace::TEXT_DOMAIN ) );
		}
	}
	
	/**
	 * Validate the card number against its type and luhn's check.
	 *
	 * @param int    $cc_number The card number, checked to be only a numbers
	 * @param string $cc_type   The card type
	 *
	 * @return void
	 * @throws \Exception
	 */
	public function card_number_check( $cc_number, $cc_type = '' ) {
		// Match the cc number with the type
		switch ( $cc_type ) :
			case 'VISA' :
				if ( ! preg_match( '/^4[0-9]{12}(?:[0-9]{3})?$/', $cc_number ) ) {
					throw new \Exception( __( 'Invalid Visa card number.', \WC_Paytrace::TEXT_DOMAIN ) );
				}
				break;
			case 'MC' :
				if ( ! preg_match( '/^5[1-5]\d{14}$|^2(?:2(?:2[1-9]|[3-9]\d)|[3-6]\d\d|7(?:[01]\d|20))\d{12}$/', $cc_number ) ) {
					throw new \Exception( __( 'Invalid Master Card number.', \WC_Paytrace::TEXT_DOMAIN ) );
				}
				break;
			case 'AMEX' :
				if ( ! preg_match( '/^3[47][0-9]{13}$/', $cc_number ) ) {
					throw new \Exception( __( 'Invalid American Express card number.', \WC_Paytrace::TEXT_DOMAIN ) );
				}
				break;
			case 'DISCOVER' :
				if ( ! preg_match( '/^6(?:011|5[0-9]{2})[0-9]{12}$/', $cc_number ) ) {
					throw new \Exception( __( 'Invalid Discover card number.', \WC_Paytrace::TEXT_DOMAIN ) );
				}
				break;
			case 'JCB' :
				if ( ! preg_match( '/^(?:2131|1800|35\d{3})\d{11}$/', $cc_number ) ) {
					throw new \Exception( __( 'Invalid JCB card number.', \WC_Paytrace::TEXT_DOMAIN ) );
				}
				break;
			default :
				if ( ! preg_match( '/^(?:4[0-9]{12}(?:[0-9]{3})?|5[1-5][0-9]{14}|6(?:011|5[0-9][0-9])[0-9]{12}|3[47][0-9]{13}|3(?:0[0-5]|[68][0-9])[0-9]{11}|(?:2131|1800|35\d{3})\d{11})$/', $cc_number ) ) {
					throw new \Exception( __( 'Invalid Card Number', \WC_Paytrace::TEXT_DOMAIN ) );
				}
				break;
		endswitch;
	}
	
	/**
	 * Perform a luhn check on the card number<br/>
	 * This will not ensure the card is legitimate, <br/>
	 * it will only ensure the number passes checksum.
	 *
	 * @param int $cc_number The credit card number
	 *
	 * @return bool
	 */
	protected function luhn_check( $cc_number ) {
		// Reverse the number to start from the back
		$cc_number = strrev( $cc_number );
		$sum       = 0;
		
		for ( $i = 0; $i < strlen( $cc_number ); $i ++ ) {
			$n = substr( $cc_number, $i, 1 );
			// Double every second digit
			$n = ( $i % 2 ) ? $n * 2 : $n;
			// Sum the digits
			$sum += $n > 9 ? $n - 9 : $n;
		}
		
		return ( $sum % 10 == 0 );
	}
	
	/**
	 * Remove spaces
	 *
	 * @param string $string Input string
	 *
	 * @return string The string with removed spaces
	 */
	public function remove_spaces( $string ) {
		return str_replace( array( ' ', '-' ), '', $string );
	}
	
	/**
	 * Decodes all HTML entities, including numeric and hexadecimal ones.
	 *
	 * @param        $string
	 * @param int    $quote_style
	 * @param string $charset
	 *
	 * @return mixed|string
	 */
	function html_entity_decode_numeric( $string, $quote_style = ENT_COMPAT, $charset = 'utf-8' ) {
		$string = str_replace( '+', '', $string );
		$string = html_entity_decode( $string, $quote_style, $charset );
		$string = preg_replace_callback( '~&#x([0-9a-fA-F]+);~i', array(
			$this,
			'chr_utf8_callback',
		), $string );
		$string = preg_replace_callback( '~&#([0-9]+);~', array( $this, 'chr_utf8' ), $string );
		$string = str_ireplace( '&apos;', "'", $string );
		
		return $string;
	}
	
	/**
	 * Callback helper
	 */
	function chr_utf8_callback( $matches ) {
		return $this->chr_utf8( hexdec( $matches[1] ) );
	}
	
	/**
	 * Multi-byte chr(): Will turn a numeric argument into a UTF-8 string.
	 *
	 * @param mixed $num
	 *
	 * @return string
	 */
	function chr_utf8( $num ) {
		if ( $num[1] < 128 ) {
			return chr( $num[1] );
		}
		if ( $num[1] < 2048 ) {
			return chr( ( $num[1] >> 6 ) + 192 ) . chr( ( $num[1] & 63 ) + 128 );
		}
		if ( $num[1] < 65536 ) {
			return chr( ( $num[1] >> 12 ) + 224 ) . chr( ( ( $num[1] >> 6 ) & 63 ) + 128 ) . chr( ( $num[1] & 63 ) + 128 );
		}
		if ( $num[1] < 2097152 ) {
			return chr( ( $num[1] >> 18 ) + 240 ) . chr( ( ( $num[1] >> 12 ) & 63 ) + 128 ) . chr( ( ( $num[1] >> 6 ) & 63 ) + 128 ) . chr( ( $num[1] & 63 ) + 128 );
		}
		
		return '';
	}
	
	/**
	 * Remove restricted characters from the string
	 *
	 * @since 2.3.1
	 *
	 * @param $string
	 *
	 * @return mixed
	 */
	public function remove_restricted_characters( $string ) {
		return str_replace( $this->restricted_characters, '', $string );
	}
}