<?php

namespace WcPaytrace\Abstracts;

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Description
 *
 * @since  2.0
 * @author VanboDevelops
 *
 *        Copyright: (c) 2017 VanboDevelops
 *        License: GNU General Public License v3.0
 *        License URI: http://www.gnu.org/licenses/gpl-3.0.html
 */
abstract class Response {
	
	public $accepted_response;
	public $accepted_response_body;
	
	/**
	 * Response constructor.
	 *
	 * @param array $response HTTP response. The response from a 'wp_remote_post'
	 */
	public function __construct( $response ) {
		$this->parse( $response );
	}
	
	public function parse( $response ) {
		$body                         = wp_remote_retrieve_body( $response );
		$this->accepted_response      = $response;
		$this->accepted_response_body = $this->parse_response_body( $body );
	}

	/**
	 * Searches value for undefined property
	 *
	 * @param $key
	 *
	 * @return mixed
	 */
	public function __get( $key ) {
		return $this->get_prop( $this->get_accepted_response_body(), $key, null );
	}
	
	/**
	 * Parses the response body into and object or returns false
	 *
	 * @param $body
	 *
	 * @return object|bool
	 */
	public function parse_response_body( $body ) {
		return $body;
	}
	
	/**
	 * Returns headers response message
	 * @since 2.0
	 * @return mixed
	 */
	public function get_headers_response_code() {
		return $this->get_prop( $this->get_prop( $this->get_accepted_response(), 'response', array() ), 'code', 0 );
	}
	
	/**
	 * Returns headers response message
	 * @since 2.0
	 * @return mixed
	 */
	public function get_headers_response_message() {
		return $this->get_prop( $this->get_prop( $this->get_accepted_response(), 'response', array() ), 'message', '' );
	}
	
	/**
	 * Returns whether request error occurred
	 *
	 * @since 2.0
	 *
	 * @return bool
	 */
	public function did_error_occur() {
		$error = $this->get_error();

		return empty( $error );
	}
	
	public function was_transaction_approved() {
		return false == $this->did_error_occur() && $this->get_success() && '' == $this->get_check_transaction_id();
	}
	
	public function was_echeck_transaction_approved() {
		return false == $this->did_error_occur() && $this->get_success() && '' != $this->get_check_transaction_id();
	}
	
	public function was_customer_request_approved() {
		return $this->was_transaction_approved() && '' != $this->get_customer_id();
	}
	
	public function was_refund_approved( $type = 'card' ) {
		return $this->was_transaction_approved() || $this->was_echeck_transaction_approved();
	}
	
	public function was_capture_approved() {
		return $this->was_transaction_approved() || $this->was_echeck_transaction_approved();
	}
	
	/**==============================================
	 * Getters and Setters
	 * ================================================*/
	
	/**
	 * Checks and returns the value for the provided name
	 *
	 * @param array|object $stack
	 * @param string       $name
	 * @param string       $default If not found, we return
	 *
	 * @return mixed
	 */
	public function get_prop( $stack, $name, $default = '' ) {
		if ( is_array( $stack ) && isset( $stack[ $name ] ) ) {
			return $stack[ $name ];
		}
		
		if ( is_object( $stack ) && isset( $stack->{$name} ) ) {
			return $stack->{$name};
		}
		
		return $default;
	}
	
	public function get_success() {
		return $this->get_prop( $this->get_accepted_response_body(), 'success', false );
	}
	
	public function get_response_code() {
		return $this->get_prop( $this->get_accepted_response_body(), 'response_code', 0 );
	}
	
	public function get_response() {
		return $this->get_prop( $this->get_accepted_response_body(), 'status_message' );
	}
	
	public function get_access_token() {
		return $this->get_prop( $this->get_accepted_response_body(), 'access_token' );
	}
	
	public function get_token_type() {
		return $this->get_prop( $this->get_accepted_response_body(), 'token_type' );
	}
	
	public function get_expires_in() {
		return $this->get_prop( $this->get_accepted_response_body(), 'expires_in' );
	}
	
	public function get_created_at() {
		return $this->get_prop( $this->get_accepted_response_body(), 'created_at' );
	}
	
	public function get_status_message() {
		return $this->get_prop( $this->get_accepted_response_body(), 'status_message' );
	}
	
	public function get_approval_code() {
		return $this->get_prop( $this->get_accepted_response_body(), 'approval_code' );
	}
	
	public function get_approval_message() {
		return $this->get_prop( $this->get_accepted_response_body(), 'approval_message' );
	}
	
	public function get_avs_response() {
		return $this->get_prop( $this->get_accepted_response_body(), 'avs_response' );
	}
	
	public function get_csc_response() {
		return $this->get_prop( $this->get_accepted_response_body(), 'csc_response' );
	}
	
	public function get_transaction_id() {
		
		// If we have a check transaction id, we will return that
		if ( '' != $this->get_check_transaction_id() ) {
			return $this->get_check_transaction_id();
		}
		
		return $this->get_prop( $this->get_accepted_response_body(), 'transaction_id', '' );
	}
	
	public function get_check_transaction_id() {
		return $this->get_prop( $this->get_accepted_response_body(), 'check_transaction_id', '' );
	}
	
	public function get_external_transaction_id() {
		return $this->get_prop( $this->get_accepted_response_body(), 'external_transaction_id' );
	}
	
	public function get_masked_card_number() {
		return $this->get_prop( $this->get_accepted_response_body(), 'masked_card_number' );
	}
	
	public function get_customer_id() {
		return $this->get_prop( $this->get_accepted_response_body(), 'customer_id' );
	}
	
	public function get_profile_id() {
		return $this->get_prop( $this->get_accepted_response_body(), 'customer_id' );
	}
	
	public function get_customers() {
		return $this->get_prop( $this->get_accepted_response_body(), 'customers', array() );
	}
	
	public function get_batch() {
		return $this->get_prop( $this->get_accepted_response_body(), 'batch', false );
	}
	
	public function get_batches() {
		return $this->get_prop( $this->get_accepted_response_body(), 'batches', array() );
	}
	
	public function get_recurrence() {
		return $this->get_prop( $this->get_accepted_response_body(), 'recurrence', false );
	}
	
	public function get_recurrences() {
		return $this->get_prop( $this->get_accepted_response_body(), 'recurrences', array() );
	}
	
	public function get_transactions() {
		return $this->get_prop( $this->get_accepted_response_body(), 'transactions', array() );
	}
	
	public function get_created() {
		return $this->get_prop( $this->get_accepted_response_body(), 'created', false );
	}
	
	public function get_next_date() {
		return $this->get_prop( $this->get_accepted_response_body(), 'next_date' );
	}
	
	public function get_description() {
		return $this->get_prop( $this->get_accepted_response_body(), 'description' );
	}
	
	public function get_amount() {
		return $this->get_prop( $this->get_accepted_response_body(), 'amount', 0 );
	}
	
	public function get_discretionary_data() {
		return $this->get_prop( $this->get_accepted_response_body(), 'discretionary_data', false );
	}
	
	public function get_ach_code() {
		return $this->get_prop( $this->get_accepted_response_body(), 'ach_code', false );
	}
	
	public function get_ach_message() {
		return $this->get_prop( $this->get_accepted_response_body(), 'ach_message', false );
	}
	
	public function get_error() {
		$all_errors = $this->get_errors();
		if ( ! empty( $all_errors ) ) {
			$errors = array();
			foreach ( $all_errors as $code => $message ) {
				$errors[] = $code . ': ' . $message[0];
			}
			
			return implode( ', ', $errors );
		}
		
		return $this->get_prop( $this->get_accepted_response_body(), 'error', null );
	}
	
	public function get_errors() {
		return $this->get_prop( $this->get_accepted_response_body(), 'errors', array() );
	}
	
	public function get_error_description() {
		return $this->get_prop( $this->get_accepted_response_body(), 'error_description', null );
	}
	
	public function get_check_identifier() {
		return $this->get_check_transaction_id();
	}
	
	public function get_accepted_response_body() {
		return $this->accepted_response_body;
	}
	
	public function get_accepted_response() {
		return $this->accepted_response;
	}
}