<?php

namespace WcPaytrace\Abstracts;

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Description
 *
 * @since  2.0
 * @author VanboDevelops
 *
 *        Copyright: (c) 2017 VanboDevelops
 *        License: GNU General Public License v3.0
 *        License URI: http://www.gnu.org/licenses/gpl-3.0.html
 */
abstract class Process extends Integration {
	
	/**
	 * Integration_Post constructor.
	 *
	 * @param \WC_Paytrace_Gateway $gateway
	 */
	public function __construct( $gateway ) {
		parent::__construct( $gateway );
	}
	
	/**
	 * Returns the transaction request parameters. Needs to be extended
	 *
	 * @param \WC_Order $order
	 * @param null      $amount
	 * @param bool      $is_subscription
	 * @param bool      $is_paid_with_profile
	 *
	 * @throws \Exception
	 * @return array
	 */
	public function get_transaction_request( $order, $amount = null, $is_subscription = false, $is_paid_with_profile = false ) {
		_doing_it_wrong( 'WcPaytrace\Abstracts\Process::get_transaction_request', sprintf( __( "Method '%s' must be overridden.", \WC_Paytrace::TEXT_DOMAIN ), __METHOD__ ), '2.0' );
		
		return array();
	}
	
	/**
	 * Returns the refund request parameters
	 *
	 * @param \WC_Order $order
	 * @param null      $amount
	 * @param string    $reason
	 *
	 * @throws \Exception
	 * @return array
	 */
	public function get_refund_request( \WC_Order $order, $amount = null, $reason = '' ) {
		_doing_it_wrong( 'WcPaytrace\Abstracts\Process::get_refund_request', sprintf( __( "Method '%s' must be overridden.", \WC_Paytrace::TEXT_DOMAIN ), __METHOD__ ), '2.0' );
		
		return array();
	}
	
	/**
	 * Returns the capture request parameters
	 *
	 * @param \WC_Order $order
	 * @param null      $amount
	 *
	 * @return array
	 */
	public function get_capture_request( \WC_Order $order, $amount ) {
		_doing_it_wrong( 'WcPaytrace\Abstracts\Process::get_capture_request', sprintf( __( "Method '%s' must be overridden.", \WC_Paytrace::TEXT_DOMAIN ), __METHOD__ ), '2.0' );
		
		return array();
	}
	
	/**
	 * Returns the "Create profile request" parameters
	 *
	 * @param        $order
	 * @param string $payment_type
	 *
	 * @return array|mixed
	 */
	public function get_profile_create_request( $order, $payment_type = 'card' ) {
		_doing_it_wrong( 'WcPaytrace\Abstracts\Process::get_profile_create_request', sprintf( __( "Method '%s' must be overridden.", \WC_Paytrace::TEXT_DOMAIN ), __METHOD__ ), '2.0' );
		
		return array();
	}
	
	/**
	 * Returns the "Profile Update request" parameters
	 *
	 * @param \WC_Payment_Token $token
	 * @param Validator         $validator
	 *
	 * @throws \Exception
	 * @return array
	 */
	public function get_profile_update_request( $token, Validator $validator ) {
		_doing_it_wrong( 'WcPaytrace\Abstracts\Process::get_profile_update_request', sprintf( __( "Method '%s' must be overridden.", \WC_Paytrace::TEXT_DOMAIN ), __METHOD__ ), '2.0' );
		
		return array();
	}
	
	/**
	 * Returns profile delete request parameters
	 *
	 * @since 2.0
	 *
	 * @param \WC_Payment_Token $token
	 *
	 * @return array
	 */
	public function get_profile_delete_request( $token ) {
		_doing_it_wrong( 'WcPaytrace\Abstracts\Process::get_profile_delete_request', sprintf( __( "Method '%s' must be overridden.", \WC_Paytrace::TEXT_DOMAIN ), __METHOD__ ), '2.0' );
		
		return array();
	}
	
	/**
	 * Adds the card or echeck payment details to the request
	 *
	 * @since 2.0
	 *
	 * @param array     $params
	 * @param Validator $validator
	 * @param string    $payment_type
	 *
	 * @return mixed
	 */
	public function add_card_or_echeck_details( $params, $validator, $payment_type = 'card' ) {
		_doing_it_wrong( 'WcPaytrace\Abstracts\Process::add_card_or_echeck_details', sprintf( __( "Method '%s' must be overridden.", \WC_Paytrace::TEXT_DOMAIN ), __METHOD__ ), '2.0' );
		
		return $params;
	}
	
	/**
	 * Adds the billing details to the request array
	 *
	 * @since 2.0
	 *
	 * @param array     $request
	 * @param \WC_Order $order
	 *
	 * @return array The request array
	 */
	public function add_billing_address_details_to_request( $request, $order ) {
		_doing_it_wrong( 'WcPaytrace\Abstracts\Process::add_billing_address_details_to_request', sprintf( __( "Method '%s' must be overridden.", \WC_Paytrace::TEXT_DOMAIN ), __METHOD__ ), '2.0' );
		
		return $request;
	}
	
	/**
	 * Adds the shipping details to the request
	 *
	 * @since 2.0
	 *
	 * @param $request
	 * @param $order
	 *
	 * @return mixed
	 */
	public function maybe_add_shipping_address_details_to_request( $request, $order ) {
		_doing_it_wrong( 'WcPaytrace\Abstracts\Process::maybe_add_shipping_address_details_to_request', sprintf( __( "Method '%s' must be overridden.", \WC_Paytrace::TEXT_DOMAIN ), __METHOD__ ), '2.0' );
		
		return $request;
	}
	
	/**===================================
	 * TRANSACTION Processes
	 * ===================================*/
	
	/**
	 * Process Payment.
	 *
	 * @since 2.0
	 *
	 * @param \WC_Order $order           The order
	 * @param int|float $initial_payment Total amount to be charged
	 * @param bool      $is_subscription Is the payment we are processing for a Subscription
	 *
	 * @throws \Exception
	 */
	public function process_payment( \WC_Order $order, $initial_payment = null, $is_subscription = false ) {
		$should_charge_the_profile = ( $is_subscription && $initial_payment > 0 ) || ! $is_subscription;
		
		$customer_id          = '';
		$is_paid_with_profile = $this->is_paid_with_profile( $is_subscription );
		
		if ( $is_paid_with_profile ) {
			$customer_id = $this->on_checkout_set_payment_profile( $order, $is_subscription );
		}
		
		// 2. Process transaction
		try {
			// If there is an initial payment
			if ( $should_charge_the_profile ) {
				// If a new card was used, lets use it
				if ( 'new_card' == $this->get_gateway()->get_used_payment_type()
				     // The Protect tokens can be used once and at this point we already used them to create the profile
				     && 'protect' !== $this->get_gateway()->get_security_type() ) {
					$is_paid_with_profile = false;
				}
				
				// Process a profile payment on a saved check
				$this->process_transaction( $order, $initial_payment, $is_subscription, $is_paid_with_profile );
			}
		}
		catch ( \Exception $e ) {
			try {
				\WC_Paytrace::add_debug_log( 'Integration process payment failed: ' . print_r( $e->getMessage(), true ) );
				
				// If we did create a customer in this transaction, but we could not successfully charge it,
				// then we will just remove the profile
				if ( $customer_id ) {
					
					\WC_Paytrace::add_debug_log( 'Deleting newly created profile...' );
					
					$customer_tokens      = new \WC_Paytrace_Customer_Tokens( $order->get_user_id() );
					$profile_id_to_delete = $customer_tokens->get_token_by_customer_id( $customer_id );
					
					// If we used a new card, we want to force delete it.
					$force_delete = true;
					if ( 'new_check' == $this->get_gateway()->get_used_payment_type()
					     || 'new_card' == $this->get_gateway()->get_used_payment_type()
					) {
						$force_delete = true;
					}
					
					// Delete customer profile
					$customer_tokens->delete_profile_token( $profile_id_to_delete, $force_delete );
					
					// Remove the token from the order as well
					$order_tokens = new \WC_Paytrace_Order( $order );
					$order_tokens->delete_customer_id();
				}
			}
			catch ( \Exception $delete_error ) {
				
				\WC_Paytrace::add_debug_log( 'Profile deletion failed: ' . print_r( $delete_error->getMessage(), true ) );
				
				// Throw an error with both error messages, from failed transaction and
				throw new \Exception( $e->getMessage() . ' ' . $delete_error->getMessage() );
			}
			
			// Throw an error message to be displayed to the customer
			throw new \Exception( $e->getMessage() );
		}
	}
	
	/**
	 * Handle payments with saved cards
	 *
	 * @param \WC_Order $order
	 * @param float     $amount               (Optional) Payment amount
	 * @param bool      $is_subscription      (Optional) Is the payment for a subscription product
	 * @param bool      $is_paid_with_profile (Optional) The payment type. Either 'single' payment or a 'profile' payment
	 *
	 * @throws \Exception
	 */
	public function process_transaction( \WC_Order $order, $amount = null, $is_subscription = false, $is_paid_with_profile = false ) {
		
		// Debug log
		\WC_Paytrace::add_debug_log( 'Process Transaction request for order #' . \WC_Paytrace_Compat::get_order_id( $order ) );
		
		$request = apply_filters( 'wc_paytrace_transaction_request', $this->get_transaction_request( $order, $amount, $is_subscription, $is_paid_with_profile ), $order, $amount, $is_subscription, $is_paid_with_profile );
		
		// Debug
		$this->log_request( $request, 'Profile Payment Request:' );
		
		try {
			/**
			 * NOTE: The process_transaction throws an exception which if thrown will not be logged in the order notes.
			 *       We will add an additional try|catch to make sure that if an exception is thrown,
			 *       we will catch and log it to the order.
			 */
			
			$response = $this->get_api_services()->transaction()->process_transaction( $request );
		}
		catch ( \Exception $e ) {
			
			$message = sprintf( __( 'Error with the transaction response. The transaction may have been processed, please contact administrator before attempting the transaction again. Error: %s', \WC_Paytrace::TEXT_DOMAIN ), $e->getMessage() );
			
			$new_status = apply_filters( 'wc_paytrace_error_payment_may_be_processed_status', 'failed', $order );
			
			// Change the order status and add the error message
			$order->update_status( $new_status, $message );
			
			// Debug log
			\WC_Paytrace::add_debug_log( 'Remote Post connection error: ' . $message );
			
			throw new \Exception( $message );
		}
		
		$this->log_response( get_object_vars( $response ), 'Profile Payment Response:' );
		
		// There was an error, so mark the order
		if ( $response->did_error_occur() ) {
			$this->mark_payment_error( $order, $response );
		}
		
		// Transaction processed
		if ( $response->was_transaction_approved() ) {
			
			$this->mark_approved_credit_card_payment( $order, $response );
		} elseif ( $response->was_echeck_transaction_approved() ) {
			
			$this->mark_approved_echeck_payment( $order, $response );
		} else {
			// Add order note that the payment was not approved with all info supplied
			$order->add_order_note(
				sprintf(
					__(
						'PayTrace Payment Declined.
				Transaction Response: %s,
				Transaction ID: %s,
				Message: %s,
				AVS Response: %s,
				CSC Response: %s.', \WC_Paytrace::TEXT_DOMAIN
					),
					$response->get_response(),
					$response->get_transaction_id(),
					$response->get_approval_message(),
					$response->get_avs_response(),
					$response->get_csc_response()
				)
			);
			
			// Change status Failed
			$order->update_status( 'failed' );
			
			// Debug log
			\WC_Paytrace::add_debug_log( 'Payment declined. Message: ' . $response->get_response() );
			
			do_action( 'wc_paytrace_transaction_declined_response_processed', $response, $order );
			
			// Show declined to the customer
			throw new \Exception( sprintf( __( 'Transaction declined. Message: %s', \WC_Paytrace::TEXT_DOMAIN ), $response->get_response() ) );
		}
		
		do_action( 'wc_paytrace_transaction_response_processed', $response, $order, $amount, $is_subscription, $is_paid_with_profile );
	}
	
	/**
	 * Mark the order after a payment error
	 *
	 * @param \WC_Order $order
	 * @param Response  $response
	 *
	 * @throws \Exception
	 */
	public function mark_payment_error( \WC_Order $order, $response ) {
		// Add order note that the payment was not approved with the needed info
		$order->add_order_note(
			sprintf(
				__(
					'Error occurred while processing the payment.'
					. ' Error message: %s', \WC_Paytrace::TEXT_DOMAIN
				),
				$response->get_error()
			)
		);
		
		// Debug log
		\WC_Paytrace::add_debug_log( 'Error occurred while processing the payment. Error message: ' . $response->get_error() );
		
		// Change status Failed
		$order->update_status( 'failed' );
		
		// Show error to the customer
		throw new \Exception(
			sprintf(
				__(
					'Error occurred while processing the payment.'
					. ' Error message: %s', \WC_Paytrace::TEXT_DOMAIN
				),
				$response->get_error()
			)
		);
	}
	
	/**
	 * Mark Card payment as approved
	 *
	 * @param \WC_Order $order
	 * @param Response  $response
	 *
	 * @throws \Exception
	 */
	public function mark_approved_credit_card_payment( \WC_Order $order, $response ) {
		
		// Transaction was approved
		switch ( (int) $response->get_response_code() ) {
			case '101' : // Payment approved
			case '104' : // Payment approved in test mode
				// Add order note with the payment info.
				$order->add_order_note(
					sprintf(
						__(
							'PayTrace Payment Completed.
					Transaction Response: %s,
					Transaction ID: %s,
					Approval Code: %s,
					Approval Message: %s,
					AVS Response: %s,
					CSC Response: %s', \WC_Paytrace::TEXT_DOMAIN
						),
						$response->get_response(),
						$response->get_transaction_id(),
						$response->get_approval_code(),
						$response->get_approval_message(),
						$response->get_avs_response(),
						$response->get_csc_response()
					)
				);
				
				// Debug log
				\WC_Paytrace::add_debug_log( 'Payment completed.' );
				
				$order_tokens = new \WC_Paytrace_Order( $order );
				$order_tokens->save_transaction_id( $response->get_transaction_id() );
				$order_tokens->save_payment_type( 'card' );
				
				if ( 'Sale' == $this->get_gateway()->get_transaction_authorization_type( 'card', $order ) ) {
					$order_tokens->save_is_payment_captured( true );
					$order_tokens->save_order_amount_captured( $order->get_total() );
				} else {
					$order_tokens->save_is_payment_captured( false );
					$order_tokens->save_order_amount_captured( 0 );
					$order_tokens->save_order_amount_authorized( $order->get_total() );
				}
				
				// Process the order
				$order->payment_complete( $response->get_transaction_id() );
				
				do_action( 'wc_paytrace_transaction_approved_response_processed', $response, $order );
				
				break;
			case '103' : // Payment approved, but voided due to AVS and/or CVC mismatch
				
				// Add order note that the payment was not approved with the needed info
				$order->add_order_note(
					sprintf(
						__( 'PayTrace payment was approved, but voided because of address and/or CSC mismatch.
					Transaction Response: %s,
					Transaction ID: %s,
					Decline Code: %s,
					Message: %s,
					AVS Response: %s,
					CSC Response: %s.', \WC_Paytrace::TEXT_DOMAIN
						),
						$response->get_response(),
						$response->get_transaction_id(),
						$response->get_approval_code(),
						$response->get_approval_message(),
						$response->get_avs_response(),
						$response->get_csc_response()
					)
				);
				
				// Change status Failed
				$order->update_status( 'failed' );
				
				// Debug log
				\WC_Paytrace::add_debug_log( 'Payment declined. Code 103. Message: ' . $response->get_approval_message() );
				
				do_action( 'wc_paytrace_transaction_declined_response_processed', $response, $order );
				
				// Show declined to the customer
				throw new \Exception(
					sprintf(
						__(
							'Transaction declined.'
							. ' Message: %s', \WC_Paytrace::TEXT_DOMAIN
						), $response->get_approval_message()
					)
				);
				
				break;
			default : // Any other code did not approve the payment
				
				// Add order note that the payment was not approved with the needed info
				$order->add_order_note(
					sprintf(
						__(
							'PayTrace Payment Declined.
					Transaction Response: %s,
					Transaction ID: %s,
					Decline Code: %s,
					Message: %s,
					AVS Response: %s,
					CSC Response: %s.', \WC_Paytrace::TEXT_DOMAIN
						),
						$response->get_response(),
						$response->get_transaction_id(),
						$response->get_approval_code(),
						$response->get_approval_message(),
						$response->get_avs_response(),
						$response->get_csc_response()
					)
				);
				
				// Change status Failed
				$order->update_status( 'failed' );
				
				// Debug log
				\WC_Paytrace::add_debug_log( 'Payment declined. Message: ' . $response->get_approval_message() );
				
				do_action( 'wc_paytrace_transaction_declined_response_processed', $response, $order );
				
				// Show declined to the customer
				throw new \Exception(
					sprintf(
						__(
							'Transaction declined.'
							. ' Message: %s', \WC_Paytrace::TEXT_DOMAIN
						),
						$response->get_approval_message()
					)
				);
				
				break;
		}
	}
	
	/**
	 * Mark approved Check payment
	 *
	 * @param \WC_Order $order
	 * @param Response  $response
	 *
	 * @throws \Exception
	 */
	public function mark_approved_echeck_payment( \WC_Order $order, $response ) {
		switch ( (int) $response->get_response_code() ) :
			case '120' : // Payment approved
			case '121' : // Payment approved in test mode
				// Add order note with the payment info.
				$order->add_order_note(
					sprintf(
						__(
							'PayTrace Check/Ach Payment Completed.
					Transaction Response: %s,
					Check Identifier: %s,
					Approval Code: %s,
					Approval Message: %s
					', \WC_Paytrace::TEXT_DOMAIN
						),
						$response->get_response(),
						$response->get_check_identifier(),
						$response->get_ach_code(),
						$response->get_ach_message()
					)
				);
				
				// Debug log
				\WC_Paytrace::add_debug_log( 'PayTrace Check/Ach Payment Accepted.' );
				
				$order_tokens = new \WC_Paytrace_Order( $order );
				$order_tokens->save_transaction_id( $response->get_check_identifier() );
				$order_tokens->save_payment_type( 'check' );
				
				if ( 'Sale' == $this->get_gateway()->get_transaction_authorization_type( 'check', $order ) ) {
					$order_tokens->save_is_payment_captured( true );
					$order_tokens->save_order_amount_captured( $order->get_total() );
				} else {
					$order_tokens->save_is_payment_captured( false );
					$order_tokens->save_order_amount_captured( 0 );
					$order_tokens->save_order_amount_authorized( $order->get_total() );
				}
				
				// Change status on-hold
				if ( $this->is_ach_payment_on_hold() ) {
					$order->update_status( 'on-hold' );
				} else {
					// Process the order
					$order->payment_complete( $response->get_check_identifier() );
				}
				
				do_action( 'wc_paytrace_echeck_transaction_approved_response_processed', $response, $order );
				
				break;
			default : // Any other code did not approve the payment
				
				// Add order note that the payment was not approved with the needed info
				$order->add_order_note(
					sprintf(
						__(
							'PayTrace Check/ACH Payment Declined.
					Transaction Response: %s,
					Check Identifier: %s,
					Approval Code: %s,
					Approval Message: %s
					', \WC_Paytrace::TEXT_DOMAIN
						),
						$response->get_response(),
						$response->get_check_identifier(),
						$response->get_ach_code(),
						$response->get_ach_message()
					)
				);
				
				// Change status Failed
				$order->update_status( 'failed' );
				
				// Debug log
				\WC_Paytrace::add_debug_log( 'Payment declined. Message: ' . $response->get_approval_message() );
				
				do_action( 'wc_paytrace_echeck_transaction_declined_response_processed', $response, $order );
				
				// Show declined to the customer
				throw new \Exception(
					sprintf(
						__(
							'Transaction declined.'
							. ' Message: %s', \WC_Paytrace::TEXT_DOMAIN
						),
						$response->get_approval_message()
					)
				);
				
				break;
		endswitch;
	}
	
	/**
	 * Processes the Post integration refund
	 *
	 * @param \WC_Order $order
	 * @param null      $amount
	 * @param string    $reason
	 *
	 * @return bool
	 * @throws \Exception
	 */
	public function process_refund( \WC_Order $order, $amount = null, $reason = '' ) {
		
		$request = apply_filters( 'wc_paytrace_refund_request', $this->get_refund_request( $order, $amount, $reason ), $order, $amount, $reason );
		
		$order_tokens = new \WC_Paytrace_Order( $order );
		$payment_type = $order_tokens->get_payment_type();
		
		// Debug
		$this->log_request( $request, 'Refund Request:' );
		
		try {
			// Process the request which will format the request, send it to the API, and parse the response
			$response = $this->get_api_services()->refund()->process_refund( $request );
		}
		catch ( \Exception $e ) {
			$message = sprintf( __( 'Error with the refund request. The refund may still have been processed, please double check in the Paytrace Dashboard before attempting it again. Error: %s', \WC_Paytrace::TEXT_DOMAIN ), $e->getMessage() );
			
			$order->add_order_note( $message );
			
			// Debug log
			\WC_Paytrace::add_debug_log( 'process_refund: Remote Post connection error: ' . $message );
			
			throw new \Exception( $message );
		}
		
		$this->log_response( get_object_vars( $response ), 'Refund Response:' );
		
		// There was an error, so mark the order
		if ( $response->did_error_occur() ) {
			// TODO: New: On error "817" : "The Transaction ID that you provided could not be refunded. Only settled transactions can be refunded. Please try to void the transaction instead.", perform a void transaction attempt. This will serve the cases where a refund is attempted before the transaction was settled by the system.
			throw new \Exception( sprintf( __( 'Refund not successful. Error Message: %s.', \WC_Paytrace::TEXT_DOMAIN ), $response->get_error() ) );
		}
		
		if ( $response->was_refund_approved( $payment_type ) ) {
			// Refund was approved
			switch ( (int) $response->get_response_code() ) {
				case '106' : // Your transaction was successfully refunded.
				case '108' : // Your TEST transaction was successfully refunded HOWEVER, NO FUNDS WILL BE REFUNDED.
				case '122' : // Your check was successfully refunded.
				case '123' : // Your TEST check was successfully refunded. HOWEVER, NO FUNDS WILL BE TRANSFERRED.
					// Debug log
					\WC_Paytrace::add_debug_log( 'Refund completed.' );
					
					// Add order note
					$order->add_order_note( sprintf( __( 'Refunded %s. Refund ID: %s. %s', \WC_Paytrace::TEXT_DOMAIN ),
						$amount,
						$response->get_transaction_id(),
						( '' != $reason ) ? sprintf(
							__(
								'Credit Note: %s.'
							), $reason
						) : ''
					) );
					
					do_action( 'wc_paytrace_refund_approved_response_processed', $response, $order, $payment_type, $amount, $reason );
					
					return true;
					
					break;
				case '107' : // Your transaction was not successfully refunded.
				default :
					// Add order note
					$order->add_order_note( sprintf( __( 'Refund declined. Message: %s.', \WC_Paytrace::TEXT_DOMAIN ), $response->get_response() ) );
					
					do_action( 'wc_paytrace_refund_declined_response_processed', $response, $order, $payment_type, $amount, $reason );
					
					break;
			}
		} else {
			$message = sprintf( __( 'Refund not successful. Message: %s.', \WC_Paytrace::TEXT_DOMAIN ), $response->get_response() );
			
			// Add order note
			$order->add_order_note( $message );
			
			do_action( 'wc_paytrace_refund_declined_response_processed', $response, $order, $payment_type, $amount, $reason );
			
			// Refund was not successful
			throw new \Exception( $message );
		}
		
		return false;
	}
	
	/**
	 * @param \WC_Order $order
	 * @param float     $amount
	 *
	 * @throws \Exception
	 *
	 * @return mixed|\WcPaytrace\Api\Json\Response|\WcPaytrace\Api\Post\Response
	 */
	public function process_capture( $order, $amount ) {
		$request = apply_filters( 'wc_paytrace_capture_request', $this->get_capture_request( $order, $amount ), $order, $amount );
		
		// Debug
		$this->log_request( $request, 'Capture Request:' );
		
		try {
			// Process the request which will format the request, send it to the API, and parse the response
			$response = $this->get_api_services()->capture()->process_capture( $request );
		}
		catch ( \Exception $e ) {
			
			$message = sprintf( __( 'Error with the capture request. The capture may still have been processed, please double check in the Paytrace Dashboard before attempting it again. Error: %s', \WC_Paytrace::TEXT_DOMAIN ), $e->getMessage() );
			
			// Debug log
			\WC_Paytrace::add_debug_log( 'process_capture: Remote Post connection error: ' . $message );
			
			throw new \Exception( $message );
		}
		
		$this->log_response( get_object_vars( $response ), 'Capture Response:' );
		
		return $response;
	}
	
	/**
	 * Runs the process of sending a Paytrace transaction email receipt to the customer
	 *
	 * @since 2.3.0
	 *
	 * @param \WC_Order $order
	 *
	 * @return void
	 */
	public function send_transaction_receipt( $order ) {
		$request = apply_filters( 'wc_paytrace_email_receipt_request', $this->get_email_receipt_request( $order ), $order );
		$error   = '';
		
		// Errors thrown here we don't need to catch and present to the user.
		try {
			// Debug
			$this->log_request( $request, 'Email Receipt Request:' );
			
			// Process the request which will format the request, send it to the API, and parse the response
			$response = $this->get_api_services()->receipt()->send_receipt( $request );
			
			$this->log_response( get_object_vars( $response ), 'Email Receipt Response:' );
		}
		catch ( \Exception $e ) {
			// Just catch the error and maybe add an order note
			$error = $e->getMessage();
		}
		
		// Allow customers disabling of the order note
		if ( apply_filters( 'wc_paytrace_email_receipt_leave_order_note', true, $order ) ) {
			$message = $error;
			if ( '' == $error ) {
				$message = __( 'Paytrace transaction email sent.', \WC_Paytrace::TEXT_DOMAIN );
				if ( isset( $response ) && $response->did_error_occur() ) {
					$message = sprintf( __( 'Paytrace transaction email receipt was not sent. Error: %s', \WC_Paytrace::TEXT_DOMAIN ), $response->get_error() );
				}
			}
			
			$order->add_order_note( $message );
		}
	}
	
	/**=================================
	 * VAULT Processes
	 * ===================================*/
	
	/**
	 * When processing payment on checkout and customer chooses to save profile,
	 * we want to setup the profile with the provided information.
	 *
	 * @param      $order
	 * @param bool $is_subscription
	 *
	 * @return bool
	 * @throws \Exception
	 */
	public function on_checkout_set_payment_profile( $order, $is_subscription = false ) {
		$customer_id = false;
		if ( 'new_check' == $this->get_gateway()->get_used_payment_type()
		     || 'new_card' == $this->get_gateway()->get_used_payment_type()
		) {
			$type = 'new_check' == $this->get_gateway()->get_used_payment_type() ? 'check' : 'card';
			// Create customer profile
			$customer_id = $this->process_profile_create( $order, $type );
		} else {
			$used_token_id = 'saved_check' == $this->get_gateway()->get_used_payment_type() ? $this->get_gateway()->get_submitted_token_value( 'check' ) : $this->get_gateway()->get_submitted_token_value( 'card' );
			
			// Save the used token to the order
			$customer_tokens = new \WC_Paytrace_Customer_Tokens( get_current_user_id() );
			$token           = $customer_tokens->get_token( $used_token_id );
			
			$order_tokens = new \WC_Paytrace_Order( $order );
			$order_tokens->save_customer_id( $token->get_token() );
			$customer_id = $token->get_token();
		}
		
		return $customer_id;
	}
	
	/**
	 * Create a customer profile on the PayTrace system.
	 *
	 * @param \WC_Order $order
	 * @param string    $payment_type Type of payment we will run, 'card' or 'check'
	 *
	 * @return string The created Customer ID
	 * @throws \Exception
	 */
	function process_profile_create( \WC_Order $order, $payment_type = 'card' ) {
		
		// Init the class
		$services  = $this->get_api_services();
		$validator = $services->get_validator();
		
		$params = apply_filters( 'wc_paytrace_profile_create_request', $this->get_profile_create_request( $order, $payment_type ), $order, $payment_type );
		
		try {
			// Run the create profile request
			$response = $services->vault()->create_profile( $params );
		}
		catch ( \Exception $e ) {
			$message = sprintf( __( 'Error with the create profile request. Error: %s', \WC_Paytrace::TEXT_DOMAIN ), $e->getMessage() );
			
			// Debug log
			\WC_Paytrace::add_debug_log( 'process_profile_create: Remote Post connection error: ' . $message );
			
			throw new \Exception( $message );
		}
		
		// Log the response
		$this->log_response( get_object_vars( $response ), 'Create Customer Profile Response:' );
		
		if ( $response->did_error_occur() ) {
			// Add order note that the payment was not approved with the needed info
			$order->add_order_note(
				sprintf(
					__(
						'Error occurred while creating a customer payment profile.'
						. ' Error message: %s', \WC_Paytrace::TEXT_DOMAIN
					),
					$response->get_error()
				)
			);
			
			// Debug log
			\WC_Paytrace::add_debug_log( 'Error occurred while creating a customer payment profile. Error message: ' . $response->get_error() );
			
			// Show error to the customer
			throw new \Exception(
				sprintf(
					__(
						'Error occurred while creating your payment profile.'
						. ' Error message: %s', \WC_Paytrace::TEXT_DOMAIN
					),
					$response->get_error()
				)
			);
		}
		
		if ( $response->was_customer_request_approved() ) {
			switch ( (int) $response->get_response_code() ) :
				case '160' : // Profile created
					// Add an order note.
					$order->add_order_note( __( 'Customer payment profile created.', \WC_Paytrace::TEXT_DOMAIN ) );
					
					// Profile was created, save the info to the customer user data
					if ( is_user_logged_in() && $this->maybe_save_customers() ) {
						
						if ( 'check' == $payment_type ) {
							$profile = array(
								'routing_last4' => substr( $validator->echeck_routing, - 4 ),
								'last4'         => substr( $validator->echeck_account, - 4 ),
								'type'          => 'Paytrace_eCheck',
							);
						} else {
							
							$last4     = substr( $validator->cc_number, - 4 );
							$exp_month = $validator->cc_exp_month;
							$exp_year  = $validator->cc_exp_year;
							$card_type = $validator->get_card_type_from_number( $validator->cc_number );
							
							if ( '' !== $this->get_gateway()->get_security_type() ) {
								$lookup = $this->export_profile( $response->get_customer_id() );
								if ( ! empty( $lookup ) && is_array( $lookup ) ) {
									$looked_customer = array_shift( $lookup );
									$card            = \WC_Paytrace::get_field( 'credit_card', $looked_customer, false );
									if ( $card ) {
										$last4     = substr( \WC_Paytrace::get_field( 'masked_number', $card, '' ), - 4 );
										$card_type = $validator->get_card_type_from_first_six( \WC_Paytrace::get_field( 'masked_number', $card, '' ) );
										if ( 'protect' == $this->get_gateway()->get_security_type() ) {
											$exp_month = \WC_Paytrace::get_field( 'expiration_month', $card, '' );
											$exp_year  = substr( \WC_Paytrace::get_field( 'expiration_year', $card, '' ), - 2 );
										}
									}
								}
							}
							
							$profile = array(
								'last4'     => $last4,
								'exp_year'  => $exp_year,
								'exp_month' => $exp_month,
								'type'      => 'Paytrace_CC',
								'card_type' => $card_type,
							);
						}
						
						$profile['token']      = $response->get_customer_id();
						$profile['profile_id'] = $response->get_profile_id();
						$new_token             = new \WC_Paytrace_Token( $profile, 0 );
						
						$customer_tokens = new \WC_Paytrace_Customer_Tokens( get_current_user_id() );
						$customer_tokens->save_profile_or_token( $new_token );
					}
					
					$order_tokens = new \WC_Paytrace_Order( $order );
					$order_tokens->save_customer_id( $response->get_customer_id() );
					
					\WC_Paytrace::add_debug_log( 'Created profile, ID: ' . $response->get_customer_id() );
					
					do_action( 'wc_paytrace_create_profile_approved_response_processed', $response, $order, $payment_type );
					
					return $response->get_customer_id();
					break;
				default : // Any other code did not create customer
					
					do_action( 'wc_paytrace_create_profile_declined_response_processed', $response, $order, $payment_type );
					
					// Show error to the customer
					throw new \Exception(
						sprintf(
							__(
								'Creating your payment profile failed.'
								. ' Error message: %s', \WC_Paytrace::TEXT_DOMAIN
							),
							$response->get_response()
						)
					);
					
					break;
			endswitch;
		} else {
			
			do_action( 'wc_paytrace_create_profile_declined_response_processed', $response, $order, $payment_type );
			
			// Show error to the customer
			throw new \Exception(
				sprintf(
					__(
						'Creating your payment profile failed.'
						. ' Error message: %s', \WC_Paytrace::TEXT_DOMAIN
					),
					$response->get_response()
				)
			);
		}
	}
	
	/**
	 * Generates the request and processes the profile update.
	 *
	 * @since 2.0
	 *
	 * @param $token_id
	 *
	 * @throws \Exception
	 */
	public function process_profile_update( $token_id ) {
		$customer_tokens = new \WC_Paytrace_Customer_Tokens( get_current_user_id() );
		$token_to_update = $customer_tokens->get_token( $token_id );
		
		$services  = $this->get_api_services();
		$validator = $services->get_validator();
		
		$request = apply_filters( 'wc_paytrace_process_profile_update_request', $this->get_profile_update_request( $token_to_update, $validator ), $token_id, $validator );
		
		// Debug
		$this->log_request( $request, 'Update Customer Profile Request:' );
		
		$response = $this->get_api_services()->vault()->update_profile( $request );
		
		$this->log_response( get_object_vars( $response ), 'Update Customer Profile Response:' );
		
		if ( $response->did_error_occur() ) {
			
			// Debug log
			\WC_Paytrace::add_debug_log( 'Error occurred while updating a customer payment profile. Error message: ' . $response->get_error() );
			
			// Show error to the customer
			throw new \Exception(
				sprintf(
					__(
						'Error occurred while updating your payment profile.'
						. ' Error message: %s', \WC_Paytrace::TEXT_DOMAIN
					),
					$response->get_error()
				)
			);
		}
		
		if ( $response->was_customer_request_approved() ) {
			switch ( (int) $response->get_response_code() ) {
				case '161' : // Profile updated
					
					if ( ! empty( $validator->echeck_routing ) ) {
						$new_profile = array(
							'routing_last4' => substr( $validator->echeck_routing, - 4 ),
							'last4'         => substr( $validator->echeck_account, - 4 ),
							'type'          => 'Paytrace_eCheck',
						);
					} else {
						$new_profile = array(
							'last4'     => substr( $validator->cc_number, - 4 ),
							'exp_year'  => $validator->cc_exp_year,
							'exp_month' => $validator->cc_exp_month,
							'card_type' => $validator->get_card_type_from_number( $validator->cc_number ),
							'type'      => 'Paytrace_CC',
						);
					}
					
					$new_profile['profile_id'] = $response->get_profile_id();
					$new_profile['token']      = $response->get_customer_id();
					
					$new_token = new \WC_Paytrace_Token( $new_profile, 0 );
					
					$customer_tokens = new \WC_Paytrace_Customer_Tokens( get_current_user_id() );
					$customer_tokens->update_profile_or_token( $new_token, $token_to_update );
					
					do_action( 'wc_paytrace_update_profile_approved_response_processed', $response, $token_id );
					
					break;
				default : // Any other code did not update customer
					
					do_action( 'wc_paytrace_update_profile_declined_response_processed', $response, $token_id );
					
					// Show error to the customer
					throw new \Exception(
						sprintf(
							__(
								'Updating your payment profile failed.'
								. ' Error message: %s', \WC_Paytrace::TEXT_DOMAIN
							),
							$response->get_response()
						)
					);
			}
		} else {
			
			do_action( 'wc_paytrace_update_profile_declined_response_processed', $response, $token_id );
			
			// Show error to the customer
			throw new \Exception(
				sprintf(
					__(
						'Updating your payment profile failed.'
						. ' Error message: %s', \WC_Paytrace::TEXT_DOMAIN
					),
					$response->get_response()
				)
			);
		}
	}
	
	/**
	 * Generates the request and runs the delete profile process
	 *
	 * @since 2.0
	 *
	 * @param $token_id
	 * @param $user_id
	 *
	 * @throws \Exception
	 */
	public function process_profile_delete( $token_id, $user_id ) {
		$user_id         = 0 < (int) $user_id ? (int) $user_id : get_current_user_id();
		$customer_tokens = new \WC_Paytrace_Customer_Tokens( $user_id );
		$token_to_delete = $customer_tokens->get_token( $token_id );
		
		$request = apply_filters( 'wc_paytrace_process_profile_delete_request', $this->get_profile_delete_request( $token_to_delete ), $token_id );
		
		// Debug
		$this->log_request( $request, 'Delete Customer Profile Request:' );
		
		$response = $this->get_api_services()->vault()->delete_profile( $request );
		
		$this->log_response( get_object_vars( $response ), 'Delete Customer Profile Response:' );
		
		if ( $response->did_error_occur() ) {
			// Debug log
			\WC_Paytrace::add_debug_log( 'Error occurred while deleting a customer payment profile. Error message: ' . $response->get_error() );
			
			switch ( (int) $response->get_error() ) {
				case '30' : // Customer ID, xxxxx, was not found or is incomplete.
					// Even though it is an error, we know from the message that this customer
					// no longer exists in the PayTrace system, so we will delete it from ours, too
					$customer_tokens->delete_profile_token( $token_to_delete );
					
					return;
			}
			
			// Show error to the customer
			throw new \Exception(
				sprintf(
					__(
						'Error occurred while deleting your payment token.'
						. ' Error message: %s', \WC_Paytrace::TEXT_DOMAIN
					),
					$response->get_error()
				)
			);
		}
		
		if ( $response->was_customer_request_approved() ) {
			switch ( (int) $response->get_response_code() ) {
				case '162' : // The customer profile for customer ID/customer Name was successfully deleted.
					
					$customer_tokens->delete_profile_token( $token_to_delete );
					
					return;
				default : // Any other code did not delete customer
					break;
			}
		}
		
		// Show error to the customer
		throw new \Exception(
			sprintf(
				__(
					'Deleting your payment token failed.'
					. ' Error message: %s', \WC_Paytrace::TEXT_DOMAIN
				),
				$response->get_response()
			)
		);
	}
	
	/**
	 * Export a Paytrace profile
	 *
	 * @since 2.2
	 *
	 * @param        $customer_id
	 * @param string $customer_email
	 *
	 * @return bool|object
	 */
	public function export_profile( $customer_id, $customer_email = '' ) {
		_doing_it_wrong( __FUNCTION__, sprintf( __( "Method '%s' must be overridden.", \WC_Paytrace::TEXT_DOMAIN ), __METHOD__ ), '2.2' );
		
		return false;
	}
	
	/**=================================
	 * GENERAL HELPERS Processes
	 * ===================================*/
	
	/**
	 * Returns the checkout payment type
	 *
	 * 'single' or 'profile'
	 *
	 * @param bool $is_subscription
	 *
	 * @return string
	 */
	public function is_paid_with_profile( $is_subscription ) {
		
		// We require the Vault to be enabled and the user to be logged in
		if ( ! $this->maybe_save_customers() || ! is_user_logged_in() ) {
			return false;
		}
		
		if ( \WC_Paytrace::get_post( 'paytrace_type_choice' ) == 'check' ) {
			// Token
			if ( 'saved_check' == $this->get_gateway()->get_used_payment_type() ) {
				return true;
			}
			
			// New check that we will create a profile for
			if ( 'new_check' == $this->get_gateway()->get_used_payment_type() && ( '1' == \WC_Paytrace::get_field( 'paytrace-save-check', $_POST, '0' ) || $is_subscription ) ) {
				return true;
			}
		} else {
			// Token
			if ( 'saved_card' == $this->get_gateway()->get_used_payment_type() ) {
				return true;
			}
			
			// New card that we will create a profile for
			if ( 'new_card' == $this->get_gateway()->get_used_payment_type() && ( '1' == \WC_Paytrace::get_field( 'paytrace-save-card', $_POST, '0' ) || $is_subscription ) ) {
				return true;
			}
		}
		
		return false;
	}
	
	/**
	 * Generate and log the request for Debug purposes
	 *
	 * @since 2.0
	 *
	 * @param array  $params
	 * @param string $message
	 */
	public function log_request( $params, $message = 'Generated request:' ) {
		// Remove possible sensitive data
		if ( is_array( $params ) ) {
			if ( 'json' == $this->get_integration() ) {
				unset( $params['encrypted_csc'] );
				unset( $params['csc'] );
				unset( $params['credit_card'] );
				unset( $params['check'] );
			} else {
				unset( $params['CSC'] );
				unset( $params['EXPMNTH'] );
				unset( $params['CC'] );
				unset( $params['EXPYR'] );
				unset( $params['DDA'] );
				unset( $params['TR'] );
			}
		}
		
		// Debug log
		\WC_Paytrace::add_debug_log( $message . ' ' . print_r( $params, true ) );
	}
	
	/**
	 * Generate and log the response for Debug purposes
	 *
	 * @since 2.0
	 *
	 * @param array  $params
	 * @param string $message
	 */
	public function log_response( $params, $message = 'Received Response:' ) {
		// Debug log
		\WC_Paytrace::add_debug_log( $message . ' ' . print_r( $params, true ) );
	}
}