<?php

namespace WcPaytrace\Abstracts;

use WcPaytrace\Api\Client;
use WcPaytrace\Api\Post\Validator;

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Description
 *
 * @since  2.0
 * @author VanboDevelops
 *
 *        Copyright: (c) 2017 VanboDevelops
 *        License: GNU General Public License v3.0
 *        License URI: http://www.gnu.org/licenses/gpl-3.0.html
 */
abstract class Integration {
	
	protected $api_client;
	
	/**
	 * Integration constructor.
	 *
	 * @param \WC_Paytrace_Gateway $gateway
	 */
	public function __construct( $gateway ) {
		$this->gateway = $gateway;
	}
	
	/**
	 * @return \WC_Paytrace_Gateway
	 */
	public function get_gateway() {
		return $this->gateway;
	}
	
	/**
	 * Loads and returns the main Paytrace API class
	 *
	 * @since 2.0
	 * @throws \Exception
	 * @return \WcPaytrace\Api\Post\Services|\WcPaytrace\Api\Json\Services
	 */
	public function get_api_services() {
		if ( null != $this->api_client ) {
			return $this->api_client;
		}
		
		$client = new Client(
			$this->get_gateway()
		);
		
		if ( 'post' == $this->get_integration() ) {
			$this->api_client = $client->get_post_services();
		} else {
			$this->api_client = $client->get_json_services();
		}
		
		return $this->api_client;
	}
	
	/**
	 * Returns the integration type
	 *
	 * @since 2.0
	 *
	 * @return string
	 */
	public function get_integration() {
		return $this->get_gateway()->get_option( 'integration' );
	}
	
	/**
	 * Returns whether the gateway is in test mode.
	 *
	 * @since 2.0
	 *
	 * @return bool
	 */
	public function is_testmode() {
		return 'yes' == $this->get_gateway()->get_option( 'testmode', 'no' );
	}
	
	/**
	 * Return whether the transaction should be deferred or not.
	 *
	 * @return bool
	 */
	public function is_authorization_only() {
		if ( 'Sale' == $this->get_gateway()->get_option( 'trans_type' ) ) {
			return false;
		}
		
		return true;
	}
	
	public function maybe_save_customers() {
		return 'yes' == $this->get_gateway()->get_option( 'save_customers' );
	}
	
	/**
	 * Get order items description
	 *
	 * @param \WC_Order $order
	 * @param bool      $is_subscription
	 *
	 * @return string
	 */
	public function get_order_description( \WC_Order $order, $is_subscription = false ) {
		$desc = '';
		if ( count( $order->get_items() ) > 0 ) {
			foreach ( $order->get_items() as $item ) {
				if ( \WC_Paytrace_Compat::get_item_quantity( $item ) ) {
					$item_meta = \WC_Paytrace_Compat::wc_display_item_meta( $item );
					
					$item_name = \WC_Paytrace_Compat::get_item_name( $item );
					if ( $item_meta ) {
						$item_name .= ' (' . $item_meta . ')';
					}
					
					$desc .= \WC_Paytrace_Compat::get_item_quantity( $item ) . ' x ' . $item_name . ', ';
				}
			}
			
			$validator = $this->get_api_services()->get_validator();
			
			// Remove the last two chars and decode any html chars
			$desc = $validator->html_entity_decode_numeric( substr( $desc, 0, - 2 ) );
			$desc = $validator->remove_restricted_characters( $desc );
		}
		
		return $desc;
	}
	
	/**
	 * Returns true if we have to set the eCheck payment order to on-hold status
	 *
	 * @return bool
	 */
	public function is_ach_payment_on_hold() {
		return 'yes' == $this->get_gateway()->get_option( 'ach_payment_onhold' );
	}
	
	/**
	 * Returns if we want to send the order descriptions with the requests
	 *
	 * @return bool
	 */
	public function maybe_send_order_description() {
		return 'yes' == $this->get_gateway()->get_option( 'send_order_description' );
	}
}