/* global checkout_params_paytrace */
(function ($) {
	$(document).ready(function () {
		paytraceForm.onInit();
	});

	$(document.body).on('updated_checkout', function () {
		paytraceForm.onUpdatedCheckout();
	});

	$(document.body).on('update_checkout', function () {
		paytraceForm.onUpdateCheckout();
	});

	var paytraceForm = {
		gatewayId            : checkout_params_paytrace.gatewayId,
		selectedPaymentMethod: checkout_params_paytrace.defaultMethod,
		publicKey            : null,

		/**
		 * Loads the form
		 */
		onInit: function () {
			// Init all variables that we want to use throughout the object
			paytraceForm.formBindField = paytraceForm.getBindElement();
			paytraceForm.wcForm = $(paytraceForm.formBindField);

			paytraceForm.onSavedPaymentMethodInputsChange();
			paytraceForm.onPaymentTypeChoiceInputChange();
			paytraceForm.onCreateAccountCheckboxChange();
			paytraceForm.mimicCreateAccountToSaveToAccountCheckbox();

			// Hide the new payment card/check inputs
			paytraceForm.hideEmptySavedMethodsInput('card');
			paytraceForm.hideEmptySavedMethodsInput('check');

			if (!checkout_params_paytrace.isWc3_0 || paytraceForm.isChangePaymentMethodPage()) {
				paytraceForm.triggerPaymentTypeInputChange();
				paytraceForm.triggerSavedMethodInputChange();
			}

			if (paytraceForm.isSeparatedForms()) {
				paytraceForm.hideMainMethod();
				paytraceForm.onPaymentMethodChange();
			}

			paytraceForm.loadSubmitForm();

			$(document.body)
				.on('paytraceErrored', paytraceForm.onError)
				.on('checkout_error', paytraceForm.resetErrors);

			$('form.woocommerce-checkout').on('change', paytraceForm.resetErrors);
			paytraceForm.wcForm.on('change keyup', paytraceForm.resetErrors);

			if (paytraceForm.isUsingProtect()) {
				PaytraceProtect.setup();
			}
		},

		/**
		 * Binds the payment method [card,check] change to select and display the correct form
		 */
		onPaymentMethodChange: function () {
			paytraceForm.wcForm.on('click', 'input[name="payment_method"]', function () {
				var clicked = $(this);
				var payBox = paytraceForm.wcForm.find('input[name="' + paytraceForm.gatewayId + '_type_choice"]:checked');

				if (paytraceForm.gatewayId !== clicked.val()) {
					payBox.prop('checked', false);
				}
			});

			paytraceForm.activateDefaultPaymentSelection();
		},

		/**
		 * Name of the page we are on
		 * @since 2.0.0
		 * @version 2.4.0 - use the checkout_params_paytrace properties
		 * @returns {*}
		 */
		getFormPageName: function () {
			// Add Payment Method
			if (checkout_params_paytrace.isAddPaymentMethodPage) {
				return 'addPaymentMethod';
			}

			// Pay for order page
			if (checkout_params_paytrace.isPayForOrderPage) {
				return 'pay'
			}

			// Checkout
			if (checkout_params_paytrace.isCheckoutPage) {
				return 'checkout'
			}
		},

		/**
		 * Loads the paytrace public key
		 * @returns {*}
		 */
		loadPaytraceKey: function () {
			// Set the key from an AJAX call (in this case via a relative URL)
			var localKey = checkout_params_paytrace.publicKey;

			if ('' !== localKey) {
				paytraceForm.setKey(localKey);
				return;
			}

			var location = checkout_params_paytrace.publicKeyUrl;
			if ('' === location) {
				location = '/public_key.pem'
			}

			paytraceForm.setKeyAjax(location);
		},

		/**
		 * Is Paytrace the selected payment option
		 * @returns {*}
		 */
		isPaytraceChecked: function () {
			return $('#payment_method_paytrace').is(':checked');
		},

		/**
		 * Resets the Paytrace displayed errors
		 */
		resetErrors: function () {
			$('.wc-paytrace-error').remove();
		},
		/**
		 * Displays the paytrace errors
		 * @param e
		 * @param result
		 */
		onError: function (e, result) {
			var message = result.message;
			var type = 'card' === result.form || 'check';
			paytraceForm.resetErrors();

			var appendTo = result.appendTo || $('#paytrace-cards-form');
			if (false === result.appendTo && 'check' === type) {
				appendTo = $('#paytrace-checks-form');
			}

			if (1 === appendTo.length) {
				appendTo.after('<ul class="woocommerce_error woocommerce-error wc-paytrace-error"><li>' + message + '</li></ul>');
			} else {
				$.each(appendTo, function (index, value) {
					if (0 === index) {
						$(value).after('<ul class="woocommerce_error woocommerce-error wc-paytrace-error"><li>' + message + '</li></ul>')
					}
				});
			}

			var errorObj = $('.wc-paytrace-error');
			if (errorObj.length) {
				$('html, body').animate({
					scrollTop: (errorObj.offset().top - 200),
				}, 200);
			}
		},

		/**
		 * Loads the on submit method and actions
		 */
		loadSubmitForm: function () {
			try {
				var checkoutForm = $('form.checkout');

				if (paytraceForm.isUsingProtect()) {
					PaytraceProtect.loadSubmitForm();
					return;
				}

				if (paytraceForm.isUsingEncryption()) {
					// Load the public key
					paytraceForm.loadPaytraceKey();
				}

				// On checkout page
				if (0 < checkoutForm.length) {
					checkoutForm.on('checkout_place_order_paytrace', paytraceForm.onSubmitForm);
				} else {
					// Take all other forms from the bind element

					// Are we on add method page?
					if (paytraceForm.isAddPaymentMethodPage()) {
						// Remove the submit action, which adds an overlay over the form
						paytraceForm.wcForm.off('submit');
					}

					paytraceForm.wcForm.on('submit', function (e) {
						var elToBlock = $('#add_payment_method');
						paytraceForm.block(elToBlock);

						if (isAddMethodForm) {
							paytraceForm.block(elToBlock);
							if (paytraceForm.onSubmitForm(e)) {
								return true;
							}

							paytraceForm.unblock(elToBlock);
						}

						return paytraceForm.onSubmitForm(e);
					});
				}
			} catch (error) {
				// Error occurred, so we want to display it and stop the form from submitting
				$(document.body).trigger('paytraceErrored', error);
				return false;
			}
		},

		/**
		 * Loads the card fields formatting.
		 */
		formatCardFields: function () {
			var number = paytraceForm.wcForm.find('#paytrace-card-number');
			var cvc = paytraceForm.wcForm.find('#paytrace-card-cvc');
			var expiry = paytraceForm.wcForm.find('#paytrace-card-expiry');

			number.payment('formatCardNumber');
			cvc.payment('formatCardCVC');
			expiry.payment('formatCardExpiry');
		},

		/**
		 * Performs all action we need to take on form submit
		 * 1. Validate form
		 * 2. Encrypt fields
		 * @returns {boolean}
		 */
		onSubmitForm: function (e) {
			try {
				if (!paytraceForm.isPaytraceChecked()) {
					return true;
				}

				var form = $(e.target);

				$(document.body).triggerHandler('paytrace_on_submit_event', [form, paytraceForm]);

				// Validate the payment form
				if (paytraceForm.isPayingWithCard()) {
					paytraceForm.validateCardFields(form);
				} else {
					paytraceForm.validateCheckFields(form);
				}

				if (paytraceForm.isUsingEncryption()) {
					// Encrypt the fields only if needed
					if (paytraceForm.isPayingWithCard() && paytraceForm.isPayingWithNewCard()) {
						var ccNumberEl = form.find('input#paytrace-card-number');
						var ccNumberEncryptedEl = form.find('input#paytrace-card-number-encrypted');
						ccNumberEncryptedEl.val(paytraceForm.encryptValue(ccNumberEl.val()));
					}

					// New or Saved, but cvc required
					if ((paytraceForm.isPayingWithNewCard() && paytraceForm.isCvcRequired())
						|| (!paytraceForm.isPayingWithNewCard() && paytraceForm.isCvcRequiredWithSavedCards())) {
						var cvcEl = form.find('input#paytrace-card-cvc');
						var cvcEncryptedEl = form.find('input#paytrace-card-cvc-encrypted');
						cvcEncryptedEl.val(paytraceForm.encryptValue(cvcEl.val()));
					}
				}

				return true;
			} catch (error) {
				// Error occurred, so we want to display it and stop the form from submitting
				$(document.body).trigger('paytraceErrored', error);
				return false;
			}
		},

		/**
		 * Validates the card fields and throws a generic error for each
		 * @param form
		 */
		validateCardFields: function (form) {

			// Allow for 3rd party to validate, too. Just throw your errors
			$(document.body).triggerHandler('paytrace_validate_card_fields', [form, paytraceForm]);

			// Validate all fields
			if (paytraceForm.isPayingWithNewCard()) {
				paytraceForm.validateCardNumber(form);
				paytraceForm.validateCardExpiry(form);

				if (paytraceForm.isCvcRequired()) {
					paytraceForm.validateCardCvc(form);
				}
			} else if (paytraceForm.isCvcRequiredWithSavedCards()) {
				paytraceForm.validateCardCvc(form);
			}
		},

		handleProtectErrors: function (errors) {
			var message = '';
			var processedNumbers = [];

			var onlyFourHundredErrors = true;
			$.each(errors, function (key, value) {
				if ('35' == value['responseCode']
					|| '43' == value['responseCode']
					|| '44' == value['responseCode']
					|| '148' == value['responseCode']
				) {
					onlyFourHundredErrors = false;
					return false;
				}
			});

			$.each(errors, function (key, value) {

				if (0 > $.inArray('35', processedNumbers) && '35' == value['responseCode']) {
					PTPayment.style({'cc': {'border_color': 'red'}});

					message += checkout_params_paytrace.il8n['cardNumberNotValid'] + '. ';
				}

				if (
					(0 > $.inArray('43', processedNumbers) && 0 > $.inArray('44', processedNumbers))
					&& ('43' == value['responseCode'] || '44' == value['responseCode'])
				) {
					PTPayment.style({'exp': {'border_color': 'red'}});
					message += checkout_params_paytrace.il8n['cardExpiryNotValid'] + '. ';
				}

				if (0 > $.inArray('148', processedNumbers) && '148' == value['responseCode']) {
					PTPayment.style({'code': {'border_color': 'red'}});

					message += checkout_params_paytrace.il8n['cardCvcNotValid'] + '. ';
				}

				// Only show 400 errors if it is the only error type
				if (true === onlyFourHundredErrors && 0 > $.inArray('400', processedNumbers) && '400' == value['responseCode']) {
					message += value['description'] + '. ';
				}

				processedNumbers.push(value['responseCode']);
			});

			throw new paytraceError(message, 'card', paytraceForm.wcForm.find('#pt_hpf_form'));
		},

		/**
		 * Validates the Check form
		 * We mainly want the fields not empty
		 * @since 2.4.0
		 * @param form
		 */
		validateCheckFields: function (form) {

			// Allow for 3rd party to validate, too. Just throw your errors
			$(document.body).triggerHandler('paytrace_validate_check_fields', [form, paytraceForm]);

			// Validate all fields
			if (paytraceForm.isPayingWithNewCheck()) {
				paytraceForm.validateCheckRoutingNumber(form);
				paytraceForm.validateCheckAccountNumber(form);
			}
		},

		/**
		 * Validates the card number of the passed form
		 * @param form
		 */
		validateCardNumber: function (form) {
			if (!$.payment.validateCardNumber(form.find('input#paytrace-card-number').val())) {
				throw new paytraceError(checkout_params_paytrace.il8n['cardNumberNotValid'], 'card', form.find('input#paytrace-card-number'));
			}
		},

		/**
		 * Validates the card expiry of the passed form
		 * @param form
		 */
		validateCardExpiry: function (form) {
			if (!$.payment.validateCardExpiry(
				$.payment.cardExpiryVal(
					form.find('input#paytrace-card-expiry').val(),
				),
			)
			) {
				throw new paytraceError(checkout_params_paytrace.il8n['cardExpiryNotValid'], 'card', form.find('input#paytrace-card-expiry'));
			}
		},

		/**
		 * Validates the card cvc of the passed form
		 * @param form
		 */
		validateCardCvc: function (form) {
			// Pass the type if paying with a new card
			var type = paytraceForm.isPayingWithNewCard() ? $.payment.cardType(form.find('input#paytrace-card-number').val()) : null;

			if (!$.payment.validateCardCVC(form.find('input#paytrace-card-cvc').val(), type)) {
				throw new paytraceError(checkout_params_paytrace.il8n['cardCvcNotValid'], 'card', form.find('input#paytrace-card-cvc'));
			}
		},

		/**
		 * Validates the check account number
		 * @since 2.4.0
		 * @param form
		 */
		validateCheckAccountNumber: function (form) {
			if ('' === form.find('input#paytrace-echeck-account-number').val()) {
				throw new paytraceError(checkout_params_paytrace.il8n['checkAccountNumberEmpty'], 'check', form.find('input#paytrace-echeck-account-number'));
			}
		},
		/**
		 * Validates check routing number
		 * @since 2.4.0
		 * @param form
		 */
		validateCheckRoutingNumber: function (form) {
			if ('' === form.find('input#paytrace-echeck-routing-number').val()) {
				throw new paytraceError(checkout_params_paytrace.il8n['checkRoutingNumberEmpty'], 'check', form.find('input#paytrace-echeck-routing-number'));
			}
		},

		/**
		 * Actions performed when the checkout is updated
		 */
		onUpdatedCheckout: function () {
			paytraceForm.mimicCreateAccountToSaveToAccountCheckbox();
			paytraceForm.hideEmptySavedMethodsInput('card');
			paytraceForm.hideEmptySavedMethodsInput('check');

			paytraceForm.onCreateAccountCheckboxChange();

			if (paytraceForm.isSeparatedForms()) {
				paytraceForm.activateDefaultPaymentSelection();
			}

			if (!checkout_params_paytrace.isWc3_0) {
				paytraceForm.triggerPaymentTypeInputChange();
				paytraceForm.triggerSavedMethodInputChange();
			}

			if (paytraceForm.isUsingProtect()) {
				PaytraceProtect.setup();
			}
		},

		onUpdateCheckout: function () {
			if (paytraceForm.isUndefined(paytraceForm.wcForm)) {
				return;
			}

			var selected = paytraceForm.wcForm.find('input[name="paytrace_type_choice"]:checked');
			if ('' !== selected.val()) {
				paytraceForm.selectedPaymentMethod = selected.val();
			}
		},

		/**
		 * @since
		 * @returns {string}
		 */
		getBindElement: function () {
			var el = 'form.woocommerce-checkout';
			if (paytraceForm.isChangePaymentMethodPage()
				|| paytraceForm.isPayForOrderPage()) {
				el = '#order_review';
			} else if (paytraceForm.isAddPaymentMethodPage()) {
				el = '#add_payment_method';
			}

			return el;
		},

		maybeShowSaveToAccountCheckbox: function () {
			var account_el = $('input#createaccount');

			paytraceForm.showSaveToAccountCheckbox();

			if (0 < account_el.length) {
				paytraceForm.mimicCreateAccountToSaveToAccountCheckbox();
			}
		},

		/**
		 * Mimics the create account checkbox and if the customer is a guest,
		 * we will not show the save to account option
		 */
		mimicCreateAccountToSaveToAccountCheckbox: function () {
			var account = $('input#createaccount');

			if (0 >= account.length) {
				return;
			}

			if (account.is(':checked')) {
				paytraceForm.showSaveToAccountCheckbox();
			} else {
				paytraceForm.hideSaveToAccountCheckbox();
			}
		},

		/**
		 * Binds the create account checkbox change.
		 * @since 2.4.0
		 */
		onCreateAccountCheckboxChange: function () {
			$('form.woocommerce-checkout').on('change', 'input#createaccount', paytraceForm.mimicCreateAccountToSaveToAccountCheckbox);
		},

		/**
		 * Triggers change on the new
		 * @param type 'card' or 'check'
		 */
		triggerSavedMethodInputChange: function (type) {
			var savedInputEls = $('.woocommerce-paytrace-SavedPaymentMethods-tokenInput');

			savedInputEls.each(function () {
				var selectedEl = $(this);
				if (selectedEl.is(':checked')) {
					selectedEl.change();
					return false;
				}
			});
		},

		/**
		 * Triggers a change to the payment type element thus allowing the visualization of the check or card forms
		 * Triggers the action to also show|hide all elements on the individual forms as well
		 */
		triggerPaymentTypeInputChange: function () {
			var type = paytraceForm.isPayingWithCard() ? 'card' : 'check';

			var paymentTypeElement = $('input#' + paytraceForm.gatewayId + '_' + type + '_choice');

			if (0 < paymentTypeElement.length) {
				paymentTypeElement.change();
			}
		},

		/**
		 * Binds the Saved methods selection with the actions needed to be performed on change
		 */
		onSavedPaymentMethodInputsChange: function () {
			// Bind all inputs
			paytraceForm.wcForm.on('change', '.woocommerce-paytrace-SavedPaymentMethods-tokenInput', function () {
				var payingWith = paytraceForm.isPayingWithCard() ? 'card' : 'check';
				var selected = $('input[name="wc-paytrace-payment-' + payingWith + '-token"]:checked').val();

				if ('new' === selected) {
					paytraceForm.showForm('card' === payingWith ? 'cards' : 'checks');
					paytraceForm.maybeShowSaveToAccountCheckbox();
				} else {
					paytraceForm.hideForm('card' === payingWith ? 'cards' : 'checks');
					paytraceForm.hideSaveToAccountCheckbox();
				}
			});
		},

		/**
		 * Binds the Payment Type (card|check) choice with the actions needed
		 */
		onPaymentTypeChoiceInputChange: function () {
			paytraceForm.wcForm.on('change', 'input[name="' + paytraceForm.gatewayId + '_type_choice"]', paytraceForm.processPaymentTypeChoiceChange);
		},

		processPaymentTypeChoiceChange: function () {
			var selected = $('input[name="' + paytraceForm.gatewayId + '_type_choice"]:checked').val();

			if (paytraceForm.isSeparatedForms()) {
				paytraceForm.activatePaymentMethod();
				paytraceForm.hideNoneGatewayPaymentBoxes();
			}

			paytraceForm.hideForm('card' === selected ? 'checks' : 'cards');
			paytraceForm.hideSavedMethods('card' === selected ? 'check' : 'card');

			paytraceForm.selectedPaymentMethod = selected;

			if (paytraceForm.isSeparatedForms()) {
				paytraceForm.hidePaymentBox('card' === selected ? 'check' : 'card');
				paytraceForm.showPaymentBox(selected);
			}

			paytraceForm.showForm('card' === selected ? 'cards' : 'checks');
			paytraceForm.showSavedMethods('card' === selected ? 'card' : 'check');

			paytraceForm.hideEmptySavedMethodsInput(selected);

			if ('card' === selected && paytraceForm.isPayingWithNewCard()
				|| 'check' === selected && paytraceForm.isPayingWithNewCheck()) {
				paytraceForm.maybeShowSaveToAccountCheckbox();
			} else {
				paytraceForm.hideSaveToAccountCheckbox();
			}
		},

		/**
		 * Hides the "New" radio button, if there are no saved payment methods
		 * @param type
		 */
		hideEmptySavedMethodsInput: function (type) {
			type = type || 'card';

			// Hide the methods field, if there are no methods
			var saved_methods = $('.woocommerce-paytrace-SavedPaymentMethods-' + type);
			if (0 === saved_methods.data('count')) {
				saved_methods.hide();
			}
		},

		/**
		 * Shows the form
		 * @param showType 'cards' or 'checks'
		 */
		showForm: function (showType) {
			showType = showType || 'cards';
			var form = $('fieldset#' + paytraceForm.gatewayId + '-' + showType + '-form');

			if ('cards' === showType) {
				// If we are paying with a new card we need the whole form shown
				if (paytraceForm.isPayingWithNewCard()) {
					if (form.hasClass('pt-hidden')) {
						// Show the fieldset and remove the hidden class
						form.slideDown(200).removeClass('pt-hidden');
					}

					if (! paytraceForm.isUsingProtect()) {
						// Show the individual rows
						form.find('.form-row').slideDown(200);
						paytraceForm.makeCvcLastField(form);
					}
				} else {
					// Paying with a saved card we only need to check and show the CVC
					if (! paytraceForm.isUsingProtect() && paytraceForm.isCvcRequiredWithSavedCards()) {
						/**
						 * 1. Show the form
						 * 2. Hide all fields
						 * 3. Show the CVC field
						 */
						if (form.hasClass('pt-hidden')) {
							// Show the form and remove the hidden class
							form.slideDown(200).removeClass('pt-hidden');
						}

						// Hide all fields, but the CVC one
						form.find('.form-row').not('.paytrace-cvc-wrapper').slideUp(200);

						paytraceForm.makeCvcWideField(form);
					} else {
						/**
						 * 1. Hide the form
						 */
						form.slideUp().addClass('pt-hidden');
					}
				}
			} else {
				// Show the checks form only if the new check is selected
				if (paytraceForm.isPayingWithNewCheck()) {
					form.slideDown(200).removeClass('pt-hidden');
				}
			}
		},

		/**
		 * Hides the form
		 * @param hideType
		 */
		hideForm: function (hideType) {
			hideType = hideType || 'cards';
			var form = $('fieldset#' + paytraceForm.gatewayId + '-' + hideType + '-form');

			if ('cards' === hideType) {
				// Paying with a card, but need to hide the form
				if (paytraceForm.isPayingWithCard()) {
					// Need to show the CVC
					if (! paytraceForm.isUsingProtect() && paytraceForm.isCvcRequiredWithSavedCards()) {
						form.find('.form-row').not('.paytrace-cvc-wrapper').slideUp(200);
						paytraceForm.makeCvcWideField(form)
					} else {
						form.slideUp().addClass('pt-hidden');
					}
				} else {

					form.slideUp().addClass('pt-hidden');
				}
			} else {
				// Hiding checks form
				form.slideUp().addClass('pt-hidden');
			}
		},

		/**
		 * Shows the form
		 * @param type 'cards' or 'checks'
		 */
		showPaymentBox: function (type) {
			type = type || 'card';
			var form = $('.payment_box_' + paytraceForm.gatewayId + '_' + type);

			if ('card' === type) {
				// Paying with a saved card we only need to check and show the CVC
				// Show the fieldset and remove the hidden class
				form.slideDown().removeClass('pt-hidden');
			} else {
				// Show the checks form only if the new check is selected
				form.slideDown().removeClass('pt-hidden');
			}
		},

		hidePaymentBox: function (type) {
			type = type || 'card';
			var form = $('.payment_box_' + paytraceForm.gatewayId + '_' + type);

			if ('card' === type) {
				// Paying with a card, but need to hide the form
				form.slideUp().addClass('pt-hidden');
			} else {
				// Hiding checks form
				form.slideUp().addClass('pt-hidden');
			}
		},

		makeCvcLastField: function (form) {
			// Show the CVC field only and reset its value
			var cvc = form.find('.paytrace-cvc-wrapper');
			if (cvc.hasClass('form-row-wide')) {
				cvc.removeClass('form-row-wide').addClass('form-row-last');

				paytraceForm.resetCvcField(cvc);
			}
		},

		makeCvcWideField: function (form) {
			var cvc = form.find('.paytrace-cvc-wrapper');
			if (cvc.hasClass('form-row-last')) {
				cvc.removeClass('form-row-last').addClass('form-row-wide');

				paytraceForm.resetCvcField(cvc);
			}
		},

		resetCvcField: function (cvc) {
			// Remove the value from the cvc on every form change
			cvc.find('input#paytrace-card-cvc').val('');
		},

		/**
		 * Shows the saved methods in the type form
		 * @param type 'card' or 'check'
		 */
		showSavedMethods: function (type) {
			type = type || 'card';
			$('.woocommerce-' + paytraceForm.gatewayId + '-SavedPaymentMethods-wrapper.' + type).slideDown();
		},

		hideSavedMethods: function (type) {
			type = type || 'card';
			$('.woocommerce-' + paytraceForm.gatewayId + '-SavedPaymentMethods-wrapper.' + type).hide();
		},

		showSaveToAccountCheckbox: function () {
			var el = $('.' + paytraceForm.gatewayId + '-create-account');

			if (0 === el.length) {
				return;
			}

			// Hide them all first
			paytraceForm.hideSaveToAccountCheckbox();

			// Show only the type we need
			if (true === paytraceForm.isPayingWithCard()) {
				el.each(function (index, element) {
					if ($(element).hasClass('card')) {
						$(element).show();
					}
				});
			} else {
				el.each(function (index, element) {
					if ($(element).hasClass('check')) {
						$(element).show();
					}
				});
			}
		},

		hideSaveToAccountCheckbox: function () {
			var el = $('.' + paytraceForm.gatewayId + '-create-account');

			if (0 === el.length) {
				return;
			}

			el.hide();
		},

		/**
		 * Moved this selector name to a method
		 * because it is quite a composition and don't want to copy it in places
		 * @param type 'card' or 'check'
		 * @returns {string}
		 */
		getSavedPaymentMethodsInputSelectorName: function (type) {
			type = type || 'card';
			return '.woocommerce-' + paytraceForm.gatewayId + '-SavedPaymentMethods-' + type + ' .woocommerce-' + paytraceForm.gatewayId + '-SavedPaymentMethods-tokenInput';
		},

		block: function (el) {
			if (!paytraceForm.isMobile()) {
				el.block({
					message   : null,
					overlayCSS: {
						background: '#fff',
						opacity   : 0.6,
					},
				});
			}
		},

		unblock: function (el) {
			el.unblock();
		},

		isMobile: function () {
			return /Android|webOS|iPhone|iPad|iPod|BlackBerry|IEMobile|Opera Mini/i.test(navigator.userAgent);
		},

		/**
		 * Returns whether the customer is paying with a card
		 *
		 * @returns {boolean}
		 */
		isPayingWithCard: function () {
			var cardOrCheckField = $('input[name="' + paytraceForm.gatewayId + '_type_choice"]:checked');
			return 0 >= cardOrCheckField.length || 'card' === cardOrCheckField.val();
		},

		/**
		 * Returns whether the customer is paying with a new card
		 *
		 * @returns {boolean|*}
		 */
		isPayingWithNewCard: function () {
			var newCardField = $('#wc-paytrace-payment-token-card-new');
			return 0 < newCardField.length && newCardField.is(':checked');
		},

		/**
		 * @returns {boolean|*}
		 */
		isPayingWithNewCheck: function () {
			var newCheckField = $('#wc-paytrace-payment-token-check-new');
			return 0 < newCheckField.length && newCheckField.is(':checked');
		},

		/**
		 * Returns whether we require the CVC number to be filled in
		 *
		 * Trigger "paytrace_require_cvc_with_tokens" allow 3rd party script to manipulate the option
		 *
		 * @returns {boolean}
		 */
		isCvcRequired: function () {
			var required_settings = !paytraceForm.isUndefined(checkout_params_paytrace.isCvcRequiredField) && '1' === checkout_params_paytrace.isCvcRequiredField;
			var form = $('fieldset#' + paytraceForm.gatewayId + '-cards-form');
			var cvc = form.find('input#paytrace-card-cvc').val();

			return (required_settings || '' !== cvc) && false !== $(document.body).triggerHandler('paytrace_require_cvc_with_tokens', paytraceForm);
		},

		/**
		 * Returns the requireCvcWithSavedCards setting value
		 * @since 2.4.1
		 * @returns {boolean}
		 */
		isCvcRequiredWithSavedCards: function () {
			return !paytraceForm.isUndefined(checkout_params_paytrace.requireCvcWithSavedCards) && '1' === checkout_params_paytrace.requireCvcWithSavedCards
		},

		/**
		 * Sets the public key inside the forge
		 * @since 2.2.1
		 * @param publicKeyPem
		 * @returns {paytraceForm}
		 */
		setKey: function (publicKeyPem) {
			paytraceForm.publicKey = forge.pki.publicKeyFromPem(publicKeyPem);
		},

		/**
		 * Retrieve the public key from the provided URL and set in in the object
		 * @since 2.2.1
		 * @param keyUrl
		 */
		setKeyAjax: function (keyUrl) {
			$.ajax({
				url     : keyUrl,
				dataType: 'text',
			})
				.done(function (publicKeyPem) {
					paytraceForm.setKey(publicKeyPem);

					return true;
				})
				.fail(function (jqXHR, textStatus) {
				});
		},

		/**
		 * Encrypt the provided value
		 * @since 2.2.1
		 * @param data
		 */
		encryptValue: function (data) {
			if (!paytraceForm.publicKey) {
				throw new paytraceError(checkout_params_paytrace.il8n['publicKeyNotLoaded'], 'card', false);
			}
			return forge.util.encode64(paytraceForm.publicKey.encrypt(data));
		},

		isUsingEncryption: function () {
			return !paytraceForm.isUndefined(checkout_params_paytrace.securityType) && 'encryption' === paytraceForm.securityType();
		},

		isUsingProtect: function () {
			return !paytraceForm.isUndefined(checkout_params_paytrace.securityType) && 'protect' === paytraceForm.securityType();
		},

		/**
		 * @since 2.4.0
		 */
		activateDefaultPaymentSelection: function () {
			var is_checked = paytraceForm.isPaytraceChecked();
			var payBox = paytraceForm.wcForm.find('input[name="' + paytraceForm.gatewayId + '_type_choice"]');

			if (is_checked) {
				// If there was a previously selected method, check that one.
				if ('' !== paytraceForm.selectedPaymentMethod) {
					var method = $('#payment_method_' + paytraceForm.selectedPaymentMethod + '_' + paytraceForm.gatewayId);
					method.prop('checked', true);
					paytraceForm.processPaymentTypeChoiceChange();
				} else {
					// Activate the default selection within the gateway
					var checkedType = $('input#payment_method_card_' + paytraceForm.gatewayId);

					if (0 < checkedType.length) {
						checkedType.prop('checked', true);
						paytraceForm.processPaymentTypeChoiceChange();
					} else {
						payBox.first().prop('checked', true);
						paytraceForm.processPaymentTypeChoiceChange();
					}
				}
			} else {
				// Hide the gateway forms when another gateway is selected
				paytraceForm.hideGatewayPaymentBoxes();
			}

		},

		/**
		 * Makes the payment option selected.
		 * Used when the payment forms are separate methods
		 * @since 2.4.0
		 */
		activatePaymentMethod: function () {
			var payMethod = paytraceForm.wcForm.find('#payment_method_' + paytraceForm.gatewayId);

			if (payMethod.is(':checked')) {
				return;
			} else if ($(document.body).triggerHandler('paytrace_activate_payment_method_with_click', [false, paytraceForm])) {
				payMethod.trigger('click');
			} else {
				payMethod.prop("checked", true);
				payMethod.trigger('click');
				payMethod.trigger('change');
			}
		},

		/**
		 * Hides all visible payment boxes
		 * @since 2.4.0
		 */
		hideAllVisiblePaymentBoxes: function () {
			var paymentBox = $('div.payment_box');
			paymentBox.filter(':visible').hide();
		},

		/**
		 * Hides the gateway payment boxes
		 * @since 2.4.0
		 */
		hideGatewayPaymentBoxes: function () {
			var paymentBox = $('div.payment_box');
			paymentBox.filter(':visible').filter('.payment_box_' + paytraceForm.gatewayId).hide();
		},

		/**
		 * Hides all payment boxes not of our gateway
		 * @since 2.4.0
		 */
		hideNoneGatewayPaymentBoxes: function () {
			var paymentBox = $('div.payment_box');
			paymentBox.filter(':visible').not('.payment_box_' + paytraceForm.gatewayId).hide();
		},

		/**
		 * @since 2.4.0
		 * @returns {boolean}
		 */
		isSeparatedForms: function () {
			var separatedSections = $('.separated_section_' + paytraceForm.gatewayId);
			return 0 < separatedSections.length;
		},

		/**
		 * Hides the Main Gateway method because we are using other boxes for the forms
		 * @since 2.4.0
		 */
		hideMainMethod: function () {
			$('input#payment_method_' + paytraceForm.gatewayId).closest('li').hide();
		},

		isUndefined: function (e) {
			return 'undefined' === typeof (e);
		},

		/**
		 * @since 2.4.1
		 * @returns {boolean}
		 */
		isAddPaymentMethodPage: function () {
			return !paytraceForm.isUndefined(checkout_params_paytrace.isAddPaymentMethodPage) && '1' === checkout_params_paytrace.isAddPaymentMethodPage;
		},

		/**
		 * @since 2.4.1
		 * @returns {boolean}
		 */
		isPayForOrderPage: function () {
			return !paytraceForm.isUndefined(checkout_params_paytrace.isPayForOrderPage) && '1' === checkout_params_paytrace.isPayForOrderPage;
		},

		/**
		 * @since 2.4.1
		 * @returns {boolean}
		 */
		isCheckoutPage: function () {
			return !paytraceForm.isUndefined(checkout_params_paytrace.isCheckoutPage) && '1' === checkout_params_paytrace.isCheckoutPage;
		},

		/**
		 * @since 2.4.1
		 * @returns {boolean}
		 */
		isChangePaymentMethodPage: function () {
			return !paytraceForm.isUndefined(checkout_params_paytrace.isChangePaymentMethodPage) && '1' === checkout_params_paytrace.isChangePaymentMethodPage;
		},

		securityType: function () {
			return !paytraceForm.isUndefined(checkout_params_paytrace.securityType) ? checkout_params_paytrace.securityType : '';
		},
	};

	var PaytraceProtect = {
		processed: false,

		init: function () {

		},

		loadSubmitForm: function () {
			try {
				var checkoutForm = $('form.checkout');

				// On checkout page
				if (0 < checkoutForm.length) {
					checkoutForm.on('checkout_place_order_paytrace', PaytraceProtect.onSubmit);
				} else {
					// Take all other forms from the bind element

					// Are we on add method page?
					if (paytraceForm.isAddPaymentMethodPage() || paytraceForm.isPayForOrderPage()) {
						// Remove the submit action, which adds an overlay over the form
						paytraceForm.wcForm.off('submit');
					}

					paytraceForm.wcForm.on('submit', function (e) {
						paytraceForm.block(paytraceForm.wcForm);

						return true === PaytraceProtect.onSubmit(e);
					});
				}
			} catch (error) {
				// Error occurred, so we want to display it and stop the form from submitting
				$(document.body).trigger('paytraceErrored', error);
				return false;
			}
		},

		onSubmit: function (e) {
			try {
				if (!paytraceForm.isPaytraceChecked()) {
					return true;
				}

				if (!paytraceForm.isPayingWithNewCard()) {
					// TODO: Do we need to validate a CVC number?
					return true;
				}

				var form = $(e.target);

				$(document.body).triggerHandler('paytrace_on_submit_event', [form, paytraceForm]);

				// Validate the payment form
				if (!paytraceForm.isPayingWithCard()) {
					paytraceForm.validateCheckFields(form);
					return true;
				}

				if (true === PaytraceProtect.processed) {
					return true;
				}

				PaytraceProtect.processProtectPayment();

				return false;
			} catch (error) {
				// Error occurred, so we want to display it and stop the form from submitting
				$(document.body).trigger('paytraceErrored', error);
				return false;
			}
		},

		processProtectPayment: function () {
			if ('undefined' === typeof PTPayment) {
				return;
			}

			// The tokenization process also validates the sensitive data payment fields
			PTPayment.process()
				.then(function (result) {
					if (true === result.success) {
						var token = paytraceForm.wcForm.find('input[name="' + paytraceForm.gatewayId + '_hpf_token"');
						var enc_key = paytraceForm.wcForm.find('input[name="' + paytraceForm.gatewayId + '_enc_key"');
						token.val(result.message.hpf_token);
						enc_key.val(result.message.enc_key);
					}

					PaytraceProtect.processed = true;

					paytraceForm.wcForm.trigger('submit');

					return true;
				})
				.catch(function (errors) {
					try {
						PaytraceProtect.processed = false;

						if (0 < errors.reason.length) {
							paytraceForm.handleProtectErrors(errors.reason);
						}
					} catch (err) {
						// Error occurred, so we want to display it and stop the form from submitting
						$(document.body).trigger('paytraceErrored', err);
						return false;
					}
				});
		},

		setup: function () {
			if ('undefined' === typeof PTPayment) {
				return;
			}

			if ('' === PaytraceProtect.clientKey()) {
				$(document.body).trigger('paytraceErrored', new paytraceError(checkout_params_paytrace.il8n['protectInvalidConfiguration'], 'card', $('.woocommerce-notices-wrapper')));
				return;
			}

			PTPayment.setup({
				styles       :
					{
						'code': checkout_params_paytrace.template.code,
						'cc'  : checkout_params_paytrace.template.cc,
						'exp' : checkout_params_paytrace.template.exp,
					},
				authorization: {clientKey: PaytraceProtect.clientKey()},
			}).then(function (instance) {
				PTPayment.getControl("creditCard").label.text("Card Number");

				$(document.body).triggerHandler('paytrace_protect_loaded', [paytraceForm.wcForm, paytraceForm, PaytraceProtect]);
			});

			PaytraceProtect.processed = false;
		},

		clientKey: function () {
			return !paytraceForm.isUndefined(checkout_params_paytrace.clientKey) ? checkout_params_paytrace.clientKey : '';
		},
	};

	/**
	 * Simple function to handle the errors
	 * @since 2.4.0
	 * @param {String} message Error message
	 * @param {String} form (optional)The form name we want to attach the error to. Default: card
	 * @param {Object} element (optional)The element we want to attach the error to. Default: false
	 */
	var paytraceError = function (message, form, element) {
		this.message = message;
		this.form = form || 'card';
		this.appendTo = element || false;
	}

})(jQuery);