<?php

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Handles script loading
 *
 * @since  1.4.9
 * @author VanboDevelops
 *
 *        Copyright: (c) 2017 VanboDevelops
 *        License: GNU General Public License v3.0
 *        License URI: http://www.gnu.org/licenses/gpl-3.0.html
 */
class WC_Paytrace_Scripts {
	
	public $suffix;
	public $version;
	
	/**
	 * WC_Paytrace_Scripts constructor.
	 * @since 2.4.0
	 */
	public function __construct() {
		$this->suffix  = defined( 'SCRIPT_DEBUG' ) && SCRIPT_DEBUG ? '' : '.min';
		$this->version = $this->suffix ? WC_Paytrace::VERSION : rand( 1, 999 );
	}
	
	public function hooks() {
		add_action( 'wp_enqueue_scripts', array( $this, 'frontend_scripts' ) );
		add_action( 'admin_enqueue_scripts', array( $this, 'admin_scripts' ) );
	}
	
	/**
	 * Registers and/or enqueues scripts
	 *
	 * @since 2.0
	 */
	public function frontend_scripts() {
		/**
		 * @var \WC_Payment_Gateway|\WC_Paytrace_Gateway_Addons $gateway
		 */
		$gateway = \WcPaytrace\Helpers\Factories::get_gateway( 'paytrace' );
		
		if ( $gateway && ( is_checkout() || is_add_payment_method_page() ) ) {
			
			$require_cvc = 'yes' == $gateway->get_option( 'require_cvc_with_saved_cards' );
			
			wp_enqueue_style( 'paytrace-styles', WC_Paytrace::plugin_url() . '/assets/css/paytrace-styles' . $this->suffix . '.css', array(), $this->version );
			
			if ( 'encryption' === $gateway->get_security_type() ) {
				wp_register_script( 'paytrace-e2ee', WC_Paytrace::plugin_url() . '/assets/js/paytrace-e2ee.js', array(
					'paytrace-js',
				), WC_Paytrace::VERSION, true );
			} elseif ( 'protect' === $gateway->get_security_type() ) {
				wp_register_script( 'paytrace-protect', WC_Paytrace::plugin_url() . '/assets/js/paytrace-protect.js', array(
					'paytrace-js',
				), WC_Paytrace::VERSION, true );
			}
			
			wp_enqueue_script( 'paytrace-js', WC_Paytrace::plugin_url() . '/assets/js/paytrace-checkout' . $this->suffix . '.js', array(
				'jquery',
				'jquery-payment',
			), $this->version, true );
			
			$checkout_data = apply_filters( 'wc_paytrace_checkout_js_variables', array(
				'isWc3_0'                   => WC_Paytrace_Compat::is_wc_3_0(),
				'gatewayId'                 => 'paytrace',
				'isAddPaymentMethodPage'    => is_add_payment_method_page(),
				'isPayForOrderPage'         => is_checkout_pay_page(),
				'isCheckoutPage'            => is_checkout(),
				'isChangePaymentMethodPage' => \WC_Paytrace_Helpers::is_change_method_page(),
				'requireCvcWithSavedCards'  => $require_cvc,
				'isCvcRequiredField'        => $gateway->is_cvc_required(),
				'defaultMethod'             => 'card',
				'il8n'                      => array(
					'cardNumberNotValid'          => __( 'Card Number is invalid', WC_Paytrace::TEXT_DOMAIN ),
					'cardExpiryNotValid'          => __( 'Expiry date is invalid', WC_Paytrace::TEXT_DOMAIN ),
					'cardCvcNotValid'             => __( 'Card Code is invalid', WC_Paytrace::TEXT_DOMAIN ),
					'cardFormNotValid'            => __( 'Payment information is invalid', WC_Paytrace::TEXT_DOMAIN ),
					'publicKeyLoadFailed'         => __( 'Failed to load the public key', WC_Paytrace::TEXT_DOMAIN ),
					'publicKeyNotLoaded'          => __( 'No encryption public key was provided', WC_Paytrace::TEXT_DOMAIN ),
					'checkAccountNumberEmpty'     => __( 'Account Number is empty', WC_Paytrace::TEXT_DOMAIN ),
					'checkRoutingNumberEmpty'     => __( 'Routing Number is empty', WC_Paytrace::TEXT_DOMAIN ),
					'protectInvalidConfiguration' => __( 'The Paytrace payment option is not configured correctly. Please refresh and try again or contact us for help.', WC_Paytrace::TEXT_DOMAIN ),
				),
				'publicKeyUrl'              => $gateway->get_option( 'json_public_key_location' ),
				'publicKey'                 => $gateway->get_option( 'json_public_key' ),
				'securityType'              => $gateway->get_security_type(),
				'clientKey'                 => '',
			) );
			
			if ( 'protect' === $gateway->get_security_type() ) {
				try {
					$auth                       = $gateway->get_integration_class()->get_api_services()->protect_authorization();
					$checkout_data['clientKey'] = $auth->get_client_key();
				}
				catch ( Exception $e ) {
					\WC_Paytrace::add_debug_log( ' Error: ' . print_r( $e->getMessage(), true ) );
				}
				
				$checkout_data['template'] = \WcPaytrace\Helpers\Protect_Template_Helpers::get_protect_template(
					$gateway->get_option( 'protect_form_theme', 'light_rounded' )
				);
				if ( isset( $checkout_data['template']['exp'] ) && ! isset( $checkout_data['template']['exp']['type'] ) ) {
					$checkout_data['template']['exp']['type'] = $gateway->get_option( 'protect_exp_type', 'dropdown' );
				}
			}
			
			wp_localize_script( 'paytrace-js', 'checkout_params_paytrace', $checkout_data );
		}
		
		if ( is_account_page() ) {
			wp_register_script( 'paytrace-my-account', WC_Paytrace::plugin_url() . '/assets/js/paytrace-my-account' . $this->suffix . '.js', array( 'jquery' ), $this->version, true );
			wp_enqueue_style( 'paytrace-my-account-css', WC_Paytrace::plugin_url() . '/assets/css/paytrace-my-account' . $this->suffix . '.css', array(), $this->version );
		}
	}
	
	/**
	 * Adds admin scripts
	 *
	 * @since 2.0
	 */
	public function admin_scripts() {
		$screen    = get_current_screen();
		$screen_id = $screen ? $screen->id : '';
		
		if ( is_admin() ) {
			wp_register_script( 'paytrace-admin', WC_Paytrace::plugin_url() . '/assets/js/paytrace-admin' . $this->suffix . '.js', array( 'jquery' ), $this->version, true );
		}
		
		if ( in_array( str_replace( 'edit-', '', $screen_id ), wc_get_order_types( 'order-meta-boxes' ) ) ) {
			wp_enqueue_script( 'paytrace-admin' );
		}
		
		wp_localize_script( 'paytrace-admin', 'wc_paytrace_admin_params', array(
			'i18n_capture_payment'     => _x( 'Are you sure you want to capture the payment?', 'capture payment', WC_Paytrace::TEXT_DOMAIN ),
			'i18n_password_incorrect'  => _x( 'Your password contains "~" character. This character is know to break the API requests. Please change your password.', 'settings notice', WC_Paytrace::TEXT_DOMAIN ),
			'i18n_integration_changed' => _x( 'Integration Type changed! Save the change before you continue.', 'settings notice', WC_Paytrace::TEXT_DOMAIN ),
			'ajax_url'                 => admin_url( 'admin-ajax.php' ),
			'capture_payment'          => wp_create_nonce( 'capture-payment' ),
			'error_wrap'               => '<div class="wc-paytrace-error error"><h4>%%message%%</h4></div>',
		) );
	}
}