<?php
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Credit Card Payment Gateway
 */
class WC_Paytrace_Payment_Form extends \WcPaytrace\Abstracts\Integration {
	
	protected $tokens = array();
	protected $saved_checks = array();
	protected $saved_cards = array();
	protected $support_checks;
	protected $show_save_customer = false;
	protected $allowed_token_type = array(
		'card',
		'check',
	);
	protected $payment_sections_to_display = array();
	protected $sections_data = array();
	
	/**
	 * WC_Paytrace_Payment_Form constructor.
	 *
	 * @param WC_Paytrace_Gateway $gateway
	 */
	public function __construct( \WC_Paytrace_Gateway $gateway ) {
		parent::__construct( $gateway );
		
		$this->support_checks = 'yes' == $this->get_gateway()->get_option( 'support_check' ) ? true : false;
		$this->load_customer_saved_tokens();
	}
	
	/**
	 * Outputs the payment fields for the plugin.
	 *
	 * @since 2.1
	 *
	 * @param $show_save_customer
	 */
	public function payment_fields( $show_save_customer ) {
		$this->set_show_save_customer( $show_save_customer );
		
		if ( is_add_payment_method_page() ) {
			// Empty out all tokens and remove the save to account checkbox
			// because we are only adding payment methods here
			$this->saved_cards  = array();
			$this->saved_checks = array();
			$this->set_show_save_customer( false );
		}
		
		if ( $this->get_gateway()->separated_forms && $this->support_checks ) {
			$this->output_separated_forms();
		} else {
			$this->output_together_forms();
		}
	}
	
	/**
	 * Prepares and outputs the card and check forms as separate sections
	 * @since 2.4.0
	 */
	public function output_separated_forms() {
		$this->prepare_sections_data();
		
		// Load the Checks form template
		wc_get_template(
			'checkout/separate/paytrace-separation-form.php',
			array(
				'gateway' => $this->get_gateway(),
				'form'    => $this,
			),
			'',
			\WC_Paytrace::plugin_path() . '/templates/'
		);
	}
	
	/**
	 * Prepares and outputs the card and check forms in one section
	 * @since 2.4.0
	 */
	public function output_together_forms() {
		$this->maybe_output_form_switcher();
		$this->maybe_output_checks_form();
		$this->output_cards_form();
	}
	
	/**
	 * Loads the payment tokens to the class props
	 *
	 * @since 2.1
	 *
	 * @return bool
	 */
	public function load_customer_saved_tokens() {
		
		// Guest has no tokens
		if ( ! is_user_logged_in() ) {
			return false;
		}
		
		// No tokens, if Vault is not enabled
		if ( ! $this->get_gateway()->save_customers ) {
			return false;
		}
		
		$customer_tokens = new \WC_Paytrace_Customer_Tokens( get_current_user_id() );
		$this->tokens    = $customer_tokens->get_tokens();
		if ( $this->tokens ) {
			/**
			 * @var \WC_Paytrace_Token $token
			 */
			foreach ( $this->tokens as $n => $token ) {
				if ( 'Paytrace_eCheck' == $token->get_type() ) {
					$this->saved_checks[ $n ] = $token;
				} else {
					$this->saved_cards[ $n ] = $token;
				}
			}
		}
		
		return true;
	}
	
	/**
	 * Outputs the form switcher (Card or Check)
	 *
	 * @since 2.1
	 */
	public function maybe_output_form_switcher() {
		// Load the Cards/Check forms switch
		if ( $this->support_checks ) {
			// Load the Checks form template
			wc_get_template(
				'checkout/paytrace-switch-forms.php',
				array(
					'gateway' => $this->get_gateway()->id,
					'form'    => $this,
				),
				'',
				\WC_Paytrace::plugin_path() . '/templates/'
			);
		}
	}
	
	/**
	 * Outputs the Cards form
	 *
	 * @since 2.1
	 */
	public function output_cards_form() {
		$this->output_saved_payment_tokens( $this->saved_cards, 'card' );
		
		if ( 'protect' === $this->get_gateway()->get_security_type() ) {
			$this->output_protect_form_theme_css();
			
			// Load the credit card form template
			wc_get_template(
				'checkout/paytrace-protect-form.php',
				array(
					'gateway_id'         => $this->get_gateway()->id,
					'save_card_text'     => $this->get_gateway()->get_option( 'save_card_text' ),
					'show_save_customer' => $this->show_save_customer,
					'security_type'      => $this->get_gateway()->get_security_type(),
					'form'               => $this,
				),
				'',
				\WC_Paytrace::plugin_path() . '/templates/'
			);
		} else {
			// Load the credit card form template
			wc_get_template(
				'checkout/paytrace-payment-form.php',
				array(
					'gateway'               => $this->get_gateway()->id,
					'gateway_id'            => $this->get_gateway()->id,
					'save_card_text'        => $this->get_gateway()->get_option( 'save_card_text' ),
					'show_save_customer'    => $this->show_save_customer,
					'use_encrypted'         => 'encryption' === $this->get_gateway()->get_security_type(),
					'security_type'         => $this->get_gateway()->get_security_type(),
					'is_cvc_required_field' => $this->get_gateway()->is_cvc_required(),
					'form'                  => $this,
				),
				'',
				\WC_Paytrace::plugin_path() . '/templates/'
			);
		}
	}
	
	public function output_protect_form_theme_css() {
		$theme = \WcPaytrace\Helpers\Protect_Template_Helpers::get_protect_template( $this->get_gateway()->get_option( 'protect_form_theme', 'light_rounded' ) );
		ob_start();
		// @formatter:off
		?>
		<style>
			div#pt_hpf_form iframe {
			    <?php foreach( $theme['iframe'] as $rule => $value ) {
				    echo esc_attr($rule).':'.esc_attr($value).';';
			    } ?>
			}
		</style>
		<?php
		// @formatter:on
		
		echo apply_filters( 'wc_paytrace_protect_form_theme_css', ob_get_clean(), $this );
	}
	
	/**
	 * Outputs the Checks form
	 *
	 * @since 2.1
	 */
	public function maybe_output_checks_form() {
		if ( ! $this->support_checks ) {
			return;
		}
		
		$this->output_saved_payment_tokens( $this->saved_checks, 'check' );
		
		// Load the checks form template
		wc_get_template(
			'checkout/paytrace-checks-form.php',
			array(
				'gateway'            => $this->get_gateway()->id,
				'gateway_id'         => $this->get_gateway()->id,
				'show_save_customer' => $this->show_save_customer,
				'form'               => $this,
			),
			'',
			\WC_Paytrace::plugin_path() . '/templates/'
		);
	}
	
	/**
	 * Outputs the same method field
	 *
	 * @param string $type
	 */
	public function output_save_method_field( $type = 'card' ) {
		if ( ! $this->show_save_customer ) {
			return;
		}
		
		// Load the checks form template
		wc_get_template(
			'checkout/paytrace-save-method-field.php',
			array(
				'gateway'          => $this->get_gateway()->id,
				'gateway_id'       => $this->get_gateway()->id,
				'save_method_text' => $this->get_gateway()->get_option( 'save_card_text' ),
				'form'             => $this,
				'type'             => $type,
			),
			'',
			\WC_Paytrace::plugin_path() . '/templates/'
		);
	}
	
	/**
	 * Outputs the saved payment tokens fields
	 *
	 * @since 2.1
	 *
	 * @param array  $tokens Tokens
	 * @param string $type   Type of the tokens 'card' or 'check'
	 */
	public function output_saved_payment_tokens( $tokens, $type = 'card' ) {
		
		$type = in_array( $type, $this->allowed_token_type ) ? $type : $this->allowed_token_type[0];
		
		// Load the saved checks form template
		wc_get_template(
			'checkout/paytrace-saved-tokens.php',
			array(
				'gateway'    => $this->get_gateway()->id,
				'gateway_id' => $this->get_gateway()->id,
				'tokens'     => $tokens,
				'token_type' => $type,
			),
			'',
			\WC_Paytrace::plugin_path() . '/templates/'
		);
	}
	
	/**
	 * Prepares the separate form sections data.
	 * 1. Which sections we should display
	 * 2. Gathers the data for each of the sections to be displayed
	 *
	 * @since 2.4.0
	 */
	public function prepare_sections_data() {
		$this->set_payment_sections_to_display();
		$this->set_sections_data();
	}
	
	/**
	 * Set the show save customer prop.
	 * Controls the Save to account checkbox.
	 *
	 * @param $show_save_customer
	 */
	public function set_show_save_customer( $show_save_customer ) {
		$this->show_save_customer = $show_save_customer;
	}
	
	/**
	 * Sections to be displayed [card, check]
	 *
	 * @since 2.4.0
	 *
	 * @return array
	 */
	public function get_payment_sections_to_display() {
		return $this->payment_sections_to_display;
	}
	
	/**
	 * Sets the sections to be displayed
	 *
	 * @since 2.4.0
	 *
	 * @filter "wc_payment_sections_to_display_paytrace" allows for 3rd party to set the sections
	 */
	public function set_payment_sections_to_display() {
		$this->payment_sections_to_display[] = 'card';
		if ( $this->support_checks ) {
			$this->payment_sections_to_display[] = 'check';
		}
		
		// Allow for modification
		apply_filters( 'wc_payment_sections_to_display_paytrace', $this->payment_sections_to_display, $this );
	}
	
	/**
	 * Sets the sections data [slug, label, description]
	 *
	 * @filter "wc_payment_section_data_paytrace" allows modification of the data per section
	 *
	 * @since 2.4.0
	 */
	public function set_sections_data() {
		$forms = $this->get_payment_sections_to_display();
		
		foreach ( $forms as $form_type ) {
			
			$label       = 'separated_forms_label_' . $form_type;
			$description = 'separated_forms_description_' . $form_type;
			
			$this->sections_data[ $form_type ] = apply_filters( 'wc_payment_section_data_paytrace', array(
				'slug'        => $form_type,
				'label'       => $this->get_gateway()->{$label},
				'description' => $this->get_gateway()->{$description},
			), $form_type, $this );
		}
	}
	
	/**
	 * Returns the sections data
	 *
	 * @since 2.4.0
	 *
	 * @filter "wc_payment_sections_data_paytrace" allows modification of the sections data
	 *
	 * @return array
	 */
	public function get_sections_data() {
		return apply_filters( 'wc_payment_sections_data_paytrace', $this->sections_data, $this );
	}
}
