<?php

/**
 * WC_Paytrace_Gateway_Addons class.
 *
 * WC_Paytrace_Gateway_Addons manages the subscription part of the PayTrace functionality.
 *
 * @extends WC_Paytrace_Gateway
 * @author  VanboDevelops
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

class WC_Paytrace_Gateway_Addons extends WC_Paytrace_Gateway {
	
	function __construct() {
		
		parent::__construct();
		
		// Load hooks
		$this->hooks();
	}
	
	/**
	 * Loads the hooks for the gateway
	 * Methods ensures the hooks are loaded only once
	 */
	public function hooks() {
		
		if ( true === self::$loaded ) {
			return;
		}
		
		if ( 2 === WC_Paytrace::get_subscriptions_version() ) {
			// Scheduled payment
			add_action( 'woocommerce_scheduled_subscription_payment_' . $this->id, array(
				$this,
				'scheduled_subscription_payment_request',
			), 10, 2 );
			
			// Meta data renewal remove
			add_action( 'wcs_resubscribe_order_created', array( $this, 'remove_renewal_order_meta' ), 10 );
			
			// Update change payment method
			add_action( 'woocommerce_subscription_failing_payment_method_updated_' . $this->id, array(
				$this,
				'changed_failing_payment_method',
			), 10, 2 );
			
			// Display card used details
			add_filter( 'woocommerce_my_subscriptions_payment_method', array(
				$this,
				'maybe_render_subscription_payment_method',
			), 10, 2 );
			
			// Handle display of the Admin facing payment method change
			add_filter( 'woocommerce_subscription_payment_meta', array(
				$this,
				'add_subscription_payment_meta',
			), 10, 2 );
			
			// Handle validation of the Admin facing payment method change
			add_action( 'woocommerce_subscription_validate_payment_meta', array(
				$this,
				'validate_subscription_payment_meta',
			), 10, 3 );
		}
		
		// Add support for Pre-Orders
		if ( WC_Paytrace::is_pre_orders_active() ) {
			add_action( 'wc_pre_orders_process_pre_order_completion_payment_' . $this->id, array(
				$this,
				'process_pre_order_release_payment',
			) );
		}
		
		// Load the parent hooks
		parent::hooks();
	}
	
	/**
	 * Process the payment
	 *
	 * @param int $order_id
	 *
	 * @return array|bool
	 */
	function process_payment( $order_id ) {
		$order          = wc_get_order( $order_id );
		$paytrace_order = new WC_Paytrace_Order( $order );
		
		if ( $paytrace_order->contains_pre_order() ) {
			// Pre-Orders Payment
			return $this->process_pre_orders_payment( $order );
		} else if ( $paytrace_order->contains_subscription() || $this->is_change_payment_method( $order ) ) {
			// Subscription Payment
			return $this->process_subscription_payment( $order );
		} else {
			// Normal Products Payment
			return parent::process_payment( $order_id );
		}
	}
	
	/**
	 * Process subscription payment
	 *
	 * @since 1.2
	 *
	 * @param \WC_Order $order
	 *
	 * @return array|bool
	 */
	public function process_subscription_payment( $order ) {
		try {
			$initial_payment = $order->get_total();
			
			$integration = $this->get_integration_class();
			
			$integration->process_payment( $order, $initial_payment, true );
			
			$this->send_transaction_receipt_action( $order );
			
			$redirect_url = $this->get_return_url( $order );
			
			// No need to do any additional order actions if we are just changing payment methods
			if ( ! $this->is_change_payment_method( $order ) ) {
				
				// Complete the order, if we did not charge anything and the order is not yet completed
				// Case for free trial subscriptions
				if ( ! $order->is_paid() && 0 == $initial_payment ) {
					$order->payment_complete();
				}
				
				// Remove cart
				wc_empty_cart();
				
				$order_tokens = new \WC_Paytrace_Order( $order );
				$customer_id  = $order_tokens->get_customer_id();
				$this->save_meta_data_to_subscription( $order, $customer_id );
			} else {
				// Change method process will redirect to view subscription page
				$redirect_url = $order->get_view_order_url();
			}
			
			// Return thank you page redirect
			return array(
				'result'   => 'success',
				'redirect' => $redirect_url,
			);
		}
		catch ( Exception $e ) {
			wc_add_notice( $e->getMessage(), 'error' );
			
			if ( 'protect' == $this->get_security_type() ) {
				WC()->session->refresh_totals = true;
			}
			
			return false;
		}
	}
	
	/**
	 * Process subscription payment for each period
	 *
	 * @since 1.4
	 *
	 * @param float     $amount_to_charge
	 * @param \WC_Order $renewal_order
	 */
	public function scheduled_subscription_payment_request( $amount_to_charge, \WC_Order $renewal_order ) {
		try {
			WC_Paytrace::add_debug_log( 'Scheduled payment: ' . print_r( WC_Paytrace_Compat::get_order_id( $renewal_order ), true ) );
			
			$integration = $this->get_integration_class();
			
			$integration->process_transaction( $renewal_order, $amount_to_charge, true, true );
			
			$this->send_transaction_receipt_action( $renewal_order );
		}
		catch ( Exception $e ) {
			$renewal_order->update_status( 'failed', $e->getMessage() );
			
			// Debug log
			WC_Paytrace::add_debug_log( $e->getMessage() );
		}
	}
	
	/**
	 * Save the transaction details to the Subscription
	 *
	 * @since 1.4
	 *
	 * @param $order
	 * @param $customer_id
	 */
	public function save_meta_data_to_subscription( $order, $customer_id ) {
		// Also store it on the subscriptions being purchased or paid for in the order
		if ( wcs_order_contains_subscription( $order ) ) {
			$subscriptions = wcs_get_subscriptions_for_order( $order );
		} elseif ( wcs_order_contains_renewal( $order ) ) {
			$subscriptions = wcs_get_subscriptions_for_renewal_order( $order );
		} else {
			$subscriptions = array();
		}
		
		foreach ( $subscriptions as $subscription ) {
			// Debug log
			WC_Paytrace::add_debug_log( 'Saving details to subscription: ' . print_r( WC_Paytrace_Compat::get_order_id( $subscription ), true ) );
			
			$order_tokens = new \WC_Paytrace_Order( $subscription );
			$order_tokens->save_customer_id( $customer_id );
		}
	}
	
	/**
	 * Don't transfer PayTrace meta to resubscribe orders.
	 *
	 * @since 1.4
	 *
	 * @param WC_Order $resubscribe_order The order created for the customer to resubscribe to the old expired/cancelled subscription
	 *
	 * @return void
	 */
	public function remove_renewal_order_meta( $resubscribe_order ) {
		$order_tokens = new \WC_Paytrace_Order( $resubscribe_order );
		$order_tokens->delete_customer_id();
	}
	
	/**
	 * Update the accountID and the Serial No on the original order, when a renewal order is placed, after it failed previously
	 *
	 * @since 1.2
	 *
	 * @param \WC_Order $subscription
	 * @param \WC_Order $renewal_order
	 */
	public function changed_failing_payment_method( \WC_Order $subscription, \WC_Order $renewal_order ) {
		$renewal_tokens    = new \WC_Paytrace_Order( $renewal_order );
		$renew_customer_id = $renewal_tokens->get_customer_id();
		
		$sub_tokens = new \WC_Paytrace_Order( $subscription );
		$sub_tokens->save_customer_id( $renew_customer_id );
	}
	
	/**
	 * Show payment method on My Subscriptions section of My Account page
	 *
	 * @since 1.2
	 *
	 * @param string   $payment_method_to_display
	 * @param WC_Order $subscription
	 *
	 * @return string
	 */
	public function maybe_render_subscription_payment_method( $payment_method_to_display, $subscription ) {
		if ( $this->id !== WC_Paytrace_Compat::get_order_prop( $subscription, 'payment_method' ) ) {
			return $payment_method_to_display;
		}
		
		$order_tokens = new \WC_Paytrace_Order( $subscription );
		$customer_id  = $order_tokens->get_customer_id();
		
		$customer_tokens = new \WC_Paytrace_Customer_Tokens( $subscription->get_user_id() );
		$token           = $customer_tokens->get_token_by_customer_id( $customer_id );
		
		if ( ! $token ) {
			return $payment_method_to_display;
		}
		
		if ( 'Paytrace_eCheck' == $token->get_type() ) {
			$payment_method_to_display = sprintf( __( 'Via Check with account ending in %s', WC_Paytrace::TEXT_DOMAIN ), $token->get_last4() );
		} else {
			$payment_method_to_display = sprintf( __( 'Via Card ending in %s', WC_Paytrace::TEXT_DOMAIN ), $token->get_last4() );
		}
		
		return $payment_method_to_display;
	}
	
	/**
	 * Add payment method change fields
	 *
	 * @since 1.4
	 *
	 * @param $payment_meta
	 * @param $subscription
	 *
	 * @return mixed
	 */
	public function add_subscription_payment_meta( $payment_meta, $subscription ) {
		$subscription_new_up = wcs_get_subscription( $subscription->get_id() );
		
		$order_tokens = new \WC_Paytrace_Order( $subscription_new_up );
		
		$payment_meta[ $this->id ] = array(
			'post_meta' => array(
				'_paytrace_customer_id' => array(
					'value' => $order_tokens->get_customer_id(),
					'label' => 'PayTrace Customer Profile ID',
				),
			),
		);
		
		return $payment_meta;
	}
	
	/**
	 * Validate Payment method change
	 *
	 * @since 1.4
	 *
	 * @param                      $payment_method_id
	 * @param                      $payment_meta
	 * @param WC_Subscription|null $subscription
	 *
	 * @throws Exception
	 */
	public function validate_subscription_payment_meta( $payment_method_id, $payment_meta, $subscription = null ) {
		if ( $this->id !== $payment_method_id ) {
			return;
		}
		
		// Process:
		//      1. Get the Vault token from Paytrace
		//      2. Add the token to the system as WC_Payment_Token_Paytrace_CC or WC_Payment_Token_Paytrace_eCheck
		try {
			// If nothing is set here, then we will can't charge the next period
			if ( ! isset( $payment_meta['post_meta']['_paytrace_customer_id']['value'] )
			     || empty( $payment_meta['post_meta']['_paytrace_customer_id']['value'] )
			) {
				throw new Exception( __( 'A "_paytrace_customer_id"(PayTrace Customer Profile ID) value is required.', WC_Paytrace::TEXT_DOMAIN ) );
			}
			
			// Can't do anything if we don't know the subscription we are adding the token for
			if ( null == $subscription ) {
				return;
			}
			
			// Bail, if we are not using the JSON API
			if ( 'json' != $this->get_option( 'integration' ) ) {
				return;
			}
			
			$paytrace_vault_customer_id = $payment_meta['post_meta']['_paytrace_customer_id']['value'];
			
			$customer_tokens = new WC_Paytrace_Customer_Tokens( $subscription->get_user_id() );
			
			// The token already exists in the system
			if ( $customer_tokens->get_token_by_customer_id( $paytrace_vault_customer_id ) ) {
				return;
			}
			
			$customer_tokens->create_wc_token_from_paytrace_customer_id( $paytrace_vault_customer_id, $this->get_integration_class() );
		}
		catch ( Exception $e ) {
			WC_Paytrace::add_debug_log( $e->getMessage() );
			
			throw new Exception( $e->getMessage() );
		}
	}
	
	/**
	 * Process Pre-Order payment request.
	 * Only Register the customer payment method, pre-order payment is delayed one.
	 *
	 * @since 1.2
	 *
	 * @param WC_Order $order
	 *
	 * @return array
	 */
	private function process_pre_orders_payment( WC_Order $order ) {
		
		if ( ! WC_Pre_Orders_Order::order_requires_payment_tokenization( $order ) ) {
			// Process order normally
			return parent::process_payment( WC_Paytrace_Compat::get_order_id( $order ) );
		}
		
		try {
			// Using Check(ACH) to pay
			if ( 'new_check' == $this->get_used_payment_type() || 'saved_check' == $this->get_used_payment_type() ) {
				$with_token = true;
				$type       = 'check';
				// Handle Customer Profile
				if ( 'new_check' == $this->get_used_payment_type() ) {
					$with_token = false;
				}
			} else {
				$with_token = true;
				$type       = 'card';
				// Handle Customer Profile
				if ( 'new_card' == $this->get_used_payment_type() ) {
					$with_token = false;
				}
			}
			
			// Handle Customer Profile
			if ( ! $with_token ) {
				// Create and save new customer profile
				$integration = $this->get_integration_class();
				
				$integration->process_profile_create( $order, $type );
			} else {
				// Get the used token
				$customer_tokens = new \WC_Paytrace_Customer_Tokens( $order->get_user_id() );
				$token           = $customer_tokens->get_token( 'check' == $type ? $this->get_submitted_token_value( 'check' ) : $this->get_submitted_token_value( 'card' ) );
				// Save it to the order
				$order_tokens = new \WC_Paytrace_Order( $order );
				$order_tokens->save_customer_id( $token->get_token() );
			}
			
			// Now that we have the info need for future payment, mark the order pre-ordered
			WC_Pre_Orders_Order::mark_order_as_pre_ordered( $order );
			
			$this->send_transaction_receipt_action( $order );
			
			// Empty cart
			wc_empty_cart();
			
			// Redirect to thank you page
			return array(
				'result'   => 'success',
				'redirect' => $this->get_return_url( $order ),
			);
		}
		catch ( Exception $e ) {
			wc_add_notice( $e->getMessage(), 'error' );
			
			if ( 'protect' == $this->get_security_type() ) {
				WC()->session->refresh_totals = true;
			}
		}
	}
	
	/**
	 * Charge the payment on order release
	 *
	 * @since 1.2
	 *
	 * @param WC_Order $order
	 */
	public function process_pre_order_release_payment( \WC_Order $order ) {
		try {
			// Create and save new customer profile
			$integration = $this->get_integration_class();
			
			$integration->process_transaction( $order, null, false, true );
		}
		catch ( Exception $e ) {
			$order->add_order_note( $e->getMessage(), 'error' );
		}
	}
	
	/**
	 * Returns true, if order contains Subscription
	 *
	 * @since 1.4
	 *
	 * @param WC_Order $order
	 *
	 * @return bool
	 */
	public function order_contains_subscription( \WC_Order $order ) {
		$paytrace_order = new WC_Paytrace_Order( $order );
		
		return $paytrace_order->contains_subscription();
	}
	
	/**
	 * Returns true, if order contains Pre-Order
	 *
	 * @since 1.4
	 *
	 * @param WC_Order $order
	 *
	 * @return bool
	 */
	public function order_contains_pre_order( WC_Order $order ) {
		$paytrace_order = new WC_Paytrace_Order( $order );
		
		return $paytrace_order->contains_pre_order();
	}
	
	/**
	 * Is the request to change payment method
	 *
	 * @since 2.4.0
	 *
	 * @param $subscription
	 *
	 * @return bool
	 */
	public function is_change_payment_method( $subscription ) {
		if ( class_exists( 'WC_Subscriptions_Change_Payment_Gateway' ) ) {
			return \WC_Subscriptions_Change_Payment_Gateway::$is_request_to_change_payment;
		}
		
		return isset( $_GET['change_payment_method'] ) && $_GET['change_payment_method'] == WC_Paytrace_Compat::get_order_id( $subscription );
	}
}