<?php
/**
 * Plugin Name: WooCommerce PayTrace Payment Gateway
 * Plugin URI: http://www.woothemes.com/products/paytrace-gateway/
 * Description: Allows you to use <a href="http://www.paytrace.com/">PayTrace</a> payment processor with the WooCommerce plugin.
 * Version: 1.4.1
 * Author: VanboDevelops
 * Author URI: http://www.vanbodevelops.com
 *
 *        Copyright: (c) 2012 - 2015 VanboDevelops
 *        License: GNU General Public License v3.0
 *        License URI: http://www.gnu.org/licenses/gpl-3.0.html
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Required functions
 */
if ( ! function_exists( 'woothemes_queue_update' ) ) {
	require_once( 'woo-includes/woo-functions.php' );
}

/**
 * Plugin updates
 */
woothemes_queue_update( plugin_basename( __FILE__ ), '75d0d10deafdfc281fd154423ef70922', '18723' );

if ( ! is_woocommerce_active() ) {
	return;
}

class WC_PayTrace {

	/**
	 * Text domain string. Constant
	 */
	const TEXT_DOMAIN = 'wc_paytrace';

	/**
	 * WC Logger object
	 * @var object
	 */
	private static $log;

	/**
	 * Plugin URL
	 * @var string
	 */
	private static $plugin_url;

	/**
	 * Plugin Path
	 * @var string
	 */
	private static $plugin_path;

	/**
	 * Is WC Subscriptions active
	 * @var bool
	 */
	private static $is_subscriptions_active;

	/**
	 * Hold WC Subscriptions main version
	 * @var bool
	 */
	private static $is_subscriptions_version;

	/**
	 * Is WC Pre-Orders active
	 * @var bool
	 */
	private static $is_pre_orders_active;

	/**
	 * Do we have debug mode enabled
	 * @var bool
	 */
	private static $is_debug_enabled;

	public function __construct() {
		// Add required files
		add_action( 'plugins_loaded', array( $this, 'init' ), 0 );

		// Add a 'Settings' link to the plugin action links
		add_filter( 'plugin_action_links_' . plugin_basename( __FILE__ ), array( $this, 'settings_support_link' ), 10, 4 );

		add_filter( 'woocommerce_payment_gateways', array( $this, 'add_payment_gateway' ) );

		// Display the saved cards on My Account page
		add_action( 'woocommerce_after_my_account', array( $this, 'display_saved_cards' ) );

		// Manage Saved Cards
		add_action( 'template_redirect', array( $this, 'manage_saved_cards' ) );

		add_action( 'admin_notices', array( $this, 'paytrace_check_ssl' ) );
	}

	/**
	 * Init localisations and files
	 */
	public function init() {
		if ( ! class_exists( 'WC_Payment_Gateway' ) ) {
			return;
		}

		// Includes
		include_once( 'classes/class-wc-compat-paytrace.php' );
		include_once( 'classes/class-wc-gateway-paytrace.php' );
		include_once( 'classes/class.paytrace-api.php' );

		if ( self::is_pre_orders_active() || self::is_subscriptions_active() ) {
			include_once( 'classes/class-wc-gateway-paytrace-addons.php' );
			if ( 1 === self::get_subscriptions_version() ) {
				include_once( 'classes/deprecated/class-wc-paytrace-gateway-addons-deprecated.php' );
			}
		}

		// Localisation
		load_plugin_textdomain( self::TEXT_DOMAIN, false, dirname( plugin_basename( __FILE__ ) ) . '/languages/' );
	}

	/**
	 * Add the gateway to WooCommerce
	 *
	 * @since 1.2
	 *
	 * @param array $methods
	 *
	 * @return array
	 */
	function add_payment_gateway( $methods ) {
		$methods[] = $this->get_gateway_class();

		return $methods;
	}

	/**
	 * Add 'Settings' link to the plugin actions links
	 *
	 * @since 1.2
	 *
	 * @param array $actions
	 *
	 * @return array associative array of plugin action links
	 */
	public function settings_support_link( $actions, $plugin_file, $plugin_data, $context ) {

		$gateway = $this->get_gateway_class();

		return array_merge(
			array( 'settings' => '<a href="' . WC_Compat_PayTrace::gateway_settings_page( $gateway ) . '">' . __( 'Settings', self::TEXT_DOMAIN ) . '</a>' ),
			$actions
		);
	}

	/**
	 * Get the correct gateway class name to load
	 *
	 * @since 1.2
	 * @return string Class name
	 */
	private function get_gateway_class() {
		if ( self::is_pre_orders_active() || self::is_subscriptions_active() ) {
			$methods = 'WC_Gateway_PayTrace_Addons';

			// Support for WooCommerce Subscriptions 1.n
			if ( 1 === self::get_subscriptions_version() ) {
				$methods = 'WC_PayTrace_Gateway_Addons_Deprecated';
			}
		} else {
			$methods = 'WC_Gateway_PayTrace';
		}

		return $methods;
	}

	/**
	 * Safely get POST variables
	 *
	 * @since 1.2
	 *
	 * @param string $name POST variable name
	 *
	 * @return string The variable value
	 */
	public static function get_post( $name ) {
		if ( isset( $_POST[ $name ] ) ) {
			return $_POST[ $name ];
		}

		return null;
	}

	/**
	 * Safely get GET variables
	 *
	 * @since 1.2
	 *
	 * @param string $name GET variable name
	 *
	 * @return string The variable value
	 */
	public static function get_get( $name ) {
		if ( isset( $_GET[ $name ] ) ) {
			return $_GET[ $name ];
		}

		return null;
	}

	/**
	 * Add debug log message
	 *
	 * @since 1.2
	 *
	 * @param string $message
	 */
	public static function add_debug_log( $message ) {
		if ( ! is_object( self::$log ) ) {
			self::$log = WC_Compat_PayTrace::get_wc_logger();
		}

		if ( self::is_debug_enabled() ) {
			self::$log->add( 'paytrace', $message );
		}
	}

	/**
	 * Check, if debug logging is enabled
	 *
	 * @since 1.3
	 * @return bool
	 */
	public static function is_debug_enabled() {
		if ( self::$is_debug_enabled ) {
			return self::$is_debug_enabled;
		} else {
			$paytrace_settings = get_option( 'woocommerce_paytrace_settings' );

			self::$is_debug_enabled = ( 'yes' == $paytrace_settings['debug'] );

			return self::$is_debug_enabled;
		}
	}

	/**
	 * Get the plugin url
	 *
	 * @since 1.2
	 * @return string
	 */
	public static function plugin_url() {
		if ( self::$plugin_url ) {
			return self::$plugin_url;
		}

		return self::$plugin_url = untrailingslashit( plugins_url( '/', __FILE__ ) );
	}

	/**
	 * Get the plugin path
	 *
	 * @since 1.2
	 * @return string
	 */
	public static function plugin_path() {
		if ( self::$plugin_path ) {
			return self::$plugin_path;
		}

		return self::$plugin_path = untrailingslashit( plugin_dir_path( __FILE__ ) );
	}

	/**
	 * Detect if WC Subscriptions is active
	 *
	 * @since 1.2
	 * @return bool True if active, False if not
	 */
	public static function is_subscriptions_active() {
		if ( is_bool( self::$is_subscriptions_active ) ) {
			return self::$is_subscriptions_active;
		}

		self::$is_subscriptions_active = false;
		if ( class_exists( 'WC_Subscriptions' ) || function_exists( 'wcs_order_contains_subscription' ) ) {
			self::$is_subscriptions_active = true;
		}

		return self::$is_subscriptions_active;
	}

	/**
	 * Get back the Subsctiptions version.
	 *
	 * @since 1.4
	 * @return bool Main Subscriptions version number (e.i. 1, 2, 3), False, if Subscriptions is not active
	 */
	public static function get_subscriptions_version() {
		if ( null !== self::$is_subscriptions_version ) {
			return self::$is_subscriptions_version;
		}

		self::$is_subscriptions_version = false;

		if ( function_exists( 'wcs_order_contains_subscription' ) ) {
			self::$is_subscriptions_version = 2;
		} elseif ( class_exists( 'WC_Subscriptions' ) ) {
			self::$is_subscriptions_version = 1;
		}

		return self::$is_subscriptions_version;
	}

	/**
	 * Detect if Pre-Orders is active
	 *
	 * @since 1.2
	 * @return bool True if actiove, False if not
	 */
	public static function is_pre_orders_active() {
		if ( is_bool( self::$is_pre_orders_active ) ) {
			return self::$is_pre_orders_active;
		}

		if ( class_exists( 'WC_Pre_Orders' ) ) {
			self::$is_pre_orders_active = true;
		} else {
			self::$is_pre_orders_active = false;
		}

		return self::$is_pre_orders_active;
	}

	/**
	 * Manage the customer saved credit cards
	 *
	 * @since 1.2
	 */
	public function display_saved_cards() {

		// Customer should be logged in
		if ( ! is_user_logged_in() ) {
			return;
		}

		// Init the gateway
		$gateway_name = $this->get_gateway_class();
		$gateway      = new $gateway_name;

		// Don't show if the gateway is not available for use
		if ( ! $gateway->is_available() ) {
			return;
		}

		$saved_profiles = get_user_meta( get_current_user_id(), '_paytrace_saved_profiles', false );

		$saved_checks = array();
		$saved_cards  = array();
		foreach ( $saved_profiles as $n => $profile ) {
			if ( isset( $profile['tr_last4'] ) ) {
				$saved_checks[ $n ] = $profile;
			} elseif ( isset( $profile['last4'] ) ) {
				$saved_cards[ $n ] = $profile;
			}
		}

		// Don't show if there are not profiles or the save customers is not enabled
		if ( ( ! $saved_cards && ! $saved_checks ) || 'no' == $gateway->save_customers ) {
			return;
		}

		if ( $saved_cards ) {
			WC_Compat_PayTrace::wc_get_template(
				'myaccount/myaccount-paytrace-saved-cards.php',
				array(
					'saved_cards'  => $saved_cards,
					'available_cc' => $gateway->available_cc,
					'card_options' => $gateway->card_options,
				),
				'',
				self::plugin_path() . '/templates/'
			);
		}

		if ( $saved_checks ) {
			WC_Compat_PayTrace::wc_get_template(
				'myaccount/myaccount-paytrace-saved-checks.php',
				array(
					'saved_checks' => $saved_checks,
				),
				'',
				self::plugin_path() . '/templates/'
			);
		}


	}

	/**
	 * Detect and process customer update and delete profile requests
	 *
	 * @since 1.2
	 * @return bool
	 */
	public function manage_saved_cards() {

		if ( ! is_user_logged_in() ) {
			return;
		}

		// Init the gateway
		$gateway_name = $this->get_gateway_class();
		$gateway      = new $gateway_name;

		try {
			if ( null != self::get_post( 'paytrace_delete_card' ) ) {
				// First verify the request was send from the manage card form
				$this->verify_request( 'paytrace-nonce', 'paytrace-delete-card' );

				$gateway->delete_customer_profile( (int) self::get_post( 'paytrace_delete_card' ) );

				WC_Compat_PayTrace::wc_add_notice( __( 'Card deleted successfully', self::TEXT_DOMAIN ), 'success' );
			} elseif ( null != self::get_post( 'paytrace_delete_check' ) ) {
				// First verify the request was send from the manage card form
				$this->verify_request( 'paytrace-nonce', 'paytrace-delete-check' );

				$gateway->delete_customer_profile( (int) self::get_post( 'paytrace_delete_check' ) );

				WC_Compat_PayTrace::wc_add_notice( __( 'Check deleted successfully', self::TEXT_DOMAIN ), 'success' );
			}

			if ( null != self::get_post( 'paytrace_update_card' ) ) {
				if ( self::get_post( 'paytrace_cc_number' ) &&
					self::get_post( 'paytrace_cc_exp_month' ) &&
					self::get_post( 'paytrace_cc_exp_year' ) &&
					self::get_post( 'paytrace_cc_name' )
				) {
					// First verify the request was send from the manage card form
					$this->verify_request( 'paytrace-nonce', 'paytrace-update-card' );

					$gateway->update_customer_profile( (int) self::get_post( 'paytrace_update_card' ) );

					WC_Compat_PayTrace::wc_add_notice( __( 'Card updated successfully', self::TEXT_DOMAIN ), 'success' );
				}
			} elseif ( null != self::get_post( 'paytrace_update_check' ) ) {

				self::add_debug_log( 'Updating check: ' . self::get_post( 'paytrace_update_check' ) );

				if ( self::get_post( 'paytrace_routing_number' ) &&
					self::get_post( 'paytrace_account_number' ) &&
					self::get_post( 'paytrace_check_name' )
				) {
					// First verify the request was send from the manage card form
					$this->verify_request( 'paytrace-nonce', 'paytrace-update-check' );

					self::add_debug_log( 'Run the update profile: ' );

					$gateway->update_customer_profile( (int) self::get_post( 'paytrace_update_check' ) );

					WC_Compat_PayTrace::wc_add_notice( __( 'Check updated successfully', self::TEXT_DOMAIN ), 'success' );
				}
			}
		}
		catch ( Exception $e ) {
			WC_Compat_PayTrace::wc_add_notice( $e->getMessage(), 'error' );
		}
	}

	/**
	 * Verify a form request action
	 *
	 * @since 1.2
	 *
	 * @param type $action
	 *
	 * @throws Exception
	 */
	private function verify_request( $name, $action ) {

		if ( ! wp_verify_nonce( self::get_post( $name ), $action ) ) {
			throw new Exception( __( 'Cannot verify the request, please try again.', self::TEXT_DOMAIN ) );
		}

	}

	/**
	 * Return the order number with stripped # or n° ( french translations )
	 *
	 * @since 1.2
	 *
	 * @param WC_Order $order
	 *
	 * @return string
	 */
	public static function get_order_number( WC_Order $order ) {
		return str_replace( array( '#', 'n°' ), '', $order->get_order_number() );
	}

	/**
	 * Check if SSL is forced on the checkout page.
	 **/
	function paytrace_check_ssl() {
		$paytrace_settings = get_option( 'woocommerce_paytrace_settings' );

		if ( 'no' == $paytrace_settings['enabled'] ) {
			return;
		}

		if ( get_option( 'woocommerce_force_ssl_checkout' ) == 'no' && 'no' == $paytrace_settings['testmode'] && ! class_exists( 'WordPressHTTPS' ) ) {
			echo '<div class="error"><p>';
			echo __(
				'PayTrace is enabled for live payments, but your checkout page is not secured with SSL.
						Your customers sensitive information is not protected.'
			);
			echo '<br/>';
			echo sprintf(
				__( 'Please enable the "Force secure checkout" option under %sSettings > Checkout%s.' ),
				'<a href="' . admin_url( 'admin.php?page=wc-settings&tab=checkout' ) . '">',
				'</a>'
			);
			echo '</p></div>';
		}
	}

}

new WC_PayTrace(); //end WC_PayTrace class
