<?php

/**
 * Fee Handler.
 */
if ( ! defined( 'ABSPATH' ) ) {
	exit ; // Exit if accessed directly.
}


if ( ! class_exists( 'EFW_Fees_Handler' ) ) {

	/**
	 * Class.
	 */
	class EFW_Fees_Handler {

		/**
		 * Calculated Fee.
		 * 
		 * @var float
		 * */
		protected static $fees_calculated ;

		/**
		 * Date filter.
		 * 
		 * @var bool
		 * */
		protected static $date_filter ;

		/**
		 * Product filter.
		 * 
		 * @var bool
		 * */
		protected static $product_filter ;

		/**
		 * User filter for Product.
		 * 
		 * @var bool
		 * */
		protected static $user_filter_for_product ;

		/**
		 * User filter for gateway.
		 * 
		 * @var bool
		 * */
		protected static $user_filter_for_gateway ;

		/**
		 * User filter for Order.
		 * 
		 * @var bool
		 * */
		protected static $user_filter_for_order ;

		/**
		 * Class Initialization.
		 */
		public static function init() {
			//Apply Gateway Deal in Checkout.
			add_action( 'wp_ajax_efw_add_fee' , array( __CLASS__ , 'efw_add_fee' ) ) ;
			//Apply Gateway Deal in Checkout.
			add_action( 'wp_ajax_nopriv_efw_add_fee' , array( __CLASS__ , 'efw_add_fee' ) ) ;
			//Add Product Fee in Cart/Checkout.
			add_action( 'woocommerce_get_cart_item_from_session' , array( __CLASS__ , 'set_product_price' ) , 10 , 3 ) ;
			//Display Fee Details in Single Product Page.
			add_action( 'woocommerce_before_add_to_cart_button' , array( __CLASS__ , 'fee_details_in_product' ) , 10 ) ;
			//Display Fee Details for Variation in Single Product Page.
			add_action( 'wp_ajax_efw_variation_notice' , array( __CLASS__ , 'efw_variation_notice' ) ) ;
			//Display Fee Details for Variation in Single Product Page.
			add_action( 'wp_ajax_nopriv_efw_variation_notice' , array( __CLASS__ , 'efw_variation_notice' ) ) ;
			//Change Add To Cart text in Shop Page.
			add_action( 'woocommerce_product_add_to_cart_text' , array( __CLASS__ , 'add_to_cart_text_in_shop' ) , 10 , 2 ) ;
			//Change Add To Cart URL in Shop Page.
			add_action( 'woocommerce_product_add_to_cart_url' , array( __CLASS__ , 'add_to_cart_url_in_shop' ) , 10 , 2 ) ;
			//Change Add To Cart URL in Shop Page when Ajax is enabled.
			add_filter( 'woocommerce_loop_add_to_cart_args' , array( __CLASS__ , 'add_to_cart_url_in_shop_for_ajax' ) , 10 , 2 ) ;
			//Display cart item data
			add_filter( 'woocommerce_get_item_data' , array( __CLASS__ , 'add_custom_item_data' ) , 10 , 2 ) ;
			//Add Gateway Fee in Cart/Checkout
			add_action( 'woocommerce_cart_calculate_fees' , array( __CLASS__ , 'gateway_fee' ) , 1001 ) ;
			//Add Order Total Fee in Cart/Checkout
			add_action( 'woocommerce_cart_calculate_fees' , array( __CLASS__ , 'order_total_fee' ) , 10 ) ;
			// May be add shipping fee.
			add_action('woocommerce_cart_calculate_fees', array(__CLASS__,'may_be_add_shipping_fee'), 10);
			//Update Fee Details in Order Meta
			add_action( 'woocommerce_checkout_update_order_meta' , array( __CLASS__ , 'update_fee_details' ) , 999 ) ;
			//Display Variation Price.
			add_action( 'woocommerce_get_price_html' , array( __CLASS__ , 'display_variation_price' ) , 10 , 2 ) ;
			//Add Product in Admin Post Table
			add_filter( 'manage_edit-product_columns' , array( __CLASS__ , 'product_fee_columns' ) , 12 ) ;
			//Add Product Data in Admin Post Table
			add_action( 'manage_product_posts_custom_column' , array( __CLASS__ , 'product_fee_content' ) , 12 , 2 ) ;
		}

		/**
		 * Add Fee in Checkout for Gateway.
		 */
		public static function efw_add_fee() {
			check_ajax_referer( 'efw-fee-nonce' , 'efw_security' ) ;

			try {
				$gateway_id = isset( $_REQUEST[ 'gatewayid' ] ) ? wc_clean( wp_unslash( $_REQUEST[ 'gatewayid' ] ) ) : '' ;
				if ( 'yes' == get_option( 'efw_gatewayfee_enable' ) && 'yes' == get_option( 'efw_enable_fee_for_' . $gateway_id ) ) {
					WC()->session->set( 'efw_gateway_id' , $gateway_id ) ;
				} else {
					WC()->session->__unset( 'efw_gateway_id' ) ;
					remove_action( 'woocommerce_cart_calculate_fees' , array( __CLASS__ , 'gateway_fee' ) ) ;
				}

				wp_send_json_success() ;
			} catch ( exception $ex ) {
				wp_send_json_error( array( 'error' => $ex->getMessage() ) ) ;
			}
		}

		/**
		 * Display Fee Details for Variation in Single Product Page.
		 */
		public static function efw_variation_notice() {
			check_ajax_referer( 'efw-fee-nonce' , 'efw_security' ) ;

			try {
				$variationid = isset( $_REQUEST[ 'variationid' ] ) ? absint( $_REQUEST[ 'variationid' ] ) : 0 ;

				$product_obj = wc_get_product( $variationid ) ;

				$price = wc_get_price_to_display( $product_obj ) + efw_get_wc_signup_fee( $product_obj , $variationid ) ;

				$product_fee = self::product_fee( $variationid , $price , 'fee' ) ;

				$total_payable_amount = self::rule_fees( $variationid , $product_fee , $price ) ;

				$fee_text = efw_get_fee_text( $variationid ) ;

				$rule_fee_text = efw_get_rule_fee_text( $variationid , $price ) ;

				if ( ! empty( $product_fee ) ) {
					ob_start() ;
					efw_get_template( 'single/product-fee-notice.php' , array( 'product' => $product_obj , 'product_fee' => $product_fee , 'fee_text' => $fee_text , 'rule_fee_texts' => $rule_fee_text , 'total_payable_amount' => $total_payable_amount , 'price' => $price , 'display_price' => true ) ) ;
					$html = ob_get_clean() ;
				} else {
					$html = wc_price( $price ) ;
				}

				wp_send_json_success( array( 'html' => $html ) ) ;
			} catch ( exception $ex ) {
				wp_send_json_error( array( 'error' => $ex->getMessage() ) ) ;
			}
		}

		/**
		 * Display Fee details in Single Product Page.
		 */
		public static function fee_details_in_product() {

			if ( 'yes' != get_option( 'efw_productfee_enable' ) ) {
				return ;
			}

			global $post ;

			$product_obj = wc_get_product( $post->ID ) ;

			if ( ! is_object( $product_obj ) ) {
				return ;
			}

			if ( 'variable' == $product_obj->get_type() || 'variable-subscription' == $product_obj->get_type() ) {
				return ;
			}

			$price = wc_get_price_to_display( $product_obj ) + efw_get_wc_signup_fee( $product_obj ) ;

			$product_fee = self::product_fee( $post->ID , $price , 'fee' ) ;

			$total_payable_amount = self::rule_fees( $post->ID , $product_fee , $price ) ;

			if ( empty( $product_fee ) ) {
				return ;
			}

			$fee_text = efw_get_fee_text( $post->ID ) ;

			$rule_fee_text = efw_get_rule_fee_text( $post->ID , $price ) ;

			efw_get_template( 'single/product-fee-notice.php' , array( 'product' => $product_obj , 'product_fee' => $product_fee , 'fee_text' => $fee_text , 'rule_fee_texts' => $rule_fee_text , 'price' => $price , 'total_payable_amount' => $total_payable_amount , 'display_price' => false ) ) ;
		}

		/**
		 * Change Add To Cart text in Shop Page.
		 */
		public static function add_to_cart_text_in_shop( $text, $product_obj ) {

			if ( 'yes' != get_option( 'efw_productfee_enable' ) ) {
				return $text ;
			}

			if ( 'variable' == $product_obj->get_type() ) {
				return $text ;
			}

			$post_id = $product_obj->get_id() ;

			$price = ( float ) wc_get_price_including_tax( $product_obj ) + efw_get_wc_signup_fee( $product_obj ) ;

			$product_fee = self::product_fee( $post_id , $price , 'fee' ) ;

			return empty( $product_fee ) ? $text : esc_html__( get_option( 'efw_productfee_add_to_cart_label' , 'View Final Value' ) ) ;
		}

		/**
		 * Change Add To Cart URL in Shop Page.
		 */
		public static function add_to_cart_url_in_shop( $url, $product_obj ) {

			if ( 'yes' != get_option( 'efw_productfee_enable' ) ) {
				return $url ;
			}

			$post_id = $product_obj->get_id() ;

			$price = ( float ) wc_get_price_including_tax( $product_obj ) + efw_get_wc_signup_fee( $product_obj ) ;

			$product_fee = self::product_fee( $post_id , $price , 'fee' ) ;

			return empty( $product_fee ) ? $url : get_the_permalink() ;
		}

		/**
		 * Change Add To Cart URL in Shop Page when Ajax enabled.
		 */
		public static function add_to_cart_url_in_shop_for_ajax( $args, $product ) {

			if ( 'yes' != get_option( 'efw_productfee_enable' ) ) {
				return $args ;
			}

			$post_id = $product->get_id() ;

			$price = ( float ) wc_get_price_including_tax( $product ) + efw_get_wc_signup_fee( $product ) ;

			$product_fee = self::product_fee( $post_id , $price , 'fee' ) ;

			$args[ 'class' ] = empty( $product_fee ) ? $args[ 'class' ] : str_replace( 'ajax_add_to_cart' , '' , $args[ 'class' ] ) ;

			return $args ;
		}

		/*
		 *  Add custom item data
		 */

		public static function add_custom_item_data( $item_data, $cart_item ) {

			if ( 'yes' != get_option( 'efw_productfee_enable' ) ) {
				return $item_data ;
			}

			$product_id = empty( $cart_item[ 'variation_id' ] ) ? $cart_item[ 'product_id' ] : $cart_item[ 'variation_id' ] ;

			$product = wc_get_product( $product_id ) ;

			$price = isset( $cart_item[ 'nyp' ] ) ? $cart_item[ 'nyp' ] : wc_get_price_excluding_tax( $product ) + efw_get_wc_signup_fee( $product ) ;

			$product_fee = self::product_fee( $product_id , $price , 'fee' ) ;

			if ( empty( $product_fee ) ) {
				return $item_data ;
			}

			$qty = isset( $cart_item[ 'quantity' ] ) ? $cart_item[ 'quantity' ] : 1 ;

			$product_fee = ( $product_fee * $qty ) ;
						/**
						 * Hook:efw_product_fee_before_custom_item_data. 
						 *
						 * @since 1.0
						 */
			$product_fee = apply_filters( 'efw_product_fee_before_custom_item_data' , $product_fee , $cart_item ) ;

			if ( ! is_numeric( $product_fee ) ) {
				return $item_data ;
			}

			$key_name = efw_get_fee_text( $product_id ) ;

			$item_data[] = array(
				'name'    => $key_name ,
				'display' => wc_price( $product_fee ) ,
					) ;

			$rule_fee_texts = efw_get_rule_fee_text( $product_id , $price ) ;

			if ( efw_check_is_array( $rule_fee_texts ) ) {
				foreach ( $rule_fee_texts as $rule_fee_text => $rule_fee_value ) {

					$rule_fee_value = ( $rule_fee_value * $qty ) ;

					$item_data[] = array(
						'name'    => $rule_fee_text ,
						'display' => wc_price( $rule_fee_value ) ,
							) ;
				}
			}

			return $item_data ;
		}

		/**
		 * Add Product Fee.
		 */
		public static function set_product_price( $session_data, $values, $key ) {

			if ( 'yes' != get_option( 'efw_productfee_enable' ) ) {
				return $session_data ;
			}
						/**
						 * Hook:efw_calculate_product_fee_in_cart. 
						 *
						 * @since 1.0
						 */
			if ( ! apply_filters( 'efw_calculate_product_fee_in_cart' , true , $session_data ) ) {
				return $session_data ;
			}

			$product_id = empty( $session_data[ 'variation_id' ] ) ? $session_data[ 'product_id' ] : $session_data[ 'variation_id' ] ;

			$product_obj = wc_get_product( $product_id ) ;

			$price = $session_data[ 'data' ]->get_price() ;

			$product_fee = self::product_fee( $product_id , $price ) ;

			$signup_fee = efw_get_wc_signup_fee( $product_obj ) ;

			if ( in_array( $session_data[ 'data' ]->get_type() , array( 'subscription' , 'subscription_variation' ) ) ) {
				if ( empty( $product_fee ) && empty( $signup_fee ) ) {
					return $session_data ;
				}
			} else {
				if ( empty( $product_fee ) ) {
					return $session_data ;
				}
			}

			$product_fee = self::rule_fees( $product_id , $product_fee , $price ) ;
						/**
						 * Hook:efw_product_fee_before_set_price. 
						 *
						 * @since 1.0
						 */
			$product_fee = apply_filters( 'efw_product_fee_before_set_price' , $product_fee ) ;

			$session_data[ 'data' ]->set_price( $product_fee ) ;

			// Only for subscription products
			if ( in_array( $session_data[ 'data' ]->get_type() , array( 'subscription' , 'subscription_variation' ) ) ) {
				$product_fee = self::product_fee( $product_id , $signup_fee ) ;
				// Change subscription Sign up fee
				$session_data[ 'data' ]->update_meta_data( '_subscription_sign_up_fee' , $product_fee ) ;
			}

			return $session_data ;
		}

		public static function product_fee( $product_id, $price, $fee = 'total' ) {

			$fee_value = 0 ;

			$user_filter_data = array(
				'user_filter_type'  => get_option( 'efw_productfee_user_filter' ) ,
				'include_users'     => get_option( 'efw_productfee_include_users' ) ,
				'exclude_users'     => get_option( 'efw_productfee_exclude_users' ) ,
				'include_user_role' => get_option( 'efw_productfee_include_userrole' ) ,
				'exclude_user_role' => get_option( 'efw_productfee_exclude_userrole' ) ,
					) ;

			self::$user_filter_for_product = self::validate_users( $user_filter_data ) ;
			if ( ! self::$user_filter_for_product ) {
				return $fee_value ;
			}

			if ( 'yes' != get_option( 'efw_productfee_enable' ) ) {
				return $fee_value ;
			}

			if ( '1' == get_option( 'efw_productfee_fee_setup' ) ) {
				if ( ! efw_product_filter( $product_id ) ) {
					return $fee_value ;
				}

				$fee_value = efw_global_fee_value( $price ) ;
			} else {
				if ( 'yes' != get_post_meta( $product_id , '_efw_enable_fee' , true ) ) {
					return $fee_value ;
				}

				if ( '1' == get_post_meta( $product_id , '_efw_fee_type' , true ) ) {
					$fee_value = get_post_meta( $product_id , '_efw_fixed_value' , true ) ;
				} else {
					$percent_value = ( float ) get_post_meta( $product_id , '_efw_percent_value' , true ) ;
					$fee_value     = ( $percent_value / 100 ) * $price ;
				}
			}

			return ( 'total' == $fee ) ? ( $fee_value + $price ) : $fee_value ;
		}

		public static function rule_fees( $product_id, $fee_value, $price ) {
			$args = array(
				'post_parent' => $product_id ,
					) ;

			$rule_ids = efw_get_fee_rule_ids( $args ) ;

			if ( ! efw_check_is_array( $rule_ids ) ) {
				return $fee_value ;
			}

			$fee_values = array() ;

			foreach ( $rule_ids as $rule_id ) {
				$rule = efw_get_fee_rule( $rule_id ) ;

				if ( '1' == $rule->get_fee_type() ) {
					$fee_values[] = $rule->get_fixed_fee() ;
				} else {
					$percent_value = ( float ) $rule->get_percent_fee() ;
					$fee_values[]  = ( $percent_value / 100 ) * $price ;
				}
			}

			$rule_fee = $fee_value + array_sum( $fee_values ) ;

			return $rule_fee ;
		}

		/**
		 * Add Gateway Fee.
		 */
		public static function gateway_fee( $cart_obj ) {

			if ( 'yes' != get_option( 'efw_gatewayfee_enable' ) ) {
				return ;
			}

			if ( ! WC()->session->get( 'efw_gateway_id' ) ) {
				return ;
			}

			try {
				// Return if the gateway id is empty.
				$gateway_id = WC()->session->get( 'efw_gateway_id' ) ;
				if ( empty( $gateway_id ) ) {
					throw new Exception( __('Invalid Gateway Id' , 'extra-fees-for-woocommerce') ) ;
				}

				// Return if the gateway fee is disabled.
				if ( 'yes' != get_option( 'efw_enable_fee_for_' . $gateway_id ) ) {
					throw new Exception( __('Fee is disabled' , 'extra-fees-for-woocommerce') ) ;
				}

				self::$date_filter = self::validate_date( $gateway_id ) ;
				if ( ! self::$date_filter ) {
					throw new Exception(__( 'Fee is expired' , 'extra-fees-for-woocommerce') ) ;
				}

				$user_filter_data = array(
					'user_filter_type'  => get_option( 'efw_user_filter_type_for_' . $gateway_id ) ,
					'include_users'     => get_option( 'efw_include_user_for_' . $gateway_id ) ,
					'exclude_users'     => get_option( 'efw_exclude_user_for_' . $gateway_id ) ,
					'include_user_role' => get_option( 'efw_include_userrole_for_' . $gateway_id ) ,
					'exclude_user_role' => get_option( 'efw_exclude_userrole_for_' . $gateway_id ) ,
						) ;

				self::$user_filter_for_gateway = self::validate_users( $user_filter_data ) ;
				if ( ! self::$user_filter_for_gateway ) {
					throw new Exception( __('This fee is not yours' , 'extra-fees-for-woocommerce') ) ;
				}

				self::$product_filter = self::validate_product_category( $cart_obj , $gateway_id ) ;
				if ( ! self::$product_filter ) {
					throw new Exception( __('Invalid Product/Category' , 'extra-fees-for-woocommerce' )) ;
				}

				$fee_details = array(
					'type'            => 'gateway' ,
					'fee_type'        => get_option( 'efw_fee_type_for_' . $gateway_id ) ,
					'fixed_value'     => get_option( 'efw_fixed_value_for_' . $gateway_id ) ,
					'percentage_type' => get_option( 'efw_percentage_type_for_' . $gateway_id ) ,
					'cart_sub_total'  => get_option( 'efw_percent_value_of_cart_subtotal_for_' . $gateway_id ) ,
					'min_fee'         => get_option( 'efw_min_fee_for_' . $gateway_id ) ,
					'min_sub_total'   => get_option( 'efw_min_subtotal_for_' . $gateway_id ) ,
					'max_sub_total'   => get_option( 'efw_max_subtotal_for_' . $gateway_id ) ,
						) ;

				$value     = get_option( 'efw_fee_text_for_' . $gateway_id ) ;
				$fee_text  = efw_get_custom_field_translate_string( 'efw_fee_text_for_' . $gateway_id , $value ) ;
				$tax_class = get_option( 'efw_tax_class_for_' . $gateway_id ) ;

				$fee_value = self::fee_value( $fee_details ) ;
								/**
								 * Hook:efw_gateway_fee_after_calculate. 
								 *
								 * @since 1.0
								 */
				$fee_value = apply_filters( 'efw_gateway_fee_after_calculate' , $fee_value ) ;

				if ( empty( $fee_value ) ) {
					throw new Exception( __('Fee Value is empty' , 'extra-fees-for-woocommerce') ) ;
				}

				$taxable = ( 'not-required' == $tax_class ) ? false : true ;

				WC()->cart->add_fee( $fee_text , $fee_value , $taxable , $tax_class ) ;
			} catch ( Exception $ex ) {
				// Remove the gateway id in the WC session.
				WC()->session->set( 'efw_gateway_id' , null ) ;
			}
		}

		/**
		 * Validate from/to date
		 */
		public static function validate_date( $gateway_id ) {
			$return       = false ;
			$from_date    = true ;
			$to_date      = true ;
			$current_date = current_time( 'timestamp' ) ;
			$fromdate     = get_option( 'efw_from_date_for_' . $gateway_id ) ;
			$todate       = get_option( 'efw_to_date_for_' . $gateway_id ) ;

			// Validate from date
			if ( $fromdate ) {
				$from_date_object = EFW_Date_Time::get_date_time_object( $fromdate ) ;

				if ( $from_date_object->getTimestamp() > $current_date ) {
					$from_date = false ;
				}
			}
			// Validate to date
			if ( $todate ) {
				$to_date_object = EFW_Date_Time::get_date_time_object( $todate ) ;
				$to_date_object->modify( '+1 days' ) ;

				if ( $to_date_object->getTimestamp() < $current_date ) {
					$to_date = false ;
				}
			}

			if ( $from_date && $to_date ) {
				$return = true ;
			}
						/**
						 * Hook:efw_validate_rule_from_to_date. 
						 *
						 * @since 1.0
						 */
			return apply_filters( 'efw_validate_rule_from_to_date' , $return ) ;
		}

		/**
		 * Validate Products/Categories
		 */
		public static function validate_product_category( $cart_obj, $gateway_id ) {
			$product_filter_type = get_option( 'efw_product_filter_type_for_' . $gateway_id ) ;

			// return if selected as all products
			if ( '1' == $product_filter_type ) {
				return true ;
			}

			// Return if cart object is not initialized.
			if ( ! is_object( $cart_obj ) ) {
				return true ;
			}

			$cart_contents = $cart_obj->get_cart() ;

			if ( ! efw_check_is_array( $cart_contents ) ) {
				return true ;
			}

			$return = false ;
			foreach ( $cart_contents as $cart_content ) {

				switch ( $product_filter_type ) {
					case '2':
						$include_product = get_option( 'efw_include_product_for_' . $gateway_id ) ;
						$product_id      = empty( $cart_content[ 'variation_id' ] ) ? $cart_content[ 'product_id' ] : $cart_content[ 'variation_id' ] ;
						// return if any selected products in the cart
						if ( in_array( $product_id , $include_product ) ) {
							return true ;
						}

						break ;
					case '3':
						$return          = true ;
						$exclude_product = get_option( 'efw_exclude_product_for_' . $gateway_id ) ;
						$product_id      = empty( $cart_content[ 'variation_id' ] ) ? $cart_content[ 'product_id' ] : $cart_content[ 'variation_id' ] ;
						// excluded products.
						if ( in_array( $product_id , $exclude_product ) ) {
							return false ;
						}
						break ;
					case '4':
						//included categories.
						$product_categories = get_the_terms( $cart_content[ 'product_id' ] , 'product_cat' ) ;
						$include_category   = get_option( 'efw_include_category_for_' . $gateway_id ) ;

						if ( efw_check_is_array( $product_categories ) ) {
							foreach ( $product_categories as $product_category ) {
								// return if any selected categories products in the cart.
								if ( in_array( $product_category->term_id , $include_category ) ) {
									return true ;
								}
							}
						}
						break ;
					case '5':
						$return             = true ;
						// excluded categories.
						$product_categories = get_the_terms( $cart_content[ 'product_id' ] , 'product_cat' ) ;
						$exclude_category   = get_option( 'efw_exclude_category_for_' . $gateway_id ) ;
						if ( efw_check_is_array( $product_categories ) ) {
							foreach ( $product_categories as $product_category ) {
								if ( in_array( $product_category->term_id , $exclude_category ) ) {
									return false ;
								}
							}
						}
				}
			}

			return $return ;
		}

		/**
		 * Add Order Total Fee.
		 */
		public static function order_total_fee( $cart_obj ) {

			if ( 'yes' != get_option( 'efw_ordertotalfee_enable' ) ) {
				return ;
			}

			$user_filter_data = array(
				'user_filter_type'  => get_option( 'efw_ordertotalfee_user_filter' ) ,
				'include_users'     => get_option( 'efw_ordertotalfee_include_users' ) ,
				'exclude_users'     => get_option( 'efw_ordertotalfee_exclude_users' ) ,
				'include_user_role' => get_option( 'efw_ordertotalfee_include_userrole' ) ,
				'exclude_user_role' => get_option( 'efw_ordertotalfee_exclude_userrole' ) ,
					) ;

			self::$user_filter_for_order = self::validate_users( $user_filter_data ) ;
			if ( ! self::$user_filter_for_order ) {
				return ;
			}

			$excluded_shipping = get_option( 'efw_ordertotalfee_excluded_shipping' ) ;
			$selected_shipping = wc_get_chosen_shipping_method_ids() ;
			$blocked_shipping  = array_intersect( $excluded_shipping , $selected_shipping ) ;

			if ( efw_check_is_array( $blocked_shipping ) ) {
				return ;
			}

			$fee_details = array(
				'type'           => 'order' ,
				'fee_type'       => get_option( 'efw_ordertotalfee_fee_type' ) ,
				'fixed_value'    => get_option( 'efw_ordertotalfee_fixed_value' ) ,
				'cart_sub_total' => get_option( 'efw_ordertotalfee_cart_subtotal_percentage' ) ,
				'min_fee'        => 0 ,
				'min_sub_total'  => get_option( 'efw_ordertotalfee_min_sub_total' ) ,
				'max_sub_total'  => get_option( 'efw_ordertotalfee_max_sub_total' ) ,
					) ;

			$fee_text  = get_option( 'efw_ordertotalfee_fee_text' ) ;
			$tax_class = get_option( 'efw_ordertotalfee_tax_class' ) ;

			$fee_value = self::fee_value( $fee_details ) ;
						/**
						 * Hook:efw_order_fee_after_calculate. 
						 *
						 * @since 1.0
						 */
			$fee_value = apply_filters( 'efw_order_fee_after_calculate' , $fee_value ) ;

			if ( empty( $fee_value ) ) {
				return ;
			}

			$taxable = ( 'not-required' == $tax_class ) ? false : true ;

			$cart_obj->add_fee( $fee_text , $fee_value , $taxable , $tax_class ) ;
		}

		/**
		 * Calculate Fee.
		 * 
		 * @return float 
		 */
		public static function fee_value( $fee_details ) {

			extract( $fee_details ) ;

			if ( 'gateway' == $type ) {
				$cart_subtotal = ( '1' == $percentage_type ) ? efw_get_wc_cart_subtotal() : efw_get_wc_cart_total() ;
			} else if ('shipping' == $type ) {
				$cart_subtotal = ( '1' == $percentage_type ) ? efw_get_wc_cart_subtotal() : efw_get_wc_cart_total() ;
								$cart_subtotal = '1' == $fee_type ? efw_get_wc_cart_total() : $cart_subtotal;
			} else {
				$cart_subtotal = ( '2' == $fee_type ) ? efw_get_wc_cart_subtotal() : efw_get_wc_cart_total() ;
			}

			if ( ! ( ( $cart_subtotal >= $min_sub_total ) && ( $cart_subtotal <= $max_sub_total ) ) ) {
				return 0 ;
			}

			if ( '1' == $fee_type ) {
				$fee_value = $fixed_value ;
			} elseif ( '2' == $fee_type ) {
				$percent_value = ( ( $cart_sub_total / 100 ) * $cart_subtotal ) ;
				$fee_value     = ( $percent_value > $min_fee ) ? $percent_value : $min_fee ;
			} else {
				$percent_value = ( ( $cart_sub_total / 100 ) * $cart_subtotal ) ;
				$fee_value     = ( 'gateway' == $type ) ? ( $fixed_value + $percent_value ) : $percent_value ;
				$fee_value     = ( $fee_value > $min_fee ) ? $fee_value : $min_fee ;
			}

			return $fee_value ;
		}

		public static function update_fee_details( $order_id ) {

			$order = new WC_Order( $order_id ) ;

			$total_product_fee = array() ;

			foreach ( $order->get_items() as $item_id => $eachitem ) {

				$product_id = ! empty( $eachitem[ 'variation_id' ] ) ? $eachitem[ 'variation_id' ] : $eachitem[ 'product_id' ] ;

				$product = wc_get_product( $product_id ) ;

				if ( ! is_object( $product ) ) {
					continue ;
				}

				$price = wc_get_price_excluding_tax( $product ) + efw_get_wc_signup_fee( $product ) ;

				$product_fee = self::product_fee( $product_id , $price , 'fee' ) ;

				if ( empty( $product_fee ) ) {
					continue ;
				}

				$qty = isset( $eachitem[ 'qty' ] ) ? $eachitem[ 'qty' ] : 1 ;

				$productfee = $product_fee * $qty ;

				$key_name = efw_get_fee_text( $product_id ) ;

				wc_add_order_item_meta( $item_id , $key_name , wc_price( $productfee ) ) ;

				$product_fee = self::rule_fees( $product_id , $product_fee , $price ) ;

				$total_product_fee[] = $product_fee * $qty ;

				$rule_fee_texts = efw_get_rule_fee_text( $product_id , $price ) ;

				if ( efw_check_is_array( $rule_fee_texts ) ) {
					foreach ( $rule_fee_texts as $rule_fee_text => $rule_fee_value ) {
						$rule_fee_value = $rule_fee_value * $qty ;
						wc_add_order_item_meta( $item_id , $rule_fee_text , wc_price( $rule_fee_value ) ) ;
					}
				}
			}

			//Insert Product Fee Value
			self::update_product_fee_value( $order_id , $total_product_fee ) ;
			//Insert Order Fee Value
			self::update_order_fee_value( $order_id ) ;
			//Insert Gateway Fee Value
			self::update_gateway_fee_value( $order_id ) ;
			//Insert Shipping Fee Value
			self::update_shipping_fee_value($order_id);
		}

		public static function update_product_fee_value( $order_id, $total_product_fee ) {
			$order = new WC_Order( $order_id ) ;

			$args = array(
				'meta_query' => array(
					'relation' => 'AND' ,
					array(
						'key'     => 'efw_user_id' ,
						'value'   => $order->get_user_id() ,
						'compare' => '==' ,
					) ,
					array(
						'key'     => 'efw_order_id' ,
						'value'   => $order_id ,
						'compare' => '==' ,
					) ,
					array(
						'key'     => 'efw_product_fee' ,
						'compare' => 'NOT EXISTS' ,
					) ,
				) ,
					) ;

			$fee_ids = efw_get_fees_ids( $args ) ;

			$meta_args = array(
				'efw_user_id'  => $order->get_user_id() ,
				'efw_order_id' => $order_id ,
					) ;

			$meta_args[ 'efw_fee_type' ] = 'product' ;

			if ( ! efw_check_is_array( $fee_ids ) ) {
				$meta_args[ 'efw_product_fee' ] = array_sum( $total_product_fee ) ;
				efw_create_new_fees( $meta_args ) ;
			} else {
				$fee_id      = reset( $fee_ids ) ;
				$fee         = efw_get_fees( $fee_id ) ;
				$updated_fee = $fee->get_product_fee() + array_sum( $total_product_fee ) ;

				$meta_args[ 'efw_product_fee' ] = $updated_fee ;

				efw_update_fees( $fee_id , $meta_args ) ;
			}
		}

		public static function update_order_fee_value( $order_id ) {

			$order = new WC_Order( $order_id ) ;

			$args = array(
				'meta_query' => array(
					'relation' => 'AND' ,
					array(
						'key'     => 'efw_user_id' ,
						'value'   => $order->get_user_id() ,
						'compare' => '==' ,
					) ,
					array(
						'key'     => 'efw_order_id' ,
						'value'   => $order_id ,
						'compare' => '==' ,
					) ,
					array(
						'key'     => 'efw_order_fee' ,
						'compare' => 'NOT EXISTS' ,
					) ,
				) ,
					) ;

			$fee_ids = efw_get_fees_ids( $args ) ;

			$total_order_fee = array() ;

			$meta_args = array(
				'efw_user_id'  => $order->get_user_id() ,
				'efw_order_id' => $order_id ,
					) ;

			$order_fee_text = get_option( 'efw_ordertotalfee_fee_text' ) ;

			foreach ( $order->get_fees() as $fee ) {
				if ( $order_fee_text == $fee->get_name() ) {
					$total_order_fee[] = ( 'yes' == get_option( 'woocommerce_calc_taxes' ) ) ? abs( $fee->get_total() + $fee->get_total_tax() ) : abs( $fee->get_total() ) ;
				}
			}

			$meta_args[ 'efw_fee_type' ] = 'order' ;

			if ( ! efw_check_is_array( $fee_ids ) ) {
				$meta_args[ 'efw_order_fee' ] = array_sum( $total_order_fee ) ;
				efw_create_new_fees( $meta_args ) ;
			} else {
				$fee_id      = reset( $fee_ids ) ;
				$fees        = efw_get_fees( $fee_id ) ;
				$updated_fee = $fees->get_order_fee() + array_sum( $total_order_fee ) ;

				$meta_args[ 'efw_order_fee' ] = $updated_fee ;

				efw_update_fees( $fee_id , $meta_args ) ;
			}
		}

		public static function update_gateway_fee_value( $order_id ) {

			$order = new WC_Order( $order_id ) ;

			$args = array(
				'meta_query' => array(
					'relation' => 'AND' ,
					array(
						'key'     => 'efw_user_id' ,
						'value'   => $order->get_user_id() ,
						'compare' => '==' ,
					) ,
					array(
						'key'     => 'efw_order_id' ,
						'value'   => $order_id ,
						'compare' => '==' ,
					) ,
					array(
						'key'     => 'efw_gateway_fee' ,
						'compare' => 'NOT EXISTS' ,
					) ,
				) ,
					) ;

			$fee_ids = efw_get_fees_ids( $args ) ;

			$total_gateway_fee = array() ;

			$meta_args = array(
				'efw_user_id'  => $order->get_user_id() ,
				'efw_order_id' => $order_id ,
					) ;

			$gateway_id = WC()->session->get( 'efw_gateway_id' ) ;

			$value            = get_option( 'efw_fee_text_for_' . $gateway_id ) ;
			$gateway_fee_text = efw_get_custom_field_translate_string( 'efw_fee_text_for_' . $gateway_id , $value ) ;

			foreach ( $order->get_fees() as $fee ) {
				if ( $gateway_fee_text == $fee->get_name() ) {
					$total_gateway_fee[] = ( 'yes' == get_option( 'woocommerce_calc_taxes' ) ) ? abs( $fee->get_total() + $fee->get_total_tax() ) : abs( $fee->get_total() ) ;
				}
			}

			$meta_args[ 'efw_fee_type' ] = 'gateway' ;

			if ( ! efw_check_is_array( $fee_ids ) ) {
				$meta_args[ 'efw_gateway_fee' ] = array_sum( $total_gateway_fee ) ;
				efw_create_new_fees( $meta_args ) ;
			} else {
				$fee_id      = reset( $fee_ids ) ;
				$fees        = efw_get_fees( $fee_id ) ;
				$updated_fee = $fees->get_gateway_fee() + array_sum( $total_gateway_fee ) ;

				$meta_args[ 'efw_gateway_fee' ] = $updated_fee ;

				efw_update_fees( $fee_id , $meta_args ) ;
			}
		}
				
		public static function update_shipping_fee_value( $order_id) {
					
			$order = wc_get_order( $order_id ) ;
			if (!is_object($order)) {
				return;
			}
					
			$args = array(
			'meta_query' => array(
			'relation' => 'AND' ,
			array(
				'key'     => 'efw_user_id' ,
				'value'   => $order->get_user_id() ,
				'compare' => '==' ,
					) ,
					array(
				'key'     => 'efw_order_id' ,
				'value'   => $order_id ,
				'compare' => '==' ,
					) ,
					array(
				'key'     => 'efw_shipping_fee' ,
				'compare' => 'NOT EXISTS' ,
					) ,
			) ,
			) ;

			$fee_ids = efw_get_fees_ids( $args ) ;
					
			$total_shipping_fees = array() ;

			$meta_args = array(
			'efw_user_id'  => $order->get_user_id() ,
			'efw_order_id' => $order_id ,
			) ;
					
			$cart_fees = WC()->cart->get_fees();
			if (!efw_check_is_array($cart_fees)) {
				return;
			}
					
			$shipping_fee_exists = false;
			foreach ( $cart_fees as $fee ) {
				if (!is_object($fee)) {
					continue;
				}
						
				if ('efw_shipping_fee' != $fee->id ) {
					continue;
				}
						
				$shipping_fee_exists = true;
			}
					
			if (!$shipping_fee_exists) {
				return;
			}
					
			foreach ( $order->get_fees() as $fee ) {
				if (!is_object($fee)) {
					continue;
				}
						
				$total_shipping_fees[] = ( 'yes' == get_option( 'woocommerce_calc_taxes' ) ) ? abs( $fee->get_total() + $fee->get_total_tax() ) : abs( $fee->get_total() ) ;
			}
					
			if (!efw_check_is_array($total_shipping_fees)) {
				return;
			}

			$meta_args[ 'efw_fee_type' ] = 'shipping' ;

			if ( ! efw_check_is_array( $fee_ids ) ) {
				$meta_args[ 'efw_shipping_fee' ] = array_sum( $total_shipping_fees ) ;
				efw_create_new_fees( $meta_args ) ;
			} else {
				$fee_id      = reset( $fee_ids ) ;
				$fees        = efw_get_fees( $fee_id ) ;
				$updated_fee = $fees->get_shipping_fee() + array_sum( $total_shipping_fees ) ;

				$meta_args[ 'efw_shipping_fee' ] = $updated_fee ;

				efw_update_fees( $fee_id , $meta_args ) ;
			}
		}

		public static function display_variation_price( $price, $product ) {
			if ( is_product() && 'variable' == $product->get_type() ) {
				return $price . "<span class='efw-variation-price'></span>" ;
			}
			return $price ;
		}

		/**
		 * Validate Users for Fee.
		 *
		 * @return bool 
		 */
		public static function validate_users( $filter_data ) {
			$user_filter_type = $filter_data[ 'user_filter_type' ] ;

			switch ( $user_filter_type ) {

				case '2':
					$user_id      = get_current_user_id() ;
					$include_user = $filter_data[ 'include_users' ] ;
					if ( in_array( $user_id , $include_user ) ) {
						return true ;
					}

					break ;
				case '3':
					$user_id      = get_current_user_id() ;
					$exclude_user = $filter_data[ 'exclude_users' ] ;
					if ( in_array( $user_id , $exclude_user ) ) {
						return false ;
					}

					return true ;
					break ;
				case '4':
					$user = wp_get_current_user() ;

					$include_user_role = $filter_data[ 'include_user_role' ] ;
					// Loggedin user restriction
					if ( efw_check_is_array( $user->roles ) ) {
						foreach ( $user->roles as $role ) {
							if ( in_array( $role , $include_user_role ) ) {
								return true ;
							}
						}
					} else {
						// Guest user restriction
						if ( in_array( 'guest' , $include_user_role ) ) {
							return true ;
						}
					}

					break ;
				case '5':
					$user = wp_get_current_user() ;

					$exclude_user_role = $filter_data[ 'exclude_user_role' ] ;
					// Loggedin user restriction
					if ( efw_check_is_array( $user->roles ) ) {
						foreach ( $user->roles as $role ) {
							if ( in_array( $role , $exclude_user_role ) ) {
								return false ;
							}
						}
					} else {
						// Guest user restriction
						if ( in_array( 'guest' , $exclude_user_role ) ) {
							return false ;
						}
					}

					return true ;
					break ;
				default:
					return true ;
			}

			return false ;
		}

		/**
		 * Add Fee in Product Table.
		 */
		public static function product_fee_columns( $columns ) {

			if ( 'yes' != get_option( 'efw_productfee_enable' ) ) {
				return $columns ;
			}

			$add_column = array() ;

			foreach ( $columns as $key => $column ) {
				$add_column[ $key ] = $column ;

				if ( 'price' == $key ) {
					$add_column[ 'product_fee' ] = __( 'Product Fee' , 'extra-fees-for-woocommerce' ) ;
				}
			}

			return $add_column ;
		}

		/**
		 * Add Fee in Product Table.
		 */
		public static function product_fee_content( $column, $post_id ) {

			if ( 'yes' != get_option( 'efw_productfee_enable' ) ) {
				return ;
			}

			if ( 'product_fee' != $column ) {
				return ;
			}

			$product = wc_get_product( $post_id ) ;

			$price = wc_get_price_excluding_tax( $product ) ;

			$product_fee = self::product_fee( $post_id , $price , 'fee' ) ;

			$product_fee = self::rule_fees( $post_id , $product_fee , $price ) ;

			echo wp_kses_post(  ! empty( $product_fee ) ? wc_price( $product_fee ) : '-'  ) ;
		}
				
				/**
		 * May be add shipping fee.
		 */
		public static function may_be_add_shipping_fee( $cart_obj) {
					
			if (!is_object($cart_obj)) {
				return;
			}
						
			if ('yes' != get_option('efw_shippingfee_enable')) {
					return;
			}
										
			$chosen_shipping_method_ids = wc_get_chosen_shipping_method_ids();
			if ( ! efw_check_is_array( $chosen_shipping_method_ids ) ) {
				return;
			}
					
			foreach ($chosen_shipping_method_ids as $chosen_shipping_method_id) {
						
				if ('on' != get_option('efw_enable_' . $chosen_shipping_method_id)) {
					continue;
				}
						
				if (!self::validate_shipping_fee($chosen_shipping_method_id, $cart_obj)) {
					continue;
				}
						
				$fee_details = array(
									'type'            => 'shipping' ,
									'fee_type'        => get_option( 'efw_shipping_fee_type_' . $chosen_shipping_method_id ) ,
									'fixed_value'     => get_option( 'efw_shipping_fixed_value_' . $chosen_shipping_method_id ) ,
									'percentage_type' => get_option( 'efw_percentage_based_on_' . $chosen_shipping_method_id ) ,
									'cart_sub_total'  => get_option( 'efw_shipping_percentage_value_' . $chosen_shipping_method_id ) ,
									'min_fee'         => get_option( 'efw_shipping_minimum_fee_value_' . $chosen_shipping_method_id ) ,
									'min_sub_total'   => get_option( 'efw_shipping_fee_minimum_restriction_value_' . $chosen_shipping_method_id ) ,
									'max_sub_total'   => get_option( 'efw_shipping_fee_maximum_restriction_value_' . $chosen_shipping_method_id ) ,
				) ;

				$fee_value = self::fee_value( $fee_details ) ;
				/**
				 * Hook:efw_shipping_fee_after_calculate. 
				 *
				 * @since 1.0
				 */
				if ( !apply_filters( 'efw_shipping_fee_after_calculate' , $fee_value ) ) {
					continue ;
				}
						
				$tax_class = str_replace('_', '-', get_option('efw_shipping_tax_class_' . $chosen_shipping_method_id));

				$taxable = 'not-required' == $tax_class ? false : true ;
						
				$fee_text = get_option('efw_shipping_fee_text_' . $chosen_shipping_method_id);
								
				// Add fee.
				WC()->cart->fees_api()->add_fee(
					array(
						'id' => 'efw_shipping_fee',
						'name' => $fee_text,
						'amount' => $fee_value,
						'taxable' => $taxable,
						'tax_class' => $tax_class,
					)
				);
			}
		}
				
		/**
		 * Validate shipping fee.
		 */
		public static function validate_shipping_fee( $chosen_shipping_method_id, $cart_obj) {
										
			if (!self::validate_shipping_fee_date_filter($chosen_shipping_method_id)) {
				return false;
			}
					
			$user_filter_data = array(
			'user_filter_type'  => get_option( 'efw_shipping_user_filter_type_' . $chosen_shipping_method_id ) ,
			'include_users'     => get_option( 'efw_shipping_include_users_' . $chosen_shipping_method_id ) ,
			'exclude_users'     => get_option( 'efw_shipping_exclude_users_' . $chosen_shipping_method_id ) ,
			'include_user_role' => get_option( 'efw_shipping_include_userroles_' . $chosen_shipping_method_id ) ,
			'exclude_user_role' => get_option( 'efw_shipping_exclude_userroles_' . $chosen_shipping_method_id ) ,
				) ;

			if (!self::validate_users( $user_filter_data ) ) {
				return false;
			}
					
			if (!self::validate_shipping_product_category( $cart_obj , $chosen_shipping_method_id ) ) {
				return false;
			}
					
			return true;
		}
				
		/**
		 * Validate Shipping Products/Categories
		 */
		public static function validate_shipping_product_category( $cart_obj, $chosen_shipping_method_id ) {
					
			$product_filter_type = get_option( 'efw_shipping_product_filter_type_' . $chosen_shipping_method_id ) ;

			// return if selected as all products
			if ( '1' == $product_filter_type ) {
				return true ;
			}

			// Return if cart object is not initialized.
			if ( ! is_object( $cart_obj ) ) {
				return true ;
			}

			$cart_contents = $cart_obj->get_cart() ;
			if ( ! efw_check_is_array( $cart_contents ) ) {
				return true ;
			}

			$return = false ;
			foreach ( $cart_contents as $cart_content ) {

				switch ( $product_filter_type ) {
					case '2':
						$include_products = get_option( 'efw_shipping_include_products_' . $chosen_shipping_method_id ) ;
												$variation_id    = isset( $cart_content[ 'variation_id' ] ) ? $cart_content[ 'variation_id' ] : 0;
						$product_id      = isset( $cart_content[ 'product_id' ] ) ? $cart_content[ 'product_id' ] : 0 ;
												$product_id      = !empty($product_id) ? $product_id : $variation_id;
												// Return if any selected products in the cart.
						if ( $product_id && in_array( $product_id , $include_products ) ) {
							return true ;
						}

						break ;
					case '3':
						$return          = true ;
						$exclude_products = get_option( 'efw_shipping_exclude_products_' . $chosen_shipping_method_id ) ;
												$variation_id    = isset( $cart_content[ 'variation_id' ] ) ? $cart_content[ 'variation_id' ] : 0;
						$product_id      = isset( $cart_content[ 'product_id' ] ) ? $cart_content[ 'product_id' ] : 0 ;
												$product_id      = !empty($product_id) ? $product_id : $variation_id;
						// Excluded products.
						if ( $product_id && in_array( $product_id , $exclude_products ) ) {
							return false ;
						}
						break ;
					case '4':
						//Included categories.
						$product_categories = get_the_terms( $cart_content[ 'product_id' ] , 'product_cat' ) ;
						$include_categories   = get_option( 'efw_shipping_include_categories_' . $chosen_shipping_method_id ) ;
						if ( efw_check_is_array( $product_categories ) ) {
							foreach ( $product_categories as $product_category ) {
								if (!is_object($product_category)) {
									continue;
								}
																
								// return if any selected categories products in the cart.
								if ( in_array( $product_category->term_id , $include_categories ) ) {
									return true ;
								}
							}
						}
						break ;
					case '5':
						$return             = true ;
						// Excluded categories.
						$product_categories = get_the_terms( $cart_content[ 'product_id' ] , 'product_cat' ) ;
						$exclude_categories   = get_option( 'efw_shipping_exclude_categories_' . $chosen_shipping_method_id ) ;
						if ( efw_check_is_array( $product_categories ) ) {
							foreach ( $product_categories as $product_category ) {
								if (!is_object($product_category)) {
									continue;
								}
																
								if ( in_array( $product_category->term_id , $exclude_categories ) ) {
									return false ;
								}
							}
						}
				}
			}

			return $return ;
		}
				
		/**
		 * Validate shipping fee from/to date filter.
		 */
		public static function validate_shipping_fee_date_filter( $chosen_shipping_method_id ) {
					
			$return       = false ;
			$from_date    = true ;
			$to_date      = true ;
			$current_date = current_time( 'timestamp' ) ;
			$fromdate     = get_option( 'efw_shipping_from_date_' . $chosen_shipping_method_id ) ;
			$todate       = get_option( 'efw_shipping_to_date_' . $chosen_shipping_method_id ) ;

			// Validate from date
			if ( $fromdate ) {
				$from_date_object = EFW_Date_Time::get_date_time_object( $fromdate ) ;

				if ( $from_date_object->getTimestamp() > $current_date ) {
					$from_date = false ;
				}
			}
						
			// Validate to date
			if ( $todate ) {
				$to_date_object = EFW_Date_Time::get_date_time_object( $todate ) ;
				$to_date_object->modify( '+1 days' ) ;

				if ( $to_date_object->getTimestamp() < $current_date ) {
					$to_date = false ;
				}
			}

			if ( $from_date && $to_date ) {
				$return = true ;
			}
						/**
						 * Hook:efw_validate_shipping_from_to_date. 
						 *
						 * @since 1.0
						 */
			return apply_filters( 'efw_validate_shipping_from_to_date' , $return ) ;
		}


	}

	EFW_Fees_Handler::init() ;
}
